// -*- mode: C++; c-file-style: "stroustrup"; c-basic-offset: 4; -*-
////////////////////////////////////////////////////////////////////
//
// $Id: ffarray.h 58 2008-01-11 15:15:59Z sebastian $
//
////////////////////////////////////////////////////////////////////

#ifndef FF_ARRAY_H
#define FF_ARRAY_H

#include <vector>
#include <algorithm>
#include <cassert>
#include <inttypes.h>

namespace ff {
    
    ////////////////////////////////////////////////////////////////////
    //
    // PtrArray is a fixed size array of pointers.
    //
    ////////////////////////////////////////////////////////////////////
    
    template<class T> class PtrArray {
    private:
	typedef T* ptr_t;
	std::vector<ptr_t> data;
	size_t lastValid;
    public:
	typedef typename std::vector<ptr_t>::iterator iterator;
	typedef typename std::vector<ptr_t>::const_iterator const_iterator;
	
	PtrArray() : data(), lastValid(0) {} 	
	PtrArray(size_t capacity) : data(capacity, static_cast<ptr_t>(0)), lastValid(0) {} 
	PtrArray(const PtrArray& other) : data(other.data), lastValid(other.lastValid) {}
	
	void assign(size_t size) {
	    lastValid = 0;
	    data.assign(size, static_cast<ptr_t>(0));
	}
	
	void add(ptr_t elem) {
	    assert(lastValid < data.size());
	    data[lastValid++] = elem;
	}

	ptr_t operator[](uint32_t index) const {
	    assert(index < lastValid);
	    return data[index];
	}
	
	size_t size() const {
	    return lastValid;
	}
	
	iterator begin() {
	    return data.begin();
	}

	const_iterator begin() const {
	    return data.begin();
	}

	iterator end() {
	    return data.begin() + lastValid;
	}

	const_iterator end() const {
	    return data.begin() + lastValid;
	}
	
	bool empty() const {
	    return lastValid == 0;
	}

	void clear() {
	    lastValid = 0;
	}

	void swap(PtrArray& other) {
	    data.swap(other.data);
	    std::swap(lastValid, other.lastValid);
	}
    };    
}

#endif /* FF_ARRAY_H */
