#include "merge_tree_factory_miasm.h"

#include "merge_tree.h"

#include "miasm/sink_set_search.h"
#include "miasm/merge_tree.h"
#include "miasm/miasm_mas.h"

#include "../options/option_parser.h"
#include "../options/plugin.h"

#include "../utils/system.h"

#include <iostream>
#include <algorithm>
#include <cassert>

using namespace std;

namespace merge_and_shrink {
MergeTreeFactoryMiasm::MergeTreeFactoryMiasm(const options::Options &opts)
    : MergeTreeFactory(opts),
      options(opts),
      miasm_internal(MiasmInternal(opts.get_enum("miasm_internal"))),
      miasm_external(MiasmExternal(opts.get_enum("miasm_external"))) {
    // TODO: We would like to store sink_set_search instead of options here,
    // but it requires a task object.
}

MiasmMergeTree *MergeTreeFactoryMiasm::compute_miasm_merge_tree(
    const shared_ptr<AbstractTask> &task) {
    /* search for sink sets */

    SinkSetSearch sink_set_search(options, task);
    sink_set_search.search();
    sink_set_search.get_sink_set(sink_sets);

    sink_set_search.miasm_abstraction->release_cache();

    /* find the maximal weighted set packing of the priority sets */
    greedy_max_set_packing();
//    cerr << "max packing" << max_packing << endl;
    /* construct the merge tree based on the max packing
     * using the internal and external merging strategy
     * specified in the options for the current MIASM */
    for(unsigned int i=0;i<max_packing.size();i++){
        set<int> entry=max_packing[i];
        cout<<" max_packing: i: "<<i<<" values: ";
        for (set<int>::iterator it = entry.begin();
             it != entry.end(); ++it){
            cout<<*it<<" ";
        }
        cout<<" "<<endl;
    }
    MiasmMergeTree *miasm_tree = new MiasmMergeTree(
        max_packing, miasm_internal, miasm_external,
        sink_set_search,
        task);
    return miasm_tree;
}

unique_ptr<MergeTree> MergeTreeFactoryMiasm::compute_merge_tree(
    const shared_ptr<AbstractTask> &task) {
    TaskProxy task_proxy(*task);
    int num_ts = task_proxy.get_variables().size();

    // compute the merge tree in MiasmMergeTree form
    MiasmMergeTree *miasm_tree = compute_miasm_merge_tree(task);
    // get the actual merge order
    vector<pair<int, int>> merge_order;
    int next_ts_index = num_ts;
    while (true) {
        pair<int, int> next_merge = miasm_tree->select_next_and_update(next_ts_index);
        if (next_merge.first == -1) {
            break;
        }
        merge_order.push_back(next_merge);
        ++next_ts_index;
    }
    delete miasm_tree;

    // compute the merge tree in MergeTree form from the order
    // TODO: change the miasm computation to use it directly!
    map<int, MergeTreeNode*> index_to_tree;
    for (int atomic_ts_index = 0; atomic_ts_index < num_ts; ++atomic_ts_index) {
        index_to_tree[atomic_ts_index] = new MergeTreeNode(atomic_ts_index);
    }
    next_ts_index = num_ts;
    for (const pair<int, int> &merge : merge_order) {
        int ts_index1 = merge.first;
        int ts_index2 = merge.second;
        index_to_tree[next_ts_index] =
            new MergeTreeNode(index_to_tree[ts_index1], index_to_tree[ts_index2]);
        ++next_ts_index;
    }
    MergeTreeNode *root = index_to_tree[next_ts_index - 1];
//    MergeTree merge_tree(root, g_rng());
//    vector<pair<int, int>> other_merge_order;
//    next_ts_index = num_ts;
//    while (!merge_tree.done()) {
//        other_merge_order.push_back(merge_tree.get_next_merge(next_ts_index));
//        ++next_ts_index;
//    }
//    assert(merge_order == other_merge_order);

    return utils::make_unique_ptr<MergeTree>(root, rng, update_option);
}



string MergeTreeFactoryMiasm::name() const {
    return "miasm";
}

void MergeTreeFactoryMiasm::dump_tree_specific_options() const {
    // TODO
}

void MergeTreeFactoryMiasm::greedy_max_set_packing() {
    max_packing.clear();
    /* the variables that have been included in the packing solution */
    set<int> included;
    /* the subsets have been sorted in the decreasing order in weight,
     * i.e., the increasing order in the R&R ratio */
    for (size_t i = 0; i < sink_sets.size(); ++i) {
        set<int> intersection;
        set_intersection(
            sink_sets[i].begin(), sink_sets[i].end(),
            included.begin(), included.end(),
            inserter(intersection, intersection.begin()));
        /* if the subset does not intersect with the set of all
         * included variables, then add this subset into the current
         * solution and add its variables as included */
        if (intersection.empty()) {
            max_packing.push_back(sink_sets[i]);
            for (set<int>::iterator j = sink_sets[i].begin();
                 j != sink_sets[i].end(); ++j) {
                included.insert(*j);
            }
        }
    }
}

void MergeTreeFactoryMiasm::add_options_to_parser(options::OptionParser &parser) {
    MergeTreeFactory::add_options_to_parser(parser);

    // Options for symmetries computation:
    vector<string> use_symmetries;
    use_symmetries.push_back("NONE");
    use_symmetries.push_back("SIMPLE");
    use_symmetries.push_back("LIMITED");
    parser.add_enum_option("use_symmetries",
                           use_symmetries,
                           "use symmetries with MIASM",
                           "NONE");
    //parser.add_option<bool>("use_symmetries", "use symmetries with MIASM", "False");

    parser.add_option<int>("limited_max_states", "number of max states for limited symmetry method",
                           "50000");

    parser.add_option<int>("max_bliss_iterations", "maximum ms iteration until "
                           "which bliss is allowed to run.",
                           "infinity");
    parser.add_option<int>("bliss_call_time_limit", "time in seconds one bliss "
                           "run is allowed to last at most (0 means no limit)",
                           "0");
    parser.add_option<int>("bliss_total_time_budget", "time in seconds bliss is "
                           "allowed to run overall (0 means no limit)",
                           "0");
    parser.add_option<bool>("stop_after_no_symmetries", "stop calling bliss "
                            "after unsuccesfull previous bliss call.",
                           "False");
    vector<string> options_for_symmetry_merges;
    options_for_symmetry_merges.push_back("RIGHT");
    options_for_symmetry_merges.push_back("LEFT");
    options_for_symmetry_merges.push_back("RANDOM");
    parser.add_enum_option("option_for_symmetries",
                           options_for_symmetry_merges,
                           "choose options for merging of symmetries: right, left, random",
                           "RANDOM");
    vector<string> symmetries_for_shrinking;
    symmetries_for_shrinking.push_back("NO_SHRINKING");
    symmetries_for_shrinking.push_back("ATOMIC");
    symmetries_for_shrinking.push_back("LOCAL");
    parser.add_enum_option("symmetries_for_shrinking",
                           symmetries_for_shrinking,
                           "choose the type of symmetries used for shrinking: "
                           "no shrinking, "
                           "only atomic symmetries, "
                           "local symmetries.",
                           "NO_SHRINKING");
    vector<string> symmetries_for_merging;
    symmetries_for_merging.push_back("NO_MERGING");
    symmetries_for_merging.push_back("SMALLEST");
    symmetries_for_merging.push_back("LARGEST");
    parser.add_enum_option("symmetries_for_merging",
                           symmetries_for_merging,
                           "choose the type of symmetries that should determine "
                           "the set of transition systems to be merged: "
                           "the smallest or the largest",
                           "SMALLEST");
    vector<string> external_merging;
    external_merging.push_back("MERGE_FOR_ATOMIC");
    external_merging.push_back("MERGE_FOR_LOCAL");
    parser.add_enum_option("external_merging",
                           external_merging,
                           "choose the set of transition systems to be merged: "
                           "merge for atomic: merge all transition systems affected "
                           "by the chosen symmetry, or "
                           "merge for local: merge only the transition systems "
                           "mapped (in cycles) to others. only merge every "
                           "cycle separately.",
                           "MERGE_FOR_ATOMIC");
    vector<string> internal_merging;
    internal_merging.push_back("LINEAR");
    internal_merging.push_back("NON_LINEAR");
    parser.add_enum_option("internal_merging",
                           internal_merging,
                           "choose the order in which to merge the set of "
                           "transition systems to be merged (only useful with "
                           "MERGE_FOR_ATOMIC): "
                           "linear (obvious), "
                           "non linear, which means to first merge every cycle, "
                           "and then the resulting intermediate transition systems.",
                           "LINEAR");

    // Options for GraphCreator
    parser.add_option<bool>("stabilize_transition_systems", "compute symmetries that "
                            "stabilize transition systems, i.e. that are local.", "false");
    parser.add_option<bool>("debug_graph_creator", "produce dot readable output "
                            "from the graph generating methods", "false");



    //Options for MIASM:
    //DEFINE_ENUM_OPT(MiasmInternal, "miasm_internal", LEVEL)
    vector<string> enum_strings;
    enum_strings.push_back("level");
    enum_strings.push_back("reverse_level");
    parser.add_enum_option("miasm_internal",
                           enum_strings,
                           "",
                           "level");

    //DEFINE_ENUM_OPT(MiasmExternal, "miasm_external", NUM_VAR_CGL)
    enum_strings.clear();
    enum_strings.push_back("num_var_cgl");
    enum_strings.push_back("rnr_size_cgl");
    enum_strings.push_back("cgrl");
    parser.add_enum_option("miasm_external",
                           enum_strings,
                           "",
                           "num_var_cgl");

    /*
      SinkSetSearch options
      This is a required option, even if it is optional for the option parser.
      This is for merge_symmetries, to avoid creating a MiasmAbstraction if
      miasm is not the fallback strategy.
    */
    parser.add_option<MiasmAbstraction *>(
        MiasmAbstraction::option_key(),
        "",
        options::OptionParser::NONE);

    //DEFINE_OPT(double, OptTimeLimit, "time_limit", "30.00")
    parser.add_option<double>("time_limit",
                              "",
                              "30.0");

    //DEFINE_OPT(size_t, OptMemoryLimit, "memory_limit", "1500000000")
    parser.add_option<int>("memory_limit",
                           "",
                           "1500000000");

    //DEFINE_OPT(int, OptSizeLimit, "size_limit", "50000")
    /** @brief An \link #DECLARE_INT_OPT int wrapper struct \endlink
     * that provides the limit on the size of an abstraction on a subset
     * that can be "enqueued" in #SinkSetSearch */
    parser.add_option<int>("size_limit",
                           "",
                           "50000");

    //DEFINE_OPT(int, OptCliqueLimit, "clique_limit", "infinity")
    parser.add_option<int>("clique_limit",
                           "",
                           "infinity");

    //DEFINE_ENUM_OPT(EnumPriority, "priority", GAIN)
    enum_strings.clear();
    enum_strings.push_back("fifo");
    enum_strings.push_back("ratio");
    enum_strings.push_back("gain");
    parser.add_enum_option("priority", enum_strings,
                           "the order in which the subsets "
                           "are dequeued in the priority queue",
                           "gain");


    //DEFINE_ENUM_OPT(EnumExpand, "expand", SINGLE)
    enum_strings.clear();
    enum_strings.push_back("single");
    enum_strings.push_back("none");
    parser.add_enum_option("expand", enum_strings,
                           "which new subsets should be added into the search"
                           "priority queue",
                           "single");

    //DEFINE_ENUM_OPT(EnumGain, "gain", ALL_ACCUR)
    enum_strings.clear();
    enum_strings.push_back("pool_guess");
    enum_strings.push_back("pool_accur");
    enum_strings.push_back("all_guess");
    enum_strings.push_back("all_accur");
    parser.add_enum_option("gain", enum_strings,
                           "",
                           "all_accur");

    //DEFINE_ENUM_OPT(EnumPrune, "prune", NONE)
    enum_strings.clear();
    enum_strings.push_back("none");
    enum_strings.push_back("cgwc_mutex");
    parser.add_enum_option("prune", enum_strings,
                           "",
                           "none");

    options::Options options = parser.parse();
    if (options.get<int>("bliss_call_time_limit")
            && options.get<int>("bliss_total_time_budget")) {
        cerr << "Please only specify bliss_call_time_limit or "
                "bliss_total_time_budget but not both" << endl;
        utils::exit_with(utils::ExitCode::CRITICAL_ERROR);
    }


}

static shared_ptr<MergeTreeFactory>_parse(options::OptionParser &parser) {
    MergeTreeFactoryMiasm::add_options_to_parser(parser);

    options::Options opts = parser.parse();

    if (parser.dry_run())
        return nullptr;

    return make_shared<MergeTreeFactoryMiasm>(opts);
}

static options::PluginShared<MergeTreeFactory> _plugin("miasm", _parse);
}
