#include "merge_strategy_factory_test.h"

#include "merge_selector.h"
#include "merge_tree_factory.h"
#include "transition_system.h"

#include "../task_proxy.h"

#include "../causal_graph.h"
#include "../scc.h"

#include "../options/option_parser.h"
#include "../options/options.h"
#include "../options/plugin.h"

#include "../utils/logging.h"
#include "../utils/system.h"

#include <algorithm>
#include <cassert>
#include <iostream>

using namespace std;

namespace merge_and_shrink {


MergeStrategyFactoryTest::MergeStrategyFactoryTest(const options::Options &options)
    : MergeStrategyFactory(),
      merge_tree_factory(nullptr),
      merge_selector(nullptr) {
    if (options.contains("merge_tree")) {
        merge_tree_factory = options.get<shared_ptr<MergeTreeFactory>>("merge_tree");
    }
    if (options.contains("merge_selector")) {
        merge_selector = options.get<shared_ptr<MergeSelector>>("merge_selector");
    }
}

unique_ptr<MergeStrategy> MergeStrategyFactoryTest::compute_merge_strategy(
        const shared_ptr<AbstractTask> &task,
        FactoredTransitionSystem &fts) {
        merge_selector->initialize(task);
        return utils::make_unique_ptr<MergeStrategyTest>(fts, merge_selector);
}

void MergeStrategyFactoryTest::dump_strategy_specific_options() const {
    cout << "Merge order of test: ";
    cout << "bla"<<endl;

}

string MergeStrategyFactoryTest::name() const {
    return "test";
}

static shared_ptr<MergeStrategyFactory>_parse(options::OptionParser &parser) {
    parser.document_synopsis(
        "Stateless merge strategy",
        "This merge strategy has a merge selector, which computes the next "
        "merge only depending on the current state of the factored transition "
        "system, not requireing any additional information.");
    parser.add_option<shared_ptr<MergeSelector>>(
        "merge_selector",
        "The merge selector to be used.");

    options::Options opts = parser.parse();
    if (parser.dry_run())
        return nullptr;
    else
        return make_shared<MergeStrategyFactoryStateless>(opts);

static options::PluginShared<MergeStrategyFactory> _plugin("merge_test", _parse);
}
