#include "landmark_cost_partitioning_heuristic.h"

#include "landmark_cost_assignment.h"
#include "landmark_factory.h"
#include "landmark_status_manager.h"

#include "../plugins/plugin.h"
#include "../task_utils/successor_generator.h"
#include "../task_utils/task_properties.h"
#include "../utils/markup.h"

#include <cmath>
#include <limits>

using namespace std;

namespace landmarks {
LandmarkCostPartitioningHeuristic::LandmarkCostPartitioningHeuristic(
    const plugins::Options &opts)
    : LandmarkHeuristic(opts),
      cost_partitioning_strategy(opts.get<CostPartitioningStrategy>("cost_partitioning")) {
    if (log.is_at_least_normal()) {
        log << "Initializing landmark cost partitioning heuristic..." << endl;
    }
    check_unsupported_features(opts);
    initialize(opts);
    set_cost_assignment(opts);
}

void LandmarkCostPartitioningHeuristic::check_unsupported_features(
    const plugins::Options &opts) {
    shared_ptr<LandmarkFactory> lm_graph_factory =
        opts.get<shared_ptr<LandmarkFactory>>("lm_factory");

    if (task_properties::has_axioms(task_proxy)) {
        cerr << "Cost partitioning does not support axioms." << endl;
        utils::exit_with(utils::ExitCode::SEARCH_UNSUPPORTED);
    }

    if (!lm_graph_factory->supports_conditional_effects()
        && task_properties::has_conditional_effects(task_proxy)) {
        cerr << "Conditional effects not supported by the landmark "
             << "generation method." << endl;
        utils::exit_with(utils::ExitCode::SEARCH_UNSUPPORTED);
    }
}

void LandmarkCostPartitioningHeuristic::set_cost_assignment(
    const plugins::Options &opts) {
    if (cost_partitioning_strategy == CostPartitioningStrategy::OPTIMAL) {
        lm_cost_assignment =
            utils::make_unique_ptr<LandmarkEfficientOptimalSharedCostAssignment>(
                task_properties::get_operator_costs(task_proxy),
                *lm_graph, opts.get<lp::LPSolverType>("lpsolver"),
                opts.get<int>("lp_solve_method"),
                opts.get<int>("initial_lp_solve_method"),
                opts.get<bool>("use_presolve"),
                opts.get<bool>("crossover"),
                opts.get<int>("folding_level"),
                opts.get<int>("solve_dual"),
                opts.get<int>("aggregator_application_limit")
                );
    } else if (cost_partitioning_strategy == CostPartitioningStrategy::UNIFORM) {
        lm_cost_assignment =
            utils::make_unique_ptr<LandmarkUniformSharedCostAssignment>(
                task_properties::get_operator_costs(task_proxy),
                *lm_graph, opts.get<bool>("alm"));
    } else {
        ABORT("Unknown cost partitioning strategy");
    }
}

int LandmarkCostPartitioningHeuristic::get_heuristic_value(
    const State &ancestor_state) {
    double epsilon = 0.01;

    double h_val = lm_cost_assignment->cost_sharing_h_value(
        *lm_status_manager, ancestor_state);
    if (h_val == numeric_limits<double>::max()) {
        return DEAD_END;
    } else {
        return static_cast<int>(ceil(h_val - epsilon));
    }
}

bool LandmarkCostPartitioningHeuristic::dead_ends_are_reliable() const {
    return true;
}

void LandmarkCostPartitioningHeuristic::print_statistics() const {
    utils::g_log << "LP statistics for optimal cost partitioning heuristic:" << endl;
    lm_cost_assignment->print_statistics();
}

class LandmarkCostPartitioningHeuristicFeature : public plugins::TypedFeature<Evaluator, LandmarkCostPartitioningHeuristic> {
public:
    LandmarkCostPartitioningHeuristicFeature() : TypedFeature("landmark_cost_partitioning") {
        document_title("Landmark cost partitioning heuristic");
        document_synopsis(
            "Formerly known as the admissible landmark heuristic.\n"
            "See the papers" +
            utils::format_conference_reference(
                {"Erez Karpas", "Carmel Domshlak"},
                "Cost-Optimal Planning with Landmarks",
                "https://www.ijcai.org/Proceedings/09/Papers/288.pdf",
                "Proceedings of the 21st International Joint Conference on "
                "Artificial Intelligence (IJCAI 2009)",
                "1728-1733",
                "AAAI Press",
                "2009") +
            "and" +
            utils::format_conference_reference(
                {"Emil Keyder and Silvia Richter and Malte Helmert"},
                "Sound and Complete Landmarks for And/Or Graphs",
                "https://ai.dmi.unibas.ch/papers/keyder-et-al-ecai2010.pdf",
                "Proceedings of the 19th European Conference on Artificial "
                "Intelligence (ECAI 2010)",
                "335-340",
                "IOS Press",
                "2010"));

        LandmarkHeuristic::add_options_to_feature(*this);
        add_option<CostPartitioningStrategy>(
            "cost_partitioning",
            "strategy for partitioning operator costs among landmarks",
            "uniform");
        add_option<bool>("alm", "use action landmarks", "true");
        add_option<int>("lp_solve_method",
            "determine which method the LP solver should use to solve the "
            "LPs given when using standard CPLEX (not twophase).",
            "0");
        add_option<int>("initial_lp_solve_method",
            "determine which method the LP solver should use to solve the "
            "initial LP when using CPLEX",
            "0");
        add_option<bool>(
            "use_presolve",
            "turn presolving on or off. Using presolve creates an overhead "
            "so turning presolve off might decrease runtime.",
            "true");
        add_option<bool>(
            "crossover",
            "when set to false, turns off crossover when using the barrier optimizer. "
            "Crossover finds a basis, which is costly, but required for a warm start. "
            "Because of the warm start, it is turned on by default.",
            "true");
        add_option<int>(
            "folding_level",
            "turn folding_level off or on. -1 is automatic, 0 is off, "
            "1 to 5 are the different levels of folding, where "
            "1 is a moderate and 5 is an extremely aggressive level of folding.",
            "-1");
        add_option<int>(
            "solve_dual",
            "sets whether this the solver should solve the dual formulation "
            "of the LP instead of the Primal LP. Per default this is set to "
            "0 (automatic), which will usually solve the primal LP. Setting "
            "it to 1 will turn it on, and setting it to -1 will turn it off "
            "entirely.",
            "0"
        );
        add_option<int>(
            "aggregator_application_limit",
            "determines the number of times the aggregator is applied maximal. " 
            "The default value is -1, which corresponds to 1 for an LP and to "
            "infinite for a MIP. 0 turns it off.",
            "-1"
        );
        lp::add_lp_solver_option_to_feature(*this);

        document_note(
            "Usage with A*",
            "We recommend to add this heuristic as lazy_evaluator when using "
            "it in the A* algorithm. This way, the heuristic is recomputed "
            "before a state is expanded, leading to improved estimates that "
            "incorporate all knowledge gained from paths that were found after "
            "the state was inserted into the open list.");
        document_note(
            "Consistency",
            "The heuristic is consistent along single paths if it is "
            "set as lazy_evaluator; i.e. when expanding s then we have "
            "h(s) <= h(s')+cost(a) for all successors s' of s reached "
            "with a. But newly found paths to s can increase h(s), at "
            "which point the above inequality might not hold anymore.");
        document_note(
            "Optimal Cost Partitioning",
            "To use ``cost_partitioning=optimal``, you must build the planner with LP "
            "support. See LPBuildInstructions.");
        document_note(
            "Preferred operators",
            "Preferred operators should not be used for optimal planning. "
            "See Evaluator#Landmark_sum_heuristic for more information "
            "on using preferred operators; the comments there also apply "
            "to this heuristic.");

        document_language_support("action costs", "supported");
        document_language_support(
            "conditional_effects",
            "supported if the LandmarkFactory supports them; otherwise "
            "not supported");
        document_language_support("axioms", "not allowed");

        document_property("admissible", "yes");
        document_property("consistent",
                          "no; see document note about consistency");
        document_property("safe", "yes");
    }
};

static plugins::FeaturePlugin<LandmarkCostPartitioningHeuristicFeature> _plugin;

static plugins::TypedEnumPlugin<CostPartitioningStrategy> _enum_plugin({
        {"optimal",
         "use optimal (LP-based) cost partitioning"},
        {"uniform",
         "partition operator costs uniformly among all landmarks "
         "achieved by that operator"},
    });
}
