//Copyright 2015 Patrik Dürrenberger
//
//This file is part of TTP.
//
//TTP is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//TTP is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with Foobar.  If not, see <http://www.gnu.org/licenses/>.

#include <algorithm>
#include <limits>

#include "gecode.h"
#include "findMinTravelDistance.h"

//for method next subset
#define CHECK_BIT(var,pos) ((var) & (1<<(pos)))

FindMinTravelDistance::FindMinTravelDistance(DistanceMatrix* d, int lowerBound, int upperBound){
	this->d = d;
	numberOfTeams = d->numberOfTeams;
	this->lowerBound = lowerBound;
	this->upperBound = upperBound;
	powerSetSize = 1 << numberOfTeams;
	illegalSubset = false;
	counter = 0;
	nextSub = NULL;
	criticalSub=NULL;
	subsetContainer = NULL;
	tmp = 0;
	dist = 0;
	currentLoc = -1;
	startingLocation = -1;
}

FindMinTravelDistance::~FindMinTravelDistance(){
	//delete d;
}

int FindMinTravelDistance::findTeamsMinTravelDistance(const Index &idx, int currentTeam){
	counter = 0;

	//storage for all the subsets with shortest distance information
	//length of data structure is initially set to the size of the power set of all teams (1<<numberOfTeams)
	subsetContainer = new SubsetContainer(powerSetSize);

	//loop through each subset of teams
	while(nextSubset(currentTeam, idx)){
		//set the starting location properly
		startingLocation = setStartingLocation(currentTeam, idx);

		//integer to store the shortest traveling distance for this subset
		nextSub->setMinDistance(std::numeric_limits<int>::max());

		//do while not all permutations of the subset have been seen
		do{
			 dist = 0;

			 //set the starting location properly
			 currentLoc = startingLocation;

			 //calculate the traveling distance in the current permutation
			 for(int j = 0; j < nextSub->getSize(); j++){
				 dist += d->distances[currentLoc][nextSub->getTeam(j)];
				 currentLoc = nextSub->getTeam(j);
			 }
			 dist += d->distances[currentLoc][currentTeam]; //return home

			 //see whether current permutation is the shortest until now
			 if(dist < nextSub->getMinDistance()){
				 nextSub->setMinDistance(dist);
			 }
		}while(std::next_permutation(nextSub->teams,nextSub->teams+nextSub->getSize()));

		//copy vector for this subset to container of all subsets for this team
		subsetContainer->push_ordered(new Subset(*nextSub));
		delete nextSub;
	}

	int travelHomeDistance;
	if(idx.getLastTeamPlayedAway() != -1){
		travelHomeDistance = d->distances[idx.getLastTeamPlayedAway()][currentTeam];
	}else{
		travelHomeDistance = 0;
	}

	//create new csp-instance
	minCostCollection* m = new minCostCollection(subsetContainer, currentTeam, numberOfTeams, upperBound, travelHomeDistance, idx);
	Gecode::BAB<minCostCollection> e(m);
	delete m;

	//int* x;
	//find next (better) solution of csp-instance as long as possible
	while (minCostCollection* s = e.next()) {
		//debugging information
		//s->print();
		//cout << s->cost() << '\n';

		//store cost of current best solution
		tmp = s->cost().val();
		//x = s->getX();
		delete s;
	}

	delete subsetContainer;
	return tmp;
}

int FindMinTravelDistance::setStartingLocation(int currentTeam, const Index &idx) {
	//set the starting location properly
	if (idx.getNumberOfConsecutiveAwayGames() > 0 && nextSub->isCritical()) {
		//if last game was an away game and the current subset is critical
		//return the location of the team last played
		return idx.getLastTeamPlayedAway();
	} else {
		//if last game was a home game
		//return own home location
		return currentTeam;
	}
}



//returns pointer to the next legal subset. if all subset have been generated returns false
bool FindMinTravelDistance::setNextSub(int currentTeam, bool* stillHasToPlayAway){
	while(true){
		illegalSubset = false;
		//check if all subsets have been generated
		if(counter < powerSetSize){
			//if current team is not in the subset corresponding to this counter
			if(!CHECK_BIT(counter,currentTeam)){
				nextSub = new Subset(upperBound, numberOfTeams);
				//loop over all teams and see if they belong into the current subset. If they do add them to the subset.
				for(int j = 0; j < numberOfTeams; ++j){
					if(CHECK_BIT(counter,j)){
						//if currentTeam has already played team j in an away game, this subset is illegal to consider
						if(!stillHasToPlayAway[j]){
							illegalSubset = true;
							break;
						}
						//if the subset is full
						if(!nextSub->push_back(j)){
							illegalSubset = true;
							break;
						}
					}
				}
				//check if current subset is legal and if numberOfTeams in the subset is allowed
				if(!illegalSubset && nextSub->getSize() >= lowerBound){
					++(counter);
					delete[] stillHasToPlayAway;
					return true;
				}else{//else allocated memory has to be deleted
					delete nextSub;
				}
			}
		}else{
			delete[] stillHasToPlayAway;
			return false;
		}
		++(counter);
	}
	delete[] stillHasToPlayAway;
	return false;
}

//checks whether nextSub is a critical subset
bool FindMinTravelDistance::isCriticalSub(int currentTeam, const Index &idx){
	//if currentTeam is not on away trip, it starts from home nothing can be critical
	if(idx.getNumberOfConsecutiveAwayGames() <= 0){
		return false;
	}
	//if currentTeam is in the middle of an away trip, a subset is critical as soon as its size is smaller or equal to the number of
	//possible opponents still to play on this away trip (If lowerBound equals 1, this is the only condition)
	if(nextSub->getSize() <= (upperBound - idx.getNumberOfConsecutiveAwayGames())
			&& (lowerBound == 1 || nextSub->getSize() >= (lowerBound - idx.getNumberOfConsecutiveAwayGames()))){
		return true;
	}
	//if a subset doesn't fulfill the condition above, it is not critical
	return false;
}

//returns pointer to the next legal subset. if all subset have been generated returns false
bool FindMinTravelDistance::nextSubset(int currentTeam, const Index &idx){
	if(criticalSub == NULL){
		if(setNextSub(currentTeam, idx.getTeamsToPlayAway())){
			if(isCriticalSub(currentTeam, idx)){
				criticalSub = new Subset(*nextSub);
				criticalSub->setCritical();
			}
			return true;
		}else{
			return false;
		}

	}else{
		nextSub = criticalSub;
		criticalSub = NULL;
		return true;
	}
}
