//Copyright 2015 Patrik Dürrenberger
//
//This file is part of TTP.
//
//TTP is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//TTP is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with Foobar.  If not, see <http://www.gnu.org/licenses/>.

#include <iostream>
#include <sys/time.h>
#include <cstring>
#include <sstream>

#include "Parser.h"
#include "DistanceMatrix.h"
#include "ILB.h"
#include "IDA.h"

using namespace std;

void printTimeNeeded(const timeval& end, const timeval& start, long long nodes) {
	long long time =   ((end.tv_sec * (unsigned int)1e6 +   end.tv_usec) -
	                 (start.tv_sec * (unsigned int)1e6 + start.tv_usec)) /
	                 1000;
	cout << "time needed: " << endl;
	long long h = time / (60*60*1000);
	cout << "h: " << h << endl;
	long long min = (time - h*(60*60*1000)) / (60*1000);
	cout << "min: " << min << endl;
	long long sec = (time - h*(60*60*1000) - min*(60*1000)) / 1000;
	cout << "sec: " << sec << endl;
	long long usec = (time - h*(60*60*1000) - min*(60*1000) - sec*1000) ;
	cout << "usec: " << usec << endl;

	cout << "in usec: " << ((end.tv_sec * (unsigned int)1e6 +   end.tv_usec) -
            (start.tv_sec * (unsigned int)1e6 + start.tv_usec)) << endl;
	cout.precision(3);
	if(nodes != 0 && ((double) time/1000) != 0) cout << "expanded nodes per second: " << fixed << ((double) nodes) / ((double) time/1000) << endl;
}

bool getBool(const char* s){
	if(strcmp(s,"true") == 0){
		return true;
	}
	return false;
}

int getInt(const char* x){
	int ret;
	stringstream s(x);
	s >> ret;
	return ret;
}

int main(int argc, const char* argv[]) {
	if(argc != 11){
		cout << "ABBORT: start program with parameters: \nDistanceMatrixFile "
				"LabelFile \nForcedDeepening ElitePaths SubtreeSkipping \nSymmetryBreaking"
				" TeamOrdering SubtreeDepth NumberOfThreads Lambda!\n";
		return 1;
	}
	timeval start, end;

	//read input data
	Parser p;
	DistanceMatrix* distMatrix = p.getData(argv[1]);
	p.addLabels(distMatrix, argv[2]);
	//p.printData(d);

	//setParameters
	bool applyForcedDeepening = getBool(argv[3]);
	bool applyElitePaths = getBool(argv[4]);
	bool applySubtreeSkipping = getBool(argv[5]);
	int symmetryBreakingType = getInt(argv[6]); //0=none, 1=A, 2=H
	int teamOrderingType = getInt(argv[7]); //0=none, 1=maxTotalDistance, 2=minTotalDistance, 3=random
	int lambda = getInt(argv[10]);
	int printDepth = 4;
	int subtreeDepth = getInt(argv[8]);
	int numberOfThreads = getInt(argv[9]);

	//create an instance of ILB, which calculates heuristic values
	//This call will also create the disjoint pattern database
	//gettimeofday(&start, 0);
	ILB* ilb = new ILB(distMatrix, 1, 3, ((distMatrix->numberOfTeams-1) + (distMatrix->numberOfTeams-1))*(distMatrix->numberOfTeams/2),
			numberOfThreads);
	//gettimeofday(&end, 0);
	//cout << "disjoint pattern database established" << endl;
	//printTimeNeeded(end,start,0);

	//test IDA*
	IDA* ida = new IDA(distMatrix, 1, 3, applyForcedDeepening, applyElitePaths, applySubtreeSkipping,
			lambda, printDepth, subtreeDepth, symmetryBreakingType, teamOrderingType, ilb, numberOfThreads);
	gettimeofday(&start, 0);
	Solution* solution = ida->findSolution(distMatrix);
	gettimeofday(&end, 0);
	ida->printSolution(solution, distMatrix);
	printTimeNeeded(end, start, solution->expandedNodes);

	//free allocated memory
	delete solution;
	delete ida;
	delete ilb;
	delete distMatrix;

	return 0;
}
