//Copyright 2015 Patrik Dürrenberger
//
//This file is part of TTP.
//
//TTP is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//TTP is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with Foobar.  If not, see <http://www.gnu.org/licenses/>.

#include <iostream>
#include <cstdio>
#include <vector>

#include "ILB.h"
#include "DistanceMatrix.h"

using namespace std;

ILB::ILB(DistanceMatrix* d, int lb, int uB, int tNoG, int nOT) :
	lowerBound(lb), upperBound(uB), numberOfTeams(d->numberOfTeams), numberOfThreads(nOT), totalNumberOfGames(tNoG),
	disjointPatternDatabase(createDisjointPatternDatabase(d)){
	previousMinTravelDistance = new int**[numberOfThreads];
	minTravelDistanceFinder = new FindMinTravelDistance(d, lb, uB);
	for(int i=0; i<numberOfThreads; ++i){
		previousMinTravelDistance[i] = new int*[totalNumberOfGames+1];
		for(int j=0; j < totalNumberOfGames+1; ++j){
			previousMinTravelDistance[i][j] = new int[numberOfTeams];
			for(int k=0; k < numberOfTeams; ++k){
				previousMinTravelDistance[i][j][k] = -1;
			}
		}
	}

}

void ILB::resetPreviousMinTravelDistance(int oldNumberOfThreads){
	//delete first
	for(int i=0; i<oldNumberOfThreads; ++i){
		for(int j=0; j<totalNumberOfGames+1; ++j){
			delete[] previousMinTravelDistance[i][j];
		}
		delete[] previousMinTravelDistance[i];
	}
	delete[] previousMinTravelDistance;

	//then create new
	previousMinTravelDistance = new int**[numberOfThreads];
	for(int i=0; i<numberOfThreads; ++i){
		previousMinTravelDistance[i] = new int*[totalNumberOfGames+1];
		for(int j=0; j < totalNumberOfGames+1; ++j){
			previousMinTravelDistance[i][j] = new int[numberOfTeams];
			for(int k=0; k < numberOfTeams; ++k){
				previousMinTravelDistance[i][j][k] = -1;
			}
		}
	}
}

ILB::~ILB(){
	//delete previousMinTravelDistance
	for(int i=0; i<numberOfThreads; ++i){
		for(int j=0; j<totalNumberOfGames+1; ++j){
			delete[] previousMinTravelDistance[i][j];
		}
		delete[] previousMinTravelDistance[i];
	}
	delete[] previousMinTravelDistance;

	//delete patternDatabase
	for(int i=0; i<numberOfTeams; ++i){
		delete disjointPatternDatabase[i];
	}
	delete[] disjointPatternDatabase;
}

const PatternDatabase** ILB::createDisjointPatternDatabase(DistanceMatrix* d){
	const PatternDatabase** disjointPatternDatabase = new const PatternDatabase*[d->numberOfTeams];
	for(int i=0; i<d->numberOfTeams; ++i){
		disjointPatternDatabase[i] = new PatternDatabase(d, i, lowerBound, upperBound);
		//cout << "established pattern database for team " << i << endl;
	}
	return disjointPatternDatabase;
}

int ILB::h(const State &state, const int& threadNumber) const{
	int ilb = 0; //independent lower bound
	int home, away;

	//look for the last scheduled match
	if(state.depth != 0){//if not first call of heuristic
		if(state.numberOfScheduledMatchdays ==  state.numberOfMatchdays || state.schedule[state.numberOfScheduledMatchdays]->numberOfScheduledMatches == 0){
			home = state.schedule[state.numberOfScheduledMatchdays-1]->getCurrentHomeTeam(state.schedule[state.numberOfScheduledMatchdays-1]->numberOfScheduledMatches-1);
			away = state.schedule[state.numberOfScheduledMatchdays-1]->getCurrentAwayTeam(state.schedule[state.numberOfScheduledMatchdays-1]->numberOfScheduledMatches-1);
		}else{
			home = state.schedule[state.numberOfScheduledMatchdays]->getCurrentHomeTeam(state.schedule[state.numberOfScheduledMatchdays]->numberOfScheduledMatches-1);
			away = state.schedule[state.numberOfScheduledMatchdays]->getCurrentAwayTeam(state.schedule[state.numberOfScheduledMatchdays]->numberOfScheduledMatches-1);
		}
	}

	//Index* idx;
	//loop through each team to find the minimum travel distance for each team i
	for(int currentTeam = 0; currentTeam < numberOfTeams; ++currentTeam){
		if(state.depth == 0 || currentTeam == home || currentTeam == away || previousMinTravelDistance[threadNumber][state.depth-1][currentTeam] == -1){
			//idx = stateToIndex(state, currentTeam, upperBound);
			previousMinTravelDistance[threadNumber][state.depth][currentTeam] = disjointPatternDatabase[currentTeam]->getEntry(state, currentTeam, upperBound);
			//minTravelDistanceFinder->findTeamsMinTravelDistance(*idx,currentTeam);
			//delete idx;
		}else{
			previousMinTravelDistance[threadNumber][state.depth][currentTeam] = previousMinTravelDistance[threadNumber][state.depth-1][currentTeam];
		}
		ilb += previousMinTravelDistance[threadNumber][state.depth][currentTeam];
	}

	return ilb;
}

Index* ILB::stateToIndex(const State& s, int currentTeam, int upperBound) const{
	int remainingAwayGames = s.numberOfMatchdays/2 - s.timesPlayedAway[currentTeam];
	int remainingHomeGames = s.numberOfMatchdays/2 - s.timesPlayedHome[currentTeam];
	Subset* teamsToPlayAway = new Subset(numberOfTeams-1, numberOfTeams);
	for(int i=0; i<numberOfTeams; ++i){
		if(s.stillHasToPlayAway[currentTeam][i]){
			teamsToPlayAway->push_back(i);
		}
	}
	int nmbOfConsevutiveAwayGames = s.numberOfConsecutiveAwayGames[currentTeam];
	int lastTeamPlayedAway = -1;
	if(s.numberOfConsecutiveAwayGames[currentTeam] > 0){
		lastTeamPlayedAway = s.lastTeamPlayed[currentTeam];
	}
	return new Index(remainingAwayGames, remainingHomeGames, teamsToPlayAway, nmbOfConsevutiveAwayGames,
			lastTeamPlayedAway, numberOfTeams, currentTeam, upperBound);
}

void ILB::setNumberOfThreads(int nOT){
	int oldNumberOfThreads = numberOfThreads;
	numberOfThreads = nOT;
	resetPreviousMinTravelDistance(oldNumberOfThreads);
}
