//Copyright 2015 Patrik Dürrenberger
//
//This file is part of TTP.
//
//TTP is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//TTP is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with Foobar.  If not, see <http://www.gnu.org/licenses/>.

#include <queue>
#include <vector>
#include <stdio.h>
#include <thread>

#include "State.h"
#include "DistanceMatrix.h"
#include "ILB.h"
#include "Limit.h"
#include "Subtree.h"
#include "SubtreeQueue.h"

#ifndef IDA_H
#define IDA_H

struct Solution{
	State* state;
	long long expandedNodes;
	long long numberOfIterations;

	Solution(){
		state = NULL;
		expandedNodes = 0;
		numberOfIterations = 0;
	}

	Solution(State &s){
		state = &s;
		expandedNodes = 0;
		numberOfIterations = 0;
	}
};

struct NextSuccessor{
	int home;
	int away;
	int elitePathSuccessor1;
	int elitePathSuccessor2;

	NextSuccessor(void){
	}

	NextSuccessor(int t1, int t2){
		home = t1;
		away = t2;
		elitePathSuccessor1 = -1;
		elitePathSuccessor2 = -1;
	}

	NextSuccessor(const NextSuccessor* p){
		home = p->home;
		away = p->away;
		elitePathSuccessor1 = p->elitePathSuccessor1;
		elitePathSuccessor2 = p->elitePathSuccessor2;
	}
};

class IDA{
private:
	const int lowerBound, upperBound;
	ILB* ilb;
	const int symmetryBreakingType;
	const int* teamOrdering;
	const int printDepth, solutionDepth, subtreeDepth;
	const int lambda;
	const bool applyForcedDeepening, applyElitePaths, applySubtreeSkipping;
	long long nodesExpanded;
	bool solutionFound;
	const int numberOfThreads;
	std::mutex printMutex, addMutex, solutionMutex, updateMutex;

	void printMatchForDebug(State& state, bool inCreateSubtreeForest, int& home,
			int& away, const int subtreeNumber);
	void updateLimits(Limit& nextFLimit, const int &f_n, Limit& subtreeLimit,
			const State& state, const int& subtreeNumber);
	void checkIfSolution(Solution* tmp, Solution *solution, int threadNumber);
	void detachAllThreads(int numberOfThreads, std::thread* threads);
	void joinAllThreads(int numberOfThreads, std::thread* threads);
	void addOne(long long &toAddOne);
	void printSubtreeInformation(const int counter, Subtree& currentSubtree,
			Limit& nextFLimit, const Limit& fLimit);

public:
	IDA(DistanceMatrix* d, int lowerBound, int upperBound, bool applyForcedDeepening,
			bool applyElitePaths, bool applySubtreeSkipping, int lambda, int printDepth,
			int subtreeDepth, int symmetryBreakingType, int teamOrderingType, ILB* ilb, int nof);
	~IDA();

	Solution* findSolution(DistanceMatrix* distMatrix);
	void printSolution(Solution* solution, DistanceMatrix* dists);
	void handleSubtree(const int counter, Subtree &currentSubtree, const Limit &fLimit, Limit &nextFLimit, const int currentDepthLimit,
			SubtreeQueue &nextSubtreeQueue, Solution *solution, int threadNumber);

private:
	SubtreeQueue* createSubtreeForest(DistanceMatrix* distMatrix);
	int countSubtrees(DistanceMatrix* distMatrix);
	int createSubtreeForestRecursive(bool counting, int counter, State &state, SubtreeQueue &subtreeQueue);
	void printSubtreeForest(SubtreeQueue &subtreeQueue);
	bool forcedDeepening(const State &state, Limit &nextFLimit, const int &f_n, const int &currentDepthLimit,
			Limit &subtreeLimit, const int &subtreeNumber);
	Solution* recursiveSearch(State &state, const Limit &fLimit, Limit &subtreeLimit, Limit &nextFLimit, const int &currentDepthLimit,
			const int &subtreeNumber, const int &threadNumber);
	void resetAllMatchdaySetsAndAllTriedElitePaths(State &state);
	void resetMatchdaySetAndTriedElitePath(Matchday* matchday);
	void resetMatchdaySet(Matchday* matchday);
	int findKey(Matchday** schedule, int numberOfTeams, int currentMatchday, int home, int away);
	bool setElitePath(State &state, NextSuccessor* nextPairToCheck, const Limit &fLimit, const int &subtreeNumber);	//returns true if elite path has been set
	bool nextSuccessor(State &state, NextSuccessor* nextPairToCheck, const Limit &fLimit, const Limit &subtreeLimit,
			const bool& inCreateSubtreeForest, const int& subtreeNumber);
	bool symmetryBreak(const State &state);
	bool isGoal(const State &state);
	int f(const State &state, const int& threadNumber);
	void propagateConstraints(State &state, NextSuccessor* nextPairToCheck, const int& home, const int& away, const bool& inCreateSubtreeForest,
			const int& subtreeNumber);
	void recallConstraints(State &state); //reverts the process of propagateConstraints()
	void adjustNumberOfConsecutiveGames(State &state, const int& currentTeam);
	bool checkIfPlayedAway(const State &state, const int& currentTeam, const int& matchdayToCheck);
	bool checkIfPlayedHome(const State &state, const int& currentTeam, const int& matchdayToCheck);
	bool isInDesiredSubtree(const State &state, int iterationNumber);
	int getCurrentDepthLimitPlusLambda(const int currentDepthLimit);
	bool isFirstIteration(Limit* fLimit);
	int* getTeamOrdering(const int& teamOrderingType, DistanceMatrix* d);
	int* noOrdering(const int& numberOfTeams);
	int* maxDistanceOrdering(DistanceMatrix* d);
	int* minDistanceOrdering(DistanceMatrix* d);
	int* randomOrdering(const int& numberOfTeams);
	int* getSummedUpDistances(DistanceMatrix* d);
	int getIndexOfMax(int* distances, const int& numberOfTeams);
	int getIndexOfMin(int* distances, const int& numberOfTeams);
	void printTeamOrdering(int numberOfTeams);
};

#endif
