//Copyright 2015 Patrik Dürrenberger
//
//This file is part of TTP.
//
//TTP is free software: you can redistribute it and/or modify
//it under the terms of the GNU General Public License as published by
//the Free Software Foundation, either version 3 of the License, or
//(at your option) any later version.
//
//TTP is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with Foobar.  If not, see <http://www.gnu.org/licenses/>.

#include <iostream>
#include <limits>
#include <cmath>
#include <algorithm>

#include "IDA.h"
#include "ILB.h"
#include "DistanceMatrix.h"
#include "State.h"
#include "Subtree.h"
#include "ThreadPool.h"

using namespace std;

IDA::IDA(DistanceMatrix* distMatrix, int lb, int ub, bool afd, bool aep, bool ass,
		int l, int pd, int sd, int sbt, int tot, ILB* _ilb, int nof)
		: lowerBound(lb), upperBound(ub),  ilb(_ilb), symmetryBreakingType(sbt), teamOrdering(getTeamOrdering(tot, distMatrix)),
		  printDepth(pd), solutionDepth(((distMatrix->numberOfTeams-1) + (distMatrix->numberOfTeams-1))*(distMatrix->numberOfTeams/2)),
		  subtreeDepth(sd), lambda(l), applyForcedDeepening(afd), applyElitePaths(aep), applySubtreeSkipping(ass), numberOfThreads(nof){
	//printTeamOrdering(distMatrix->numberOfTeams);
	nodesExpanded = 0;
	solutionFound = false;
}

IDA::~IDA(){
	delete[] teamOrdering;
}

Solution* IDA::findSolution(DistanceMatrix* distMatrix){
	State* rootState;
	SubtreeQueue* subtreeQueue;
	SubtreeQueue* nextSubtreeQueue;
	Subtree* currentSubtree;
	int numberOfSubtrees = countSubtrees(distMatrix);
	if(applySubtreeSkipping){
		subtreeQueue = createSubtreeForest(distMatrix);
		nextSubtreeQueue = new SubtreeQueue(numberOfSubtrees);
		ilb->setNumberOfThreads(numberOfSubtrees);
		//printSubtreeForest();
		//state->printUnequal(*(subtreeQueue->top()->state));
		//return NULL;
	}else{
		rootState = State::makeRootState(distMatrix);
	}

	Limit* fLimit = new Limit(0);
	Limit* nextFLimit = new Limit(numeric_limits<int>::max());
	Solution* solution = new Solution();

	int currentDepthLimit = 0;
	int iterationNumber = 0;

	while(fLimit->distance != -1){
		++iterationNumber;
		long long numberOfNodesAtIterationStart = nodesExpanded;
		cout << endl << "new iteration with limit: " << fLimit->distance << " from depth: " << fLimit->depth << endl;
		int counter = 0;
		ThreadPool* pool;

		if(applySubtreeSkipping){
			pool = new ThreadPool(solution, this, fLimit, nextFLimit, currentDepthLimit, nextSubtreeQueue, numberOfThreads);
			while(!subtreeQueue->empty()){
				//get next subtree
				currentSubtree = subtreeQueue->top();
				subtreeQueue->pop();
				++counter;

				HandleSubtreeWithParameters* handle = new HandleSubtreeWithParameters(&IDA::handleSubtree, counter, *currentSubtree);
				pool->submit(*handle);
				delete handle;

				if(solution->state != NULL){
					delete pool;
					delete fLimit;
					delete nextFLimit;
					solution->expandedNodes = nodesExpanded;
					solution->numberOfIterations= iterationNumber;
					delete subtreeQueue;
					delete nextSubtreeQueue;
					return solution;
				}
			}
			delete pool;

			//subtreeQueue = nextSubtreeQueue
			while(!nextSubtreeQueue->empty()){
				subtreeQueue->push(nextSubtreeQueue->top());
				nextSubtreeQueue->pop();
			}
		}else{ //if subtree skipping is not applied
			resetAllMatchdaySetsAndAllTriedElitePaths(*rootState);
			Limit* dummyLimit = new Limit();
			solution= recursiveSearch(*rootState, *fLimit, *dummyLimit, *nextFLimit, currentDepthLimit, 0, 0);
			--(rootState->depth);
		}

		if((!applySubtreeSkipping && solution != NULL) || (applySubtreeSkipping && solution->state != NULL)){
			delete fLimit;
			delete nextFLimit;
			solution->expandedNodes = nodesExpanded;
			solution->numberOfIterations= iterationNumber;
			return solution;
		}

		//if no solution was found
		delete fLimit;
		fLimit = new Limit(*nextFLimit);
		currentDepthLimit = nextFLimit->depth;
		delete nextFLimit;
		nextFLimit = new Limit(numeric_limits<int>::max());

		cout << "expanded nodes in this iteration: " << nodesExpanded - numberOfNodesAtIterationStart << endl;
	}
	//free heap space for state because it hasn't led to a solution
	delete rootState;
	delete fLimit;
	delete nextFLimit;
	return NULL;
}

void IDA::printSolution(Solution* solution, DistanceMatrix* dists){
	if(solution->state->numberOfScheduledMatchdays != solution->state->numberOfMatchdays){
		cout << "WARNING: not all matchdays have been scheduled" << endl;
	}

	//for every scheduled matchday
	for(int i = 0; i < solution->state->numberOfScheduledMatchdays; ++i){
		cout << "Matchday " << i+1 << endl;
		cout << "-----------" << endl;
		if(solution->state->schedule[i]->numberOfScheduledMatches != solution->state->schedule[i]->numberOfMatches){
			cout << "WARNING: not all matches on matchday " << i << " have been scheduled" << endl;
		}

		solution->state->schedule[i]->printAllScheduledMatches(dists);
		cout << endl;
	}
	cout << "traveled distance: " << solution->state->traveledDistance << endl;
	cout << "expanded nodes: " << solution->expandedNodes << endl;
	cout << "number of iterations: " << solution->numberOfIterations << endl;
}

SubtreeQueue* IDA::createSubtreeForest(DistanceMatrix* distMatrix){
	SubtreeQueue* subtreeQueue = new SubtreeQueue(countSubtrees(distMatrix));

	//set root state
	State* state = State::makeRootState(distMatrix);
	createSubtreeForestRecursive(false, 0, *state, *subtreeQueue);

	//recreate start conditions for tree search
	delete state;
	return subtreeQueue;
}

int IDA::countSubtrees(DistanceMatrix* distMatrix){
	State* state = State::makeRootState(distMatrix);
	int counter = 0;
	SubtreeQueue* dummyQueue = new SubtreeQueue();
	counter = createSubtreeForestRecursive(true, counter, *state, *dummyQueue);

	//recreate start conditions for tree search
	delete state;
	return counter;
}

int IDA::createSubtreeForestRecursive(bool counting, int counter, State &state, SubtreeQueue &subtreeQueue) {
	++(state.depth);
	NextSuccessor* nextPairToCheck = new NextSuccessor(0, 0);
	Limit* fLimit = new Limit(numeric_limits<int>::max());
	while(nextSuccessor(state, nextPairToCheck, *fLimit, NULL, true, 0)){
		if(state.depth < subtreeDepth-1){
			counter = createSubtreeForestRecursive(counting, counter, state, subtreeQueue);
			recallConstraints(state);
			--(state.depth);
		}else{//if depth == subtreeDepth
			Limit* limit = new Limit(0,0);
			if(counting){
				++counter;
			}else{
				Subtree* s = new Subtree(state, *limit, subtreeDepth);
				subtreeQueue.push(s);
			}
			recallConstraints(state);
			delete limit;
		}
	}

	delete nextPairToCheck;
	delete fLimit;
	return counter;
}

void IDA::printSubtreeForest(SubtreeQueue &subtreeQueue){
	int i = 0;
	while(!subtreeQueue.empty()){
		cout << "printing subtree " << i << endl;
		//printSolution(new Solution(*subtreeQueue.top()->state));
		//subtreeQueue->top()->state->print();
		cout << "fLimit: " << subtreeQueue.top()->limit->distance << endl << endl;
		subtreeQueue.pop();
		++i;
	}
}

int IDA::getCurrentDepthLimitPlusLambda(const int currentDepthLimit){
	//A restriction for fLimit.depth + lambda (L) is that its value cannot exceed solutionDepth (D).
	//This could only happen when lambda > 1. When this is true and if L > D, then L is set to D.
	if(lambda > 1 && currentDepthLimit + lambda > solutionDepth){
		return solutionDepth;
	}
	return currentDepthLimit + lambda;
}

void IDA::updateLimits(Limit& nextFLimit, const int &f_n, Limit& subtreeLimit, const State& state, const int &subtreeNumber) {
	lock_guard < mutex > guard(updateMutex);
	if (applySubtreeSkipping){
		if(f_n < subtreeLimit.distance || (f_n == subtreeLimit.distance && state.depth > subtreeLimit.depth)) {
			subtreeLimit.update(f_n, state.depth, new ElitePath(state), applyElitePaths && (f_n < nextFLimit.distance));
		}
	}
	if(f_n < nextFLimit.distance || (f_n == nextFLimit.distance && state.depth > nextFLimit.depth)){
		nextFLimit.update(f_n, state.depth, new ElitePath(state), applyElitePaths);
		/*lock_guard < mutex > guard2(printMutex);
		cout << "new Limit: " << nextFLimit.distance << " from depth: " << nextFLimit.depth  << " (#" << subtreeNumber << ")" << endl;*/
	}
}

bool IDA::forcedDeepening(const State &state, Limit &nextFLimit, const int &f_n, const int &currentDepthLimit,
		Limit &subtreeLimit, const int &subtreeNumber){
	//cout << (depth >= getCurrentDepthLimitPlusLambda()) << " " << (f_n >= nextFLimit.distance) << " " << (currentDepthLimit >= solutionDepth) << endl;
	if(state.depth >= getCurrentDepthLimitPlusLambda(currentDepthLimit)
			|| f_n > nextFLimit.distance
			|| currentDepthLimit >= solutionDepth) {
		updateLimits(nextFLimit, f_n, subtreeLimit, state, subtreeNumber);
		return false;
	}

	return true;
}

bool IDA::isInDesiredSubtree(const State &state, int iterationNumber){
	if(state.schedule[0]->homeTeams[0] == 3
			&& state.schedule[0]->homeTeams[1] == 4
			&& state.schedule[0]->homeTeams[2] == 5
			&& state.schedule[0]->awayTeams[0] == 0
			&& state.schedule[0]->awayTeams[1] == 1
			&& state.schedule[0]->awayTeams[2] == 2
			&& iterationNumber >= 3){
		return true;
	}
	return false;
}

void IDA::addOne(long long &toAddOne) {
	lock_guard<mutex> guard(addMutex);
	++toAddOne;
}

Solution* IDA::recursiveSearch(State &state, const Limit &fLimit, Limit &subtreeLimit, Limit &nextFLimit,
		const int &currentDepthLimit, const int &subtreeNumber, const int &threadNumber){
	if(solutionFound) return NULL;
	++(state.depth);

	int f_n = f(state, threadNumber);
	//cout << ", " << f_n << ")" << endl;

	if(f_n > fLimit.distance){
		//if(state.depth < printDepth+1) cout << "-" << endl;

		if(applyForcedDeepening){
			bool goDeeper = forcedDeepening(state, nextFLimit, f_n, currentDepthLimit, subtreeLimit, subtreeNumber);
			if(!goDeeper){
				return NULL;
			}
		}else{
			updateLimits(nextFLimit, f_n, subtreeLimit, state, subtreeNumber);
			return NULL;
		}
	}
	//if(state.depth < printDepth+1) cout << "+" << endl;

	if(isGoal(state)){
		lock_guard<mutex> guard(solutionMutex);
		if(!solutionFound){
			solutionFound = true;
			return new Solution(state);
		}else{
			return NULL;
		}
	}

	addOne(nodesExpanded);

	NextSuccessor* nextPairToCheck = new NextSuccessor(0,0);
	while(nextSuccessor(state, nextPairToCheck, fLimit, subtreeLimit, false, subtreeNumber)){
		Solution* solution = recursiveSearch(state, fLimit, subtreeLimit, nextFLimit, currentDepthLimit,
				subtreeNumber, threadNumber);
		--(state.depth);
		if(solution != NULL){
			delete nextPairToCheck;
			return solution;
		}
		recallConstraints(state);
	}
	delete nextPairToCheck;
	return NULL;
}

void IDA::resetAllMatchdaySetsAndAllTriedElitePaths(State &state){
	//if(state != NULL){
		for(int i=0; i < state.numberOfMatchdays; ++i){
			resetMatchdaySetAndTriedElitePath(state.schedule[i]);
		}
	//}
}

void IDA::resetMatchdaySetAndTriedElitePath(Matchday* matchday) {
	matchday->matchdaySet.clear();
	for(int j=0; j<(matchday->numberOfMatches); ++j){
		matchday->triedElitePath[j] = false;
	}
}

void IDA::resetMatchdaySet(Matchday* matchday) {
	matchday->matchdaySet.clear();
}

//returns true if elite path could have been set
bool IDA::setElitePath(State &state, NextSuccessor* nextPairToCheck, const Limit &limit, const int &subtreeNumber){
	int currentMatchday = state.numberOfScheduledMatchdays;
	int currentMatch = state.schedule[currentMatchday]->numberOfScheduledMatches;
	if(limit.elitePath != NULL && state.depth < limit.depth  && !(state.schedule[currentMatchday]->triedElitePath[currentMatch])){
		//set the elite paths match as the current match
		int home = limit.elitePath->schedule[currentMatchday]->getCurrentHomeTeam(currentMatch);
		int away = limit.elitePath->schedule[currentMatchday]->getCurrentAwayTeam(currentMatch);
		if(home >= state.numberOfTeams || away >= state.numberOfTeams) return false;

		nextPairToCheck->elitePathSuccessor1 = home;
		nextPairToCheck->elitePathSuccessor2 = away;

		state.schedule[currentMatchday]->triedElitePath[currentMatch] = true;
		propagateConstraints(state, NULL, home, away, false, subtreeNumber);
		return true;
	}else{
		return false;
	}
}

bool IDA::symmetryBreak(const State &state){
	//if half of the schedule is constructed and symmetry breaking type is not NONE
	if(symmetryBreakingType == 1){ //symmetry-A
		if(state.timesPlayedHome[0] > state.timesPlayedAway[0]){
			return true;
		}
	}else if(symmetryBreakingType == 2){ //symmetry-H
		if(state.timesPlayedHome[0] < state.timesPlayedAway[0]){
			return true;
		}
	}
	return false;
}

bool IDA::nextSuccessor(State &state, NextSuccessor* nextPairToCheck, const Limit &fLimit,
		const Limit &subtreeLimit, const bool &inCreateSubtreeForest, const int &subtreeNumber){
	int currentMatchday = state.numberOfScheduledMatchdays;
	int currentMatch = state.schedule[currentMatchday]->numberOfScheduledMatches;

	//try elite path first
	if(!inCreateSubtreeForest && applyElitePaths){
		if(applySubtreeSkipping){
			if(setElitePath(state, nextPairToCheck, subtreeLimit, subtreeNumber)){
				//if elite path is next successor, next successor has been found => return true
				return true;
			}
		}else{
			if(setElitePath(state, nextPairToCheck, fLimit, subtreeNumber)){
				//if elite path is next successor, next successor has been found => return true
				return true;
			}
		}
	}

	//find team1
	for(; (nextPairToCheck->home) < (state.distMatrix->numberOfTeams); ++(nextPairToCheck->home)){
		int home = teamOrdering[nextPairToCheck->home];
		int away;
		//if current matchday wasn't played by home team and
		//if the home team to assign is greater than the last home team
		if(!(state.matchdayWasPlayedByTeam[currentMatchday][home]) &&
				home > state.schedule[currentMatchday]->getCurrentHomeTeam(currentMatch-1)){

			//find team2
			for(; (nextPairToCheck->away) < (state.distMatrix->numberOfTeams); ++(nextPairToCheck->away)){
				away = teamOrdering[nextPairToCheck->away];
				//if current matchday wasn't played by team team1 and
				//if the last team that team2 played wasn't team1 and
				//if team2 still has to play team1 away and
				//if number of consecutive away games for team2 is lower than the defined upper bound and
				//if number of consecutive home games for team1 is lower than the defined upper bound and
				//if team1 hasn't been seen already in the elite path match or
				//if team2 hasn't been seen already in the elite path match
				if(!(state.matchdayWasPlayedByTeam[currentMatchday][away]) &&
						(state.lastTeamPlayed[away] != home) &&
						state.stillHasToPlayAway[away][home] &&
						((state.numberOfConsecutiveAwayGames[away]) < upperBound) &&
						((state.numberOfConsecutiveHomeGames[home]) < upperBound) &&
						((home != nextPairToCheck->elitePathSuccessor1) ||
						(away != nextPairToCheck->elitePathSuccessor2)))
				{
					if(state.numberOfScheduledMatchdays == state.numberOfMatchdays/2
							&& currentMatch == 0
							&& symmetryBreakingType != 0
							&& symmetryBreak(state)
							&& !inCreateSubtreeForest){
						//if this partial solution should not be pursued due to symmetry
						//do nothing == look for the next possible match by taking the next step in the iteration through team2
					}else{//if its not the last match of this matchday, and partial solution should be pursued
						//then assign that team1 plays home, team2 plays away
						break;
					}
				}
			}
			//if no suitable team2 for given team1 has been found
			if((nextPairToCheck->away) == (state.distMatrix->numberOfTeams)){
				//cout << "in no suitable team2" << endl;
				//set last seen successor for team2 to zero (because for the next team1 team2=0 might be possible again)
				nextPairToCheck->away = 0;
			}else{//if suitable team2 has been found
				propagateConstraints(state, nextPairToCheck, home, away, inCreateSubtreeForest, subtreeNumber);

				//next successor found
				return true;
			}
		}
	}
	//if no suitable match has been found
	//there is no successor left
	return false;
}

bool IDA::isGoal(const State &state){
	if(state.numberOfMatchdays == state.numberOfScheduledMatchdays){
		return true;
	}
	return false;
}

int IDA::f(const State &state, const int &threadNumber){
	return state.traveledDistance + ilb->h(state, threadNumber);
}

void IDA::printMatchForDebug(State& state, bool inCreateSubtreeForest,
		int& home, int& away, const int subtreeNumber) {

	if (state.depth < printDepth && !inCreateSubtreeForest) {
		for (int i = -1; i < state.depth - 1; ++i)
			cout << '\t';
		cout << home << " - " << away << " (#" << subtreeNumber;
	}
}

void IDA::propagateConstraints(State &state, NextSuccessor* nextPairToCheck, const int& home, const int& away,
		const bool& inCreateSubtreeForest, const int& subtreeNumber) {
	int currentMatchday = state.numberOfScheduledMatchdays;

	//lock_guard < mutex > guard(printMutex);
	//printMatchForDebug(state, inCreateSubtreeForest, home, away, subtreeNumber);

	//count timesPlayedHome/Away
	++(state.timesPlayedHome[home]);
	++(state.timesPlayedAway[away]);

	//make sure that teams don't play again in the current matchday
	state.matchdayWasPlayedByTeam[currentMatchday][home] = true;
	state.matchdayWasPlayedByTeam[currentMatchday][away] = true;

	//if it's not a call from an elite path
	if(nextPairToCheck != NULL){
		//update the next successor to be checked
		if ((state.distMatrix->numberOfTeams) == (nextPairToCheck->away+1)) {
			//if all team2's for a given team1 have been seen
			//next successor is next team1 and set team2 to zero (all team2's are possible again)
			nextPairToCheck->home += 1;
			nextPairToCheck->away = 0;
		} else {
			//next successor is simply the next team2 and the same team1
			nextPairToCheck->away += 1;
		}
	}


	//make sure the away team doesn't play the home team away again
	state.stillHasToPlayAway[away][home] = false;

	//adjust traveled distance (one or two teams per game have to travel)
	//the away team always has to travel to the home team
	if (state.numberOfConsecutiveAwayGames[away] > 0) {//they come from another away match
		state.traveledDistance += state.distMatrix->distances[state.lastTeamPlayed[away]][home];
	} else {//they come from home
		state.traveledDistance += state.distMatrix->distances[away][home];
	}
	//if the home team has played away the last match, it also has to travel home first
	if (state.numberOfConsecutiveAwayGames[home] > 0) {
		state.traveledDistance += state.distMatrix->distances[state.lastTeamPlayed[home]][home];
	}
	//for the last matchday one must also consider that the away teams have to travel home again
	if (state.numberOfScheduledMatchdays == state.numberOfMatchdays - 1) {
		state.traveledDistance += state.distMatrix->distances[home][away];
	}

	//adjust number of consecutive games played home/away
	++(state.numberOfConsecutiveAwayGames[away]);
	state.numberOfConsecutiveHomeGames[away] = 0;
	state.numberOfConsecutiveAwayGames[home] = 0;
	++(state.numberOfConsecutiveHomeGames[home]);

	//adjust last teams played
	state.lastTeamPlayed[home] = away;
	state.lastTeamPlayed[away] = home;

	//add the match to the current matchday
	state.schedule[currentMatchday]->addMatch(home, away);

	//adjust number of scheduled matches
	++(state.schedule[currentMatchday]->numberOfScheduledMatches);
	//if all matches of this matchday have been scheduled
	if (state.schedule[currentMatchday]->numberOfMatches == state.schedule[currentMatchday]->numberOfScheduledMatches) {
		//adjust number of scheduled matchdays
		++(state.numberOfScheduledMatchdays);
	}

	/*if(state.depth < printDepth && !inCreateSubtreeForest && !solutionFound){
		cout << ", " << f(state, 0) << ")" << endl;
	}*/
}

//revert the process of propagateConstraints()
void IDA::recallConstraints(State &state){
	//adjust number of scheduled matches
	//if the schedule is complete or no match of the current matchday has been scheduled so far
	if(isGoal(state) || state.schedule[state.numberOfScheduledMatchdays]->numberOfScheduledMatches == 0){
		//before we go to the previous matchday we need to reset the MatchdayAlreadySeenBooleans
		if(!isGoal(state)) resetMatchdaySet(state.schedule[state.numberOfScheduledMatchdays]);
		//go to the previous matchday
		--(state.numberOfScheduledMatchdays);
		//go to the last scheduled match of the previous matchday
		--(state.schedule[state.numberOfScheduledMatchdays]->numberOfScheduledMatches);
	}else{//if there is a scheduled match on this matchday
		//go to the last scheduled match of this matchday
		--(state.schedule[state.numberOfScheduledMatchdays]->numberOfScheduledMatches);
	}

	//set the matchday and match for which we want to recall the constraints
	int currentMatchday = state.numberOfScheduledMatchdays;
	int currentMatch = state.schedule[currentMatchday]->numberOfScheduledMatches;

	//store involved teams of the current match
	int home = state.schedule[currentMatchday]->getCurrentHomeTeam(currentMatch);
	int away = state.schedule[currentMatchday]->getCurrentAwayTeam(currentMatch);

	//delete the current match
	state.schedule[currentMatchday]->deleteCurrentMatch();

	//adjust last teams played
	if(currentMatchday == 0){ //if we are on the first matchday
		//lastTeamPlayed is set to the initial value
		state.lastTeamPlayed[home] = -1;
		state.lastTeamPlayed[away] = -1;
	}else{ //if we are not on the first matchday
		state.lastTeamPlayed[home] = state.schedule[currentMatchday-1]->hasPlayedAgainst(home);
		state.lastTeamPlayed[away] = state.schedule[currentMatchday-1]->hasPlayedAgainst(away);
	}

	//adjust number of consecutive games played home/away
	adjustNumberOfConsecutiveGames(state, home);
	adjustNumberOfConsecutiveGames(state, away);

	//adjust timesPlayedHome/Away
	--(state.timesPlayedHome[home]);
	--(state.timesPlayedAway[away]);

	//adjust traveled distance (subtract everything that was added by propagateConstraints()
	if (state.numberOfConsecutiveAwayGames[away] > 0) {
		state.traveledDistance -= state.distMatrix->distances[state.lastTeamPlayed[away]][home];
	} else {
		state.traveledDistance -= state.distMatrix->distances[away][home];
	}
	if (state.numberOfConsecutiveAwayGames[home] > 0) {
		state.traveledDistance -= state.distMatrix->distances[state.lastTeamPlayed[home]][home];
	}
	if (state.numberOfScheduledMatchdays == state.numberOfMatchdays - 1) {
		state.traveledDistance -= state.distMatrix->distances[home][away];
	}

	//make sure the away team can play the home team away again
	state.stillHasToPlayAway[away][home] = true;

	//make sure that teams can play again in the current matchday
	state.matchdayWasPlayedByTeam[currentMatchday][home] = false;
	state.matchdayWasPlayedByTeam[currentMatchday][away] = false;
}

void IDA::adjustNumberOfConsecutiveGames(State &state, const int& currentTeam){
	int currentMatchday = state.numberOfScheduledMatchdays;
	int previousMatchday;

	if(currentMatchday == 0){//if its the first matchday adjusting the number of consecutive games is simple
		//set consecutive home and away games to 0
		state.numberOfConsecutiveHomeGames[currentTeam] = 0;
		state.numberOfConsecutiveAwayGames[currentTeam] = 0;
	}else{//if its not the first matchday adjusting the number of consecutive games is more complex
		if(state.numberOfConsecutiveHomeGames[currentTeam] > 1){//if the number of consecutive home games is greater than 1

			//just subtract one consecutive home game (away games are 0 anyway, so no need to examine them)
			--(state.numberOfConsecutiveHomeGames[currentTeam]);

		}else if(state.numberOfConsecutiveAwayGames[currentTeam] > 1){ //if the number of consecutive away games is greater than 1

			//just subtract one consecutive away game (home games are 0 anyway, so no need to examine them)
			--(state.numberOfConsecutiveAwayGames[currentTeam]);

		}else{
			//if the number of consecutive home games is 1
			if(state.numberOfConsecutiveHomeGames[currentTeam] == 1){
				//set consecutive home and away games to 0
				state.numberOfConsecutiveHomeGames[currentTeam] = 0;
				state.numberOfConsecutiveAwayGames[currentTeam] = 0;

				previousMatchday = currentMatchday - 1;
				//we can say for sure that the current team has been on an away trip before this matchday
				//so we check how long this away trip was
				while(checkIfPlayedAway(state, currentTeam, previousMatchday)){
					++(state.numberOfConsecutiveAwayGames[currentTeam]);
					if(previousMatchday > 0){
						--previousMatchday;
					}else{
						break;
					}
				}
			}else{//if the number of consecutive away games is 1
				//set consecutive home and away games to 0
				state.numberOfConsecutiveHomeGames[currentTeam] = 0;
				state.numberOfConsecutiveAwayGames[currentTeam] = 0;

				previousMatchday = currentMatchday - 1;
				//we can say for sure that the current team has been at least one game home before this matchday
				//so we check the exact amount of consecutive home games
				while(checkIfPlayedHome(state, currentTeam, previousMatchday)){
					++(state.numberOfConsecutiveHomeGames[currentTeam]);
					if(previousMatchday > 0){
						--previousMatchday;
					}else{
						break;
					}
				}
			}
		}
	}
}

bool IDA::checkIfPlayedAway(const State &state, const int& currentTeam, const int& matchdayToCheck){
	return state.schedule[matchdayToCheck]->hasPlayedAway(currentTeam);
}

bool IDA::checkIfPlayedHome(const State &state, const int& currentTeam, const int& matchdayToCheck){
	return state.schedule[matchdayToCheck]->hasPlayedHome(currentTeam);
}

int* IDA::getTeamOrdering(const int& teamOrderingType, DistanceMatrix* d){
	if(teamOrderingType == 0){
		return noOrdering(d->numberOfTeams);
	}else if(teamOrderingType == 1){
		return maxDistanceOrdering(d);
	}else if(teamOrderingType == 2){
		return minDistanceOrdering(d);
	}else if(teamOrderingType == 3){
		return randomOrdering(d->numberOfTeams);
	}
	return noOrdering(d->numberOfTeams);
}

int* IDA::noOrdering(const int& numberOfTeams){
	int* noOrdering = new int[numberOfTeams];
	for(int i=0; i<numberOfTeams; ++i){
		noOrdering[i] = i;
	}
	return noOrdering;
}

int* IDA::maxDistanceOrdering(DistanceMatrix* d){
	int* distances = getSummedUpDistances(d);
	int* maxDistanceOrdering = new int[d->numberOfTeams];
	int index = -1;
	for(int i=0; i<d->numberOfTeams; ++i){
		index = getIndexOfMax(distances, d->numberOfTeams);
		maxDistanceOrdering[i] = index;
		distances[index] = 0;
	}
	delete[] distances;
	return maxDistanceOrdering;
}

int* IDA::minDistanceOrdering(DistanceMatrix* d){
	int* distances = getSummedUpDistances(d);
	int* minDistanceOrdering = new int[d->numberOfTeams];
	int index = -1;
	for(int i=0; i<d->numberOfTeams; ++i){
		index = getIndexOfMin(distances, d->numberOfTeams);
		minDistanceOrdering[i] = index;
		distances[index] = numeric_limits<int>::max();
	}
	delete[] distances;
	return minDistanceOrdering;
}

int* IDA::randomOrdering(const int& numberOfTeams){
	int* randomOrdering = noOrdering(numberOfTeams);
	srand(time(0));
	random_shuffle(randomOrdering, randomOrdering+numberOfTeams);
	return randomOrdering;
}

int* IDA::getSummedUpDistances(DistanceMatrix* d){
	int* distances = new int[d->numberOfTeams];
	for(int i=0; i<d->numberOfTeams; ++i){
		distances[i] = 0;
		for(int j=0; j<d->numberOfTeams; ++j){
			if(i != j){
				distances[i] += d->distances[i][j];
			}
		}
	}
	return distances;
}

int IDA::getIndexOfMax(int* distances, const int& numberOfTeams){
	int index = -1;
	int max = 0;
	for(int j=0; j<numberOfTeams; ++j){
		if(distances[j] > max){
			max = distances[j];
			index = j;
		}
	}
	return index;
}

int IDA::getIndexOfMin(int* distances, const int& numberOfTeams){
	int index = -1;
	int min = numeric_limits<int>::max();
	for(int j=0; j<numberOfTeams; ++j){
		if(distances[j] < min){
			min = distances[j];
			index = j;
		}
	}
	return index;
}

void IDA::printTeamOrdering(int numberOfTeams){
	cout << "[";
	for(int i=0; i<numberOfTeams; ++i){
		cout << teamOrdering[i];
		if(i != numberOfTeams-1) cout << ",";
	}
	cout << "]" << endl;
}

void IDA::printSubtreeInformation(const int counter, Subtree& currentSubtree,
		Limit& nextFLimit, const Limit& fLimit) {
	lock_guard<mutex> guard(printMutex);
	if(solutionFound) return;

	if (applyForcedDeepening) {
		cout << "next Subtree (#" << counter << ") "
				<< currentSubtree.limit->distance << " from depth "
				<< currentSubtree.limit->depth << " /" << nextFLimit.distance
				<< endl;
	} else {
		cout << "next Subtree (#" << counter << ") "
				<< currentSubtree.limit->distance << " from depth "
				<< currentSubtree.limit->depth << " /" << fLimit.distance
				<< endl;
	}

	currentSubtree.printSubtree();
	cout << endl;
}

void IDA::handleSubtree(const int subtreeNumber, Subtree &currentSubtree, const Limit &fLimit, Limit &nextFLimit,
		const int currentDepthLimit, SubtreeQueue &nextSubtreeQueue, Solution *solution, int threadNumber) {
		//printSubtreeInformation(subtreeNumber, currentSubtree, nextFLimit, fLimit);

	//if subtree is not skipped
	if((applyForcedDeepening && currentSubtree.limit->distance <= nextFLimit.distance)
			|| (!applyForcedDeepening && currentSubtree.limit->distance <= fLimit.distance)){
		currentSubtree.limit->distance = numeric_limits<int>::max();

		//reset all matchday sets and tried elite paths for this search
		resetAllMatchdaySetsAndAllTriedElitePaths(*currentSubtree.state);

		//start search
		Solution* tmp = recursiveSearch(*currentSubtree.state, fLimit, *currentSubtree.limit, nextFLimit,
				currentDepthLimit, subtreeNumber, threadNumber);
		if (tmp != NULL) {
			solution->state = tmp->state;
			solution->expandedNodes = tmp->expandedNodes;
			return;
		}
		--(currentSubtree.state->depth);
	}else{
		if(currentSubtree.limit->distance < nextFLimit.distance){
			lock_guard<mutex> guard(updateMutex);
			nextFLimit.update(currentSubtree.limit->distance, currentSubtree.limit->depth, NULL, false);
		}
	}

	nextSubtreeQueue.push(&currentSubtree);
}
