#include "local_optimum_search.h"

#include "search_common.h"


#include "../plugin.h"

#include "../evaluation_context.h"
#include "../heuristic.h"
#include "../open_list_factory.h"
#include "../option_parser.h"
#include "../pruning_method.h"

#include "../algorithms/ordered_set.h"
#include "../task_utils/successor_generator.h"

#include "../utils/logging.h"

#include <cassert>
#include <cstdlib>
#include <memory>
#include <optional.hh>
#include <set>

using namespace std;

namespace local_optimum_search {
LocalOptimumSearch::LocalOptimumSearch(const Options &opts)
    : SearchEngine(opts),
      heuristic(opts.get<shared_ptr<Evaluator>>("eval"))
      {
      	
      }


void LocalOptimumSearch::initialize() {
    utils::g_log << "Conducting local Optimum search"
                 << endl;

    State initial_state = state_registry.get_initial_state();
    heuristic->notify_initial_state(initial_state);

    EvaluationContext eval_context(initial_state, 0, true, &statistics);

    statistics.inc_evaluated_states();

    
    SearchNode node = search_space.get_node(initial_state);
    node.open_initial();

    open_list.push(initial_state.get_id());

    utils::g_log << "end init"
                 << endl;
    //std::cerr << "##### End Init ########" << endl;
}

void LocalOptimumSearch::print_statistics() const {
    utils::g_log << "no print_statistics() available for local Optimum search"
                 << endl;
}

SearchStatus LocalOptimumSearch::step() {
    tl::optional<SearchNode> node;
    while (!open_list.empty()) {
    	    
        //utils::g_log << "node " << node->get_state().get_id() << " is closed?: " << node->is_closed()
        //         << endl;
        
                
        StateID id = open_list.top();
        //utils::g_log << "open_list top ID: " << id
        //         << endl;
        
        open_list.pop();
        //utils::g_log << "open_list empty after pop?" << open_list.empty()
        //         << endl;
        State s = state_registry.lookup_state(id);
        
        node.emplace(search_space.get_node(s)); //?
        
        if (node->is_closed()){
            //utils::g_log << "this node is closed" 
            //     << endl;
            continue;
	}
	vector<OperatorID> applicable_ops;
        successor_generator.generate_applicable_ops(s, applicable_ops);
	
        
        EvaluationContext eval_context(s, node->get_g(), false, &statistics);
        
        int parent_h = eval_context.get_evaluator_value_or_infinity(heuristic.get());
        //utils::g_log << "<parent_h:" << parent_h << ">"
        //         << endl;
	bool has_improving_succ = false;
	

        for (OperatorID op_id : applicable_ops) {
            
            
            
            
                 
            OperatorProxy op = task_proxy.get_operators()[op_id];
            //utils::g_log << "( " << op.get_name() << " )"
            //     << endl;

            State succ_state = state_registry.get_successor_state(s, op);
            statistics.inc_generated();
        
            SearchNode succ_node = search_space.get_node(succ_state);

            heuristic->notify_state_transition(s, op_id, succ_state);
        

            // Previously encountered dead end. Don't re-evaluate.
            if (succ_node.is_dead_end())
                continue;

            if (true){
                int succ_g = node->get_g() + get_adjusted_cost(op);

                EvaluationContext succ_eval_context(
                    succ_state, succ_g, true, &statistics);
                statistics.inc_evaluated_states();

            
                int succ_h = succ_eval_context.get_evaluator_value_or_infinity(heuristic.get());
                //utils::g_log << "<succ_h:" << succ_h << ">"
                //    << endl;
                if (succ_h < parent_h){
            	    has_improving_succ = true;
                    succ_node.open(*node, op, get_adjusted_cost(op));
                    if (!check_goal_and_set_plan(succ_state)){
                        open_list.push(succ_state.get_id());
                    }
            
                    
                }
            
            } else {
                utils::g_log << "<node is not new>"
                    << endl; 
            }
        }
        
        
    	if (!has_improving_succ){
    	    utils::g_log << "FOUND STATE WITHOUT IMPROVING SUCC  " << id
                 << endl;
    	    //dump_search_space();
    	    
    	    
    	    
    	    //utils::g_log << "node: " << node->dump(task_proxy)
            //     << endl;
            //search_space.trace_path(task_proxy);
            //succ_nodeinfo.creating_operator
            
            vector<OperatorID> path;
            utils::g_log << "...  " 
                 << endl;
            search_space.trace_path(s, path);

            utils::g_log << "......"
                    << endl;
            //search_space.trace_path(s,my_path);
            
            
            
            
            
            utils::g_log << "== begin path =="
                    << endl;
            for (OperatorID path_op_id : path){
                OperatorProxy path_op = task_proxy.get_operators()[path_op_id];
                utils::g_log << path_op.get_name() << " #" << path_op_id
                    << endl;
            }
            utils::g_log << "== end path =="
                    << endl;
            utils::g_log << "== applicable operators from local optimum =="
                    << endl;
            for (OperatorID op_id : applicable_ops) {                 
                OperatorProxy op = task_proxy.get_operators()[op_id];
                utils::g_log << op.get_name() << " #" << op_id
                    << endl;
            }
            utils::g_log << "== end of applicable operators from local optimum =="
                    << endl;
            
    	    return FAILED;
    	}
    	//utils::g_log << "has_improving_succ"
        //         << endl;
    	
    	//utils::g_log << "open list empty? " << open_list.empty()
        //         << endl;
        

        node->close();
        assert(!node->is_dead_end());
        
        statistics.inc_expanded();
        
    }
    utils::g_log << "Completely explored state space -- no local optimum reachable by a descending path!" << endl;
    //dump_search_space();
    return SOLVED;



}



void LocalOptimumSearch::dump_search_space() const {
    search_space.dump(task_proxy);
}


static shared_ptr<SearchEngine> _parse(OptionParser &parser) {
    parser.add_option<shared_ptr<Evaluator>>("eval");
    SearchEngine::add_options_to_parser(parser);
    Options opts = parser.parse();
    if (parser.dry_run()){
        return nullptr;
    }
    return make_shared<LocalOptimumSearch>(opts);    
}

static Plugin<SearchEngine> _plugin("los", _parse);
}
