#include "disj_action_landmark_graph.h"

#include <cassert>

using namespace std;

namespace landmarks {
DisjActionLandmarkNode::DisjActionLandmarkNode(std::set<int> actions)
    : actions(move(actions)) {
}

bool DisjActionLandmarkNode::overlaps_with(
    DisjActionLandmarkNode &other) const {

    vector<int> intersect;
    set_intersection(actions.begin(), actions.end(),
                     other.actions.begin(), other.actions.end(),
                     back_inserter(intersect));
    return !intersect.empty();
}

bool DisjActionLandmarkNode::satisfied_by(int op_id) const {
    return actions.find(op_id) != actions.end();
}

void DisjActionLandmarkNode::add_dependency(int node_id, OrderingType type) {
    if (depends_on(node_id)) {
        add_dependency_if_stronger(node_id, type);
    } else {
        dependencies.emplace(node_id, type);
    }
}

void DisjActionLandmarkNode::add_dependency_if_stronger(
    int node_id, OrderingType type) {

    assert(depends_on(node_id));
    if (dependencies[node_id] == OrderingType::weak
        && type == OrderingType::strong) {

        dependencies[node_id] = type;
    }
}

OrderingType DisjActionLandmarkNode::get_dependency(int node_id) const {
    assert(depends_on(node_id));
    return dependencies.at(node_id);
}

map<int, OrderingType> DisjActionLandmarkNode::get_dependencies() const {
    return dependencies;
}

bool DisjActionLandmarkNode::depends_on(int node_id) const {
    return dependencies.count(node_id);
}

DisjActionLandmarkGraph::DisjActionLandmarkGraph(
    const LandmarkGraph &lm_graph, const State &init_state) {

    add_nodes(lm_graph, init_state);
    add_edges(lm_graph, init_state);
}

void DisjActionLandmarkGraph::add_nodes(const LandmarkGraph &lm_graph,
                                        const State &init_state) {
    for (const unique_ptr<LandmarkNode> &node : lm_graph.get_nodes()) {
        if (!node->is_true_in_state(init_state) || !node->parents.empty()) {
            //add_node(node->possible_achievers);
            add_node(node->first_achievers);
        }
    }
}

void DisjActionLandmarkGraph::add_node(const set<int> &actions) {
    if (ids.find(actions) == ids.end()) {
        int id = ids.size();
        ids[actions] = id;
        DisjActionLandmarkNode dalm(actions);
        lms.emplace_back(move(dalm));
    }
}

void DisjActionLandmarkGraph::add_edges(const LandmarkGraph &lm_graph,
                                        const State &init_state) {
    for (auto &node : lm_graph.get_nodes()) {
        if (node->is_true_in_state(init_state)) {
            /* All edges starting in initially true facts are not
               interesting for us since the cycles they possibly induce
               are already resolved initially. */
            continue;
        }
        for (std::pair<LandmarkNode* const, EdgeType> &child : node->children) {
            add_edge(*node, child);
        }
    }
}

void DisjActionLandmarkGraph::add_edge(
    LandmarkNode &from, std::pair<LandmarkNode* const, EdgeType> &to) {

    int from_id = ids[from.possible_achievers];
    int to_id = ids[to.first->possible_achievers];

    /* If there is an action which occurs in both landmarks, applying it
       resolves both landmarks as well as the ordering in one step.
       This special case (which is a consequence of the definition of
       reasonable orderings) makes a lot of things very complicated.
       Ignoring these cases may be desired sometimes which is why we do
       not take them over into our DALM-graph here if the
       *keep_intersecting_orderings* flag is set to false (default). */
    if (to.second >= EdgeType::NATURAL) {
        add_edge(from_id, to_id, OrderingType::strong);
    } else if (to.second == EdgeType::REASONABLE
        && !lms[from_id].overlaps_with(lms[to_id])) {
        add_edge(from_id, to_id, OrderingType::weak);
    }
}

void DisjActionLandmarkGraph::add_edge(int from, int to,
                                       OrderingType type) {
    assert(0 <= from && from < static_cast<int>(ids.size()));
    assert(0 <= to && to < static_cast<int>(ids.size()));

    if (!lms[to].depends_on(from)
        || lms[to].get_dependency(from) == OrderingType::weak) {

        lms[to].add_dependency(from, type);
    }
}

void DisjActionLandmarkGraph::dump_lm(int id, const TaskProxy &task) const {
    OperatorsProxy operators = task.get_operators(); 
    //cout << "lm" << id << ": <\n";
    //cout << "lm" << id << " " << lms[id].actions.size() << "<\n"; 
    //cout << "lm\n" << lms[id].actions.size() << "\n<\n"; 
    cout << "<\n"; 
    for (int action : lms[id].actions) {
        cout << action << " ";
        //cout << operators[action].get_name() << " \n"; 
        //cout << "Preconditions " << operators[action].get_preconditions().size() << ":" << endl; 
        /*for (auto fact : operators[action].get_preconditions()) {
		//cout << fact.get_name() << endl;
		//cout << fact.get_pair().var << " " << fact.get_pair().value << endl; 
	} 
	//cout << "Effects " << operators[action].get_effects().size() << ":" << endl; 
	for (auto fact : operators[action].get_effects()) {
		//cout << fact.get_fact().get_name() << endl;
		//cout << fact.get_pair().var << " " << fact.get_pair().value << endl; 
	} */
    }
    cout << "\n#"; 
    for (int action : lms[id].actions) {
    	cout << operators[action].get_name() << ", ";
    }
    cout << "\n>" << endl;

    /*if (!lms[id].get_dependencies().empty()) {
        cout << "\tdepends on ";
        for (auto dep : lms[id].get_dependencies()) {
            cout << "lm" << dep.first << "(";
            switch (dep.second) {
            case OrderingType::strong:
                cout << "s";
                break;
            case OrderingType::weak:
                cout << "w";
                break;
            default:
                cout << "?";
                break;
            }
            cout << ") ";
        }
        cout << endl; 
    }*/
}

void DisjActionLandmarkGraph::dump(const TaskProxy &task) const {
    cout << "== Disjunctive Action Landmark Graph ==" << endl;
    for (size_t id = 0; id < lms.size(); ++id) {
        dump_lm(id, task);
    }
    cout << "== End of Graph ==" << endl;
}

void DisjActionLandmarkGraph::dump_dot() const {
    cout << "digraph graphname {\n";
    for (size_t id = 0; id < lms.size(); ++id) {
        cout << "  lm" << id << " [label=\"<";
        for (int a : lms[id].actions) {
            cout << a << " ";
        }
        cout << ">\"];\n";
    }
    cout << "\n";
    for (size_t id = 0; id < lms.size(); ++id) {
        for (pair<int, OrderingType> dep : lms[id].get_dependencies()) {
            cout << "  lm" << dep.first << " -> lm" << id;
            if (dep.second == OrderingType::weak) {
                cout << " [style=dotted]";
            }
            cout << ";\n";
        }
    }
    cout << "}" << endl;
}

size_t DisjActionLandmarkGraph::get_number_of_landmarks() {
    return lms.size();
}

OrderingType DisjActionLandmarkGraph::get_ordering_type(int from, int to) {
    assert(0 <= from && from < static_cast<int>(lms.size()));
    assert(0 <= to && to < static_cast<int>(lms.size()));
    assert(lms[to].depends_on(from));
    return lms[to].get_dependency(from);
}

std::set<int> DisjActionLandmarkGraph::get_actions(int id) {
    assert(0 <= id && id < static_cast<int>(lms.size()));
    return lms[id].actions;
}

std::map<int, OrderingType> DisjActionLandmarkGraph::get_dependencies(int id) {
    assert(0 <= id && id < static_cast<int>(lms.size()));
    return lms[id].get_dependencies();
}
}
