#include "parity_heuristic.h"

#include "constraint_constructor.h"
#include "xor_solver.h"

#include "../option_parser.h"
#include "../plugin.h"

#include "../utils/logging.h"
#include "../utils/memory.h"
#include "../utils/system.h"

#include <memory>

using namespace std;
using utils::ExitCode;

namespace parity_potentials {
ParityHeuristic::ParityHeuristic(const options::Options &opts)
    : Heuristic(opts),
      h_value(0) {
    utils::g_log << "Start of parity potential computation." << endl;
    utils::Timer construction_timer;
    ConstraintConstructor cc(opts);
    utils::g_log << "Constructing constraints... " << endl;
    AugmentedMatrix constraint_matrix = cc.construct_constraints();
    construction_timer.stop();
    utils::g_log << "done!" << endl;
    utils::g_log << "Construction time: " << construction_timer << endl;
    utils::g_log << "Number of rows: "
                 << constraint_matrix.get_num_rows()
                 << endl;
    utils::g_log << "Number of columns: "
                 << constraint_matrix.get_num_cols()
                 << endl;
    utils::g_log << "Solving constraints... ";
    utils::Timer solve_timer;
    XorSolver solver(move(constraint_matrix));
    bool task_unsolvable = solver.is_unsolvable();
    solve_timer.stop();
    utils::g_timer.stop();
    utils::g_log << "done!" << endl;
    utils::g_log << "Solve time: " << solve_timer << endl;
    utils::g_log << "Total time: " << utils::g_timer << endl;
    if (task_unsolvable) {
        utils::g_log << "Separation function found! "
                     << "Task is unsolvable." << endl;
        if (opts.get<bool>("verify")) {
            utils::g_log << "Calculating solution weights... ";
            Bitset solution = solver.compute_solution_weights();
            utils::g_log << "Verifying solution... ";
            if (!solver.verify_solution(solution)) {
                utils::g_log << "NOT VERIFIED!" << endl;
                utils::exit_with(ExitCode::SEARCH_CRITICAL_ERROR);
            } else {
                utils::g_log << "verified!" <<endl;
            }
        }
        utils::exit_with(ExitCode::SEARCH_UNSOLVABLE);
    } else {
        utils::g_log << "No separation function found. "
                     << "Task may be solvable." << endl;
        utils::exit_with(ExitCode::SEARCH_UNSOLVED_INCOMPLETE);
    }
}

int ParityHeuristic::compute_heuristic(const GlobalState &/*global_state*/) {
    return h_value;
}

static shared_ptr<Heuristic> _parse(OptionParser &parser) {
    parser.add_option<bool>(
        "verify",
        "compute solution weights and verify their correctness",
        "false");
    Heuristic::add_options_to_parser(parser);
    Options opts = parser.parse();
    if (parser.dry_run())
        return nullptr;
    else
        return make_shared<ParityHeuristic>(opts);
}

static Plugin<Evaluator> _plugin("parity", _parse);
}
