#! /usr/bin/env python

"""Check for false positives on easy solvable tasks from all domains."""

import os
import os.path
import platform
from pathlib import Path

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.scatter import ScatterPlotReport
from lab.reports import *
from lab import cached_revision
from lab.environments import BaselSlurmEnvironment, LocalEnvironment


NODE = platform.node()
if NODE.endswith(".scicore.unibas.ch") or NODE.endswith(".cluster.bc2.ch"):
	ENV = BaselSlurmEnvironment(email="remo.christen@stud.unibas.ch")
else:
	ENV = LocalEnvironment(processes=2)

SUITE = ["3unsat",
	"bag-barman",
	"bag-gripper",
	"bag-transport",
	"bottleneck",
	"cave-diving",
	"chessboard-pebbling",
	"diagnosis",
	"document-transfer",
	"mystery",
	"over-nomystery",
	"over-rovers",
	"over-tpp",
	"pegsol",
	"pegsol-row5",
	"sliding-tiles",
	"tetris",
	"unsat-nomystery",
	"unsat-rovers",
	"unsat-tpp"]

# Use path to your Fast Downward repository.
REPO = os.environ["DOWNWARD_REPO"]
BENCHMARKS_DIR = os.environ["ERIKSSON_BENCHMARKS"]

# If REVISION_CACHE is None, the default ./data/revision-cache is used.
REVISION_CACHE = os.environ.get("DOWNWARD_REVISION_CACHE")

exp = FastDownwardExperiment(environment=ENV, revision_cache=REVISION_CACHE)

# Add built-in parsers to the experiment.
exp.add_parser(exp.EXITCODE_PARSER)
exp.add_parser(exp.TRANSLATOR_PARSER)
exp.add_parser(exp.SINGLE_SEARCH_PARSER)
exp.add_parser(exp.PLANNER_PARSER)
DIR = Path(__file__).resolve().parent
exp.add_parser(DIR / "timer-parser.py")
exp.add_parser(DIR / "matrix-parser.py")

exp.add_suite(BENCHMARKS_DIR, SUITE)


COMMAND = ["--search", "astar(parity())"]
exp.add_algorithm("Full Bitset", REPO, "parity-functions", COMMAND)
exp.add_algorithm("Sparse vector<vector<int>>", REPO, "sparse", COMMAND)
exp.add_algorithm("Sparse vector<set<int>>", REPO, "sparse-set", COMMAND)

# Add step that writes experiment files to disk.
exp.add_step("build", exp.build)

# Add step that executes all runs.
exp.add_step("start", exp.start_runs)

# Add step that collects properties from run directories and
# writes them to *-eval/properties.
exp.add_fetcher(name="fetch")

exp.add_parse_again_step()

# Add report step (AbsoluteReport is the standard report).
"""
Default values for Attribute:
name = Attribute(
	"name",
	absolute=False,
	min_wins=True,
	function=sum,
	scale=None,
	digits=2
"""
construction_time = Attribute(
	"construction_time",
	function=geometric_mean,
)
solve_time = Attribute(
	"solve_time",
	function=geometric_mean,
)
memory = Attribute(
	"memory",
	function=geometric_mean,
)
num_cols = Attribute(
	"num_cols",
	absolute=True,
)
num_rows = Attribute(
	"num_rows",
	absolute=True,
)
num_ops = Attribute(
	"num_ops",
	absolute=True,
)
ATTRIBUTES = ["error",
	memory,
	construction_time,
	solve_time,
	"total_time",
	num_rows,
	num_cols,
	num_ops]

REPORT_DOMAINS = list(set(SUITE)-set(["diagnosis"]))
exp.add_report(AbsoluteReport(attributes=ATTRIBUTES, filter_domain=REPORT_DOMAINS), outfile="report.html")

def domain_as_category(run1, run2):
	return run1["domain"]

# Test Scatter Plot
exp.add_report(ScatterPlotReport(attributes=["memory"], filter_algorithm=["Full Bitset", "Sparse vector<set<int>>"], get_category=domain_as_category, show_missing=False, format="tex",), name="test-plot")

# Error Result Tables
# Full Bitset
exp.add_report(AbsoluteReport(attributes=["error"], filter_domain=REPORT_DOMAINS, filter_algorithm=["Full Bitset"], format="tex"), outfile="full-bitset-error.tex")
# Sparse Set
exp.add_report(AbsoluteReport(attributes=["error"], filter_domain=REPORT_DOMAINS, filter_algorithm=["Sparse vector<set<int>>"], format="tex"), outfile="sparse-set-error.tex")
# Sparse Vector
exp.add_report(AbsoluteReport(attributes=["error"], filter_domain=REPORT_DOMAINS, filter_algorithm=["Sparse vector<vector<int>>"], format="tex"), outfile="sparse-vector-error.tex")

# Full Bitset vs Sparse Set Plots
# Memory
exp.add_report(ScatterPlotReport(attributes=["memory"], filter_algorithm=["Full Bitset", "Sparse vector<set<int>>"], get_category=domain_as_category, show_missing=True, format="tex",), name="full-bitset-vs-sparse-set-memory")
# Time
exp.add_report(ScatterPlotReport(attributes=["total_time"], filter_algorithm=["Full Bitset", "Sparse vector<set<int>>"], get_category=domain_as_category, show_missing=True, format="tex",), name="full-bitset-vs-sparse-set-time")

# X vs Sparse Vector Plots
# Memory Sparse Set vs Sparse Vector
exp.add_report(ScatterPlotReport(attributes=["memory"], filter_algorithm=["Sparse vector<set<int>>", "Sparse vector<vector<int>>"], filter_domain=REPORT_DOMAINS, get_category=domain_as_category, show_missing=True, format="tex",), name="sparse-set-vs-sparse-vector-memory")
# Time Full Bitset vs Sparse Vector
exp.add_report(ScatterPlotReport(attributes=["total_time"], filter_algorithm=["Full Bitset", "Sparse vector<vector<int>>"], filter_domain=REPORT_DOMAINS, get_category=domain_as_category, show_missing=True, format="tex",), name="full-bitset-vs-sparse-vector-time")

# Geometric mean of num_ops
num_ops = Attribute(
	"num_ops",
	function=geometric_mean,
	absolute=True,
)
exp.add_report(AbsoluteReport(attributes=[num_ops], filter_domain=REPORT_DOMAINS), outfile="geometric-num-ops.html")


# Parse the commandline and show or run experiment steps.
exp.run_steps()
