#! /usr/bin/env python2.7

"""Solve some tasks with A* and the LM-Cut heuristic."""

import os
import os.path
import platform

from lab.environments import LocalEnvironment, BaselSlurmEnvironment

from downward.experiment import FastDownwardExperiment
from downward.reports.absolute import AbsoluteReport
from downward.reports.scatter import ScatterPlotReport


ATTRIBUTES = ['evaluations', 'coverage', 'error', 'expansions', 'total_time', 'plan_length', 'backwards_ops_in_plan', 'backwards_expansions', 'original_goal_found', 'initial_state_found', 'undone', 'syntax_warning', 'goal_expansion_stopped']

NODE = platform.node()
if NODE.endswith(".scicore.unibas.ch") or NODE.endswith(".cluster.bc2.ch"):
    # Create bigger suites with suites.py from the downward-benchmarks repo.
    # This is optimal_strips
    SUITE = ['gripper']
    ENV = BaselSlurmEnvironment(email="remo.christen@stud.unibas.ch")
else:
    SUITE = ['gripper']
#    SUITE = ['gripper:prob01.pddl', 'gripper:prob02.pddl', 'gripper:prob03.pddl', 'gripper:prob04.pddl', 'gripper:prob05.pddl', 'gripper:prob06.pddl', 'gripper:prob07.pddl']
    ENV = LocalEnvironment(processes=2)
# Use path to your Fast Downward repository.
REPO = os.environ["DOWNWARD_REPO"]
BENCHMARKS_DIR = os.environ["DOWNWARD_BENCHMARKS"]
REVISION_CACHE = os.path.expanduser('~/lab/revision-cache')

exp = FastDownwardExperiment(environment=ENV, revision_cache=REVISION_CACHE)
sage_parser = os.path.abspath(os.path.join(__file__, '../sage_parser.py'))
exp.add_parser(sage_parser)
# Add built-in parsers to the experiment.
exp.add_parser(exp.EXITCODE_PARSER)
exp.add_parser(exp.TRANSLATOR_PARSER)
exp.add_parser(exp.SINGLE_SEARCH_PARSER)
exp.add_parser(exp.PLANNER_PARSER)
exp.add_suite(BENCHMARKS_DIR, SUITE)

# new_min
exp.add_algorithm(
    'new_min most sat',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=new_min, operator_order=most_sat)], use_bgg=true)'])

exp.add_algorithm(
    'new_min lowest p',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=new_min, operator_order=lowest_layer)], use_bgg=true)'])

exp.add_algorithm(
    'new_min fifo',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=new_min)], use_bgg=true)'])

# counter
exp.add_algorithm(
    'counter 50 90%  most sat',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=counter, state_value=50, percentage=90, operator_order=most_sat)], use_bgg=true)'])

exp.add_algorithm(
    'counter 50 90% lowest p',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=counter, state_value=50, percentage=90, operator_order=lowest_layer)], use_bgg=true)'])

exp.add_algorithm(
    'counter 50 90% fifo',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=counter, state_value=50, percentage=90)], use_bgg=true)'])

# diff
exp.add_algorithm(
    'diff 0 most sat',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=diff, state_value=0, operator_order=most_sat)], use_bgg=true)'])

exp.add_algorithm(
    'diff 0 lowest p',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=diff, state_value=0, operator_order=lowest_layer)], use_bgg=true)'])

exp.add_algorithm(
    'diff 0 fifo',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff(state_decision=diff, state_value=0)], use_bgg=true)'])

exp.add_algorithm(
    'plain eager_greedy',
    REPO,
    'fix_seg',
    ['--search', 'eager_greedy([ff()], use_bgg=false)'])

#exp.add_algorithm(
#        'ff', REPO, 'default', ['--search', 'astar(ff())'], build_options=["--debug"], driver_options=["--debug"])

# Add step that writes experiment files to disk.
exp.add_step('build', exp.build)

# Add step that executes all runs.
exp.add_step('start', exp.start_runs)

# Add step that collects properties from run directories and
# writes them to *-eval/properties.
exp.add_fetcher(name='fetch')

# Add report step (AbsoluteReport is the standard report).
exp.add_report(
    AbsoluteReport(attributes=ATTRIBUTES), outfile='report.html')

# Add scatter plot report step.
exp.add_report(
    ScatterPlotReport(
        attributes=["expansions"], filter_algorithm=["blind", "lmcut"]),
    outfile='scatterplot.png')

# Parse the commandline and show or run experiment steps.
exp.run_steps()
