#include "relaxation_heuristic.h"

#include "../task_utils/task_properties.h"
#include "../utils/collections.h"
#include "../utils/timer.h"

#include <algorithm>
#include <cassert>
#include <cstddef>
#include <unordered_map>
#include <vector>

using namespace std;

namespace relaxation_heuristic {
Proposition::Proposition()
    : cost(-1),
      reached_by(NO_OP),
      is_goal(false),
      marked(false),
      num_precondition_occurences(-1) {
}


UnaryOperator::UnaryOperator(
    int num_preconditions, array_pool::ArrayPoolIndex preconditions,
    PropID effect, int operator_no, int base_cost)
    : effect(effect),
      base_cost(base_cost),
      num_preconditions(num_preconditions),
      preconditions(preconditions),
      operator_no(operator_no) {
}


// construction and destruction
RelaxationHeuristic::RelaxationHeuristic(const options::Options &opts)
    : Heuristic(opts),
      num_expansions(0),
      no_expansion_possible(false),
      use_bgg(false),
      cost(0),
      h_initial(-1),
      h_min(numeric_limits<int>::max()),
      state_value(0),
      percentage(1) {
    // Build propositions.
    propositions.resize(task_properties::get_num_facts(task_proxy));

    // Build proposition offsets.
    VariablesProxy variables = task_proxy.get_variables();
    proposition_offsets.reserve(variables.size());
    PropID offset = 0;
    for (VariableProxy var : variables) {
        proposition_offsets.push_back(offset);
        offset += var.get_domain_size();
    }
    assert(offset == static_cast<int>(propositions.size()));

    // Build goal propositions.
    GoalsProxy goals = task_proxy.get_goals();
    goal_propositions.reserve(goals.size());
    for (FactProxy goal : goals) {
        PropID prop_id = get_prop_id(goal);
        propositions[prop_id].is_goal = true;
        goal_propositions.push_back(prop_id);
    }

    // Build unary operators for operators and axioms.
    unary_operators.reserve(
        task_properties::get_num_total_effects(task_proxy));
    for (OperatorProxy op : task_proxy.get_operators())
        build_unary_operators(op);
    for (OperatorProxy axiom : task_proxy.get_axioms())
        build_unary_operators(axiom);

    // Simplify unary operators.
    utils::Timer simplify_timer;
    simplify();
    cout << "time to simplify: " << simplify_timer << endl;

    // Cross-reference unary operators.
    vector<vector<OpID>> precondition_of_vectors(propositions.size());

    int num_unary_ops = unary_operators.size();
    for (OpID op_id = 0; op_id < num_unary_ops; ++op_id) {
        for (PropID precond : get_preconditions(op_id))
            precondition_of_vectors[precond].push_back(op_id);
    }

    int num_propositions = propositions.size();
    for (PropID prop_id = 0; prop_id < num_propositions; ++prop_id) {
        auto precondition_of_vec = move(precondition_of_vectors[prop_id]);
        propositions[prop_id].precondition_of =
            precondition_of_pool.append(precondition_of_vec);
        propositions[prop_id].num_precondition_occurences = precondition_of_vec.size();
    }

    // Initialize the proposition layers, starting with 1 for the original goal facts.
    for (FactProxy const &goal : task_proxy.get_goals()) {
        proposition_layers.insert(make_pair(get_prop_id(goal), 1));
    }
}

// Operator sorting functions
void RelaxationHeuristic::most_satisfied_first(vector<candidate> &candidates) {
    sort(candidates.begin(), candidates.end(),
        [](const candidate &a, const candidate &b) {
            return a.second.size() > b.second.size();
        }
    );
}

void RelaxationHeuristic::lowest_proposition_layer_first(vector<candidate> &candidates) {
    // Build a map with the operator ID as a key and a multiset of the layers at which the goal propositions appear.
    map<int, multiset<int>> all_satisfied_layers;
    for (const candidate &c : candidates) {
        for (const FactProxy &replaced_goal : c.second) {
            assert(proposition_layers.at(get_prop_id(replaced_goal)) <= (int)task_proxy.get_ex_goals().size());
            all_satisfied_layers[c.first.get_id()].insert(proposition_layers.at(get_prop_id(replaced_goal)));
        }    
    }

    int counter = 0;
    function<bool(candidate, candidate)> compare_layers;
    compare_layers = [&all_satisfied_layers, &counter, &compare_layers](const candidate &a, const candidate &b) {
        multiset<int> satisfied_layers_a = all_satisfied_layers.at(a.first.get_id());
        multiset<int> satisfied_layers_b = all_satisfied_layers.at(b.first.get_id());
        multiset<int>::iterator it_a = satisfied_layers_a.begin();
        multiset<int>::iterator it_b = satisfied_layers_b.begin();
        advance(it_a, counter);
        advance(it_b, counter);
        assert(satisfied_layers_a.size() > (size_t)counter);
        assert(satisfied_layers_b.size() > (size_t)counter);
        int current_a = *(it_a);
        int current_b = *(it_b);
        if(current_a < current_b) {
            counter = 0;
            return true;
        } else if (current_a > current_b) {
            counter = 0;
            return false;
        } else { // The current value is the same.
            assert(current_a == current_b);
            int count_of_current_a = satisfied_layers_a.count(current_a);
            int count_of_current_b = satisfied_layers_b.count(current_b); 
            if (count_of_current_a > count_of_current_b) {
                counter = 0;
                return true;
            } else if (count_of_current_a < count_of_current_b) {
                counter = 0;
                return false;
            } else {
                assert(count_of_current_a == count_of_current_b);
                counter += count_of_current_a;
                if ((size_t)counter == satisfied_layers_a.size()) { // b is longer than a or both are the same.
                    // Has to be false to ensure strict weak ordering.
                    counter = 0;
                    return false;
                } else if ((size_t)counter == satisfied_layers_b.size()) {
                    counter = 0;
                    return true;
                } else {
                    return compare_layers(a, b);
                }
            }
        }
    };
    sort(candidates.begin(), candidates.end(), compare_layers);
}

// Try to expand with a specific operator (candidate), return false if not possible.
bool RelaxationHeuristic::expand_goal(const candidate &c, const vector<FactProxy> &old_goals) {
    OperatorProxy op_proxy = c.first;
    vector<FactProxy> replaced_goal_facts = c.second;
    vector<PropID> deleted_goals;
    set<FactProxy> new_goals; // Set because we don't want duplicates.
    // Add the old goal facts that are not replaced by the operator to new_goals.
    set_difference(
        old_goals.begin(), old_goals.end(),
        replaced_goal_facts.begin(), replaced_goal_facts.end(),
        inserter(new_goals, new_goals.begin())
    );
    
    // Check for mutex between preconditions of the operator and old goals that are not replaced. If not mutex add the precondition to new_goals, if mutex go to next candidate.
    for (FactProxy precon : op_proxy.get_preconditions()) {
        for (FactProxy new_goal : new_goals) {
            if (precon.is_mutex(new_goal)) {
                add_to_blocked(op_proxy.get_id(), true);
                //cout << "This candidate was not valid! (is mutex): " << op_proxy.get_name() << " between " << precon.get_name() << " and " << new_goal.get_name() << endl;
                return false;
            }
        }
        new_goals.insert(precon);
    }

    // Check whether the expansion with this candidate leads to a subset of a goal we have seen before.
    vector<FactPair> new_goal_pairs;
    for (FactProxy new_goal : new_goals) {
        new_goal_pairs.push_back(new_goal.get_pair());
    }   
    sort(new_goal_pairs.begin(), new_goal_pairs.end());

    // Print for testing
    /*
    cout << "new pairs: ";
    for (FactPair pair : new_goal_pairs) {
        cout << pair << ", ";
    }
    cout << endl;
    */

    // Check if the resulting new goal is dominated by a past goal.
    for (ExGoal ex_goal : task_proxy.get_ex_goals()) {
        vector<FactPair> ex_pairs = ex_goal.ex_goal_pairs;
        sort(ex_pairs.begin(), ex_pairs.end());
        
        // Print for testing
        /*
        cout << "Ex goal: ";
        for (FactPair pair : ex_pairs) {
            cout << pair << ", ";
        }
        cout << endl;
        */

        if (includes(new_goal_pairs.begin(), new_goal_pairs.end(), ex_pairs.begin(), ex_pairs.end())) {
            add_to_blocked(op_proxy.get_id(), true);
            //cout << "This candidate was not valid! (is ex goal): " << op_proxy.get_name() << endl;
            return false;
        }
    }

    // Construct new_goals vector from set for update_goal functions.
    vector<FactProxy> new_goals_vec(new_goals.begin(), new_goals.end());

    // Update the goal propositions in RelaxationHeuristic.
    update_goal_propositions(new_goals_vec, old_goals);
    // Update the goal in task_proxy and add the new ex_goal.
    task_proxy.update_goals(new_goals_vec, op_proxy.get_id());
    
    // Update proposition layers.
    for (FactProxy replaced_goal : c.second) {
        proposition_layers.erase(get_prop_id(replaced_goal));
    }
    int current_layer = task_proxy.get_ex_goals().size();
    for (FactProxy precon : op_proxy.get_preconditions()) {
        proposition_layers.insert(make_pair(get_prop_id(precon), current_layer));
    }

    // Print for testing.
    /*
    for (FactProxy fact : new_goals) {
        cout << "NEW Fact: " << fact.get_name() << endl;
    }
    */

    /* Test for whether goal propositions are adjusted correctly
    for (PropID id : goal_propositions) {
            pair<int,int> pair = get_var_val(id);
            cout << "goal prop: " << pair.first << " = " << pair.second << endl;
    }

    for (size_t i = 0; i < propositions.size(); ++i) {
        if (propositions[i].is_goal) {
            pair<int,int> pair = get_var_val(i);
            cout << "is_goal: " << pair.first << " = " << pair.second << endl;
        }
    }
    */
    
    return true;
}

// Adds operators to blocked list
// next_level is true when blocking operators that are being blocked for future expansion checks on the same expansion level.
void RelaxationHeuristic::add_to_blocked(const OpID to_add, const bool next_level) {
    int level = task_proxy.get_ex_goals().size();
    if (next_level) {
        ++level;
    }
    map<int, set<OpID>>::iterator it = blocked_ops.find(level);

    //cout << "Blocking op with id " << to_add << " at level " << level <<  endl;
    
    if (it == blocked_ops.end()) {
        set<OpID> s;
        s.insert(to_add);
        blocked_ops.insert({level, s});
    } else {
        it->second.insert(to_add);
    }
}

// This function collects the goals that are replaced by an operator and sorts out operators that delete a goal proposition.
void RelaxationHeuristic::get_replacing_facts(vector<candidate> &candidates, const vector<FactProxy> &old_goals, const OperatorProxy &op) {
    candidates.push_back(make_pair(op, vector<FactProxy> {}));
    for (FactProxy goal : old_goals) {
        for (EffectProxy eff : op.get_effects()) {
            if (goal.get_pair().var == eff.get_fact().get_pair().var) {
                if (goal.get_pair().value == eff.get_fact().get_pair().value) {
                    candidates.back().second.push_back(goal);
                    break;
                } else {
                    // The operator deletes a goal proposition, thus cannot be used to expand goal.
                    candidates.pop_back();
                    add_to_blocked(op.get_id(), true);
                    //cout << "This candidate was not valid! (deletes goal): " << op.get_name() << endl;
                    return;
                }
            }
        }
    }
    if (candidates.back().second.size() == 0) {
        candidates.pop_back();
        return;
    }
}

// This function is called form compute_heuristic in FF.
// It checks whether the goal should be expanded at all, 
// if yes it tries to do so with the given operators and 
// returns true upon successful expansion, false if it did
// not expand or not try to.
bool RelaxationHeuristic::check_expansion(const vector<OperatorProxy> &ops, const int h_value) {

    // Do not expand if all possible expansions have been blocked.
    if (no_expansion_possible) {
        return false;
    }

    /*
    for (auto &op : ops) {
        cout << "Possible operators " << op.get_name() << endl;
    }
    */

    //stupid fix for count decision
    vector<OperatorProxy> ops_tmp = ops;

    // Check whether this state should be expanded based on the chosen method.
    if (state_decision.compare("new_min") == 0) {
        if (h_value >= h_min) {
            return false;
        } else {
            h_min = h_value;
        }
    } else if (state_decision.compare("counter") == 0) {
        set<OpID> ops_set;
        for (OperatorProxy &op : ops_tmp) {
            ops_set.insert(op.get_id());
        }

        // Add current operators.
        for (OpID op_id : ops_set) {
            map<OpID, int>::iterator it = op_counter.find(op_id);
            if (it == op_counter.end()) {
                op_counter.insert({op_id, 1});
            } else {
                it->second = it->second + 1;
            }
        }
        ++num_expansions;

        // If enough expansions have passed, collect all common operators.
        if (num_expansions >= state_value) {
            ops_tmp.clear();
            for (auto const &element : op_counter) {
                if (element.second >= num_expansions * (percentage / 100)) {
                    ops_tmp.push_back(task_proxy.get_operators()[element.first]);
                }
            }

            if (ops_tmp.empty()) {
                return false;
            } else {
                op_counter.clear();
                num_expansions = 0;
            }

        } else {
            return false;
        }

    } else if (state_decision.compare("diff") == 0) {
        if (abs((h_initial - cost) - h_value) > state_value) {
            return false;
        }
    } else {
        // always expand
    }

    //cout << "Potential Operators size: " << ops.size() << endl;

    // Check whether any of the potential operators are already blocked, remove blocked ones.
    if (state_decision.compare("counter") != 0) {
        map<int, set<OpID>>::iterator blocked_it = blocked_ops.find(task_proxy.get_ex_goals().size() + 1);
        if (blocked_it != blocked_ops.end()) {
            for (vector<OperatorProxy>::iterator op_it = ops_tmp.begin(); op_it != ops_tmp.end(); ) {
                bool erased = false;
                for (OpID id : blocked_it->second) {
                    if (op_it->get_id() == id) {
                        //cout << "Candidate not valid: Blocked Operator " << op_it->get_name() << endl;
                        op_it = ops_tmp.erase(op_it);
                        erased = true;
                        break;
                    }
                }
                if (erased) {
                    continue;
                } else {
                    ++op_it;
                }
            }
            
        }
    }

    // Collect goal vector of the goal that is going to be replaced.
    vector<FactProxy> old_goals;
    for (FactProxy const &fact : task_proxy.get_goals()) {
        old_goals.push_back(fact);
    }

    // Construct candidates by collecting the goal facts satisfied by each of the operators, also remove ops that delete a goal fact.
    vector<candidate> candidates;
    for (OperatorProxy &op : ops_tmp) {
        get_replacing_facts(candidates, old_goals, op);
    }

    // Sort operators according to the selected order.
    if (operator_order.compare("most_sat") == 0) {
        most_satisfied_first(candidates);
    } else if (operator_order.compare("lowest_layer") == 0) {
        lowest_proposition_layer_first(candidates);
    } else {
        // It's fifo, so we do nothing.
    }

    //cout << "Candidate size: " << candidates.size() << endl;
    
    // Try to expand operators according to sorting order.
    for (candidate &c : candidates) {
        //cout << "Trying to expand with " << c.first.get_name() << endl;
        if (expand_goal(c, old_goals)) {
            //cout << "Goal expanded!" << endl;
            statistics->inc_backwards_expanded();
            return true;
        }
    }

    // If the previous loop did not terminate this function by returning true,
    // it means that none of the given operators can legally expand the goal from here
    // so we undo the previous expansion and block the operator that lead us here.

    // Do not undo expansions with the counter option, instead reset the count.
    if (state_decision.compare("counter") == 0) {
        op_counter.clear();
        num_expansions = 0;
        return false;
    }

    //cout << "Start undoing last expansion." << endl;
   
    vector<ExGoal> ex_goals = task_proxy.get_ex_goals();
    int level = ex_goals.size();


    // Block operator and remove ex_goal
    OpID to_block = ex_goals.back().followed_by;
    if (level == 1 && to_block == -1) { // Stop trying to expand because we blocked everything.
        cout << "Stop goal expansion, no legal expansion found." << endl;
        no_expansion_possible = true;
    } else {
        add_to_blocked(to_block, false);
        task_proxy.pop_ex_goal();
        // Decrease level because last ex_goal has been popped
        --level;
    }
    
    statistics->inc_undone();

    // Remove blocks from deeper levels so they are considered again once we expand down a different path.
    for (std::map<int, set<OpID>>::iterator it = blocked_ops.begin(); it != blocked_ops.end();) {
        if (it->first > level + 1) {
            it = blocked_ops.erase(it);
        } else {
            ++it;
        }
    }

    // Erase current proposition_layer
    for (map<PropID, int>::iterator it = proposition_layers.begin(); it != proposition_layers.end(); ) {
        if (it->second >= level) {
            it = proposition_layers.erase(it);
        } else {
            ++it;
        }
    }
    
    // Restore previous goal
    vector<FactProxy> to_restore = task_proxy.get_last_ex_goal_facts();
    update_goal_propositions(to_restore, old_goals);
    task_proxy.update_goals(to_restore, -1);
    
    // Readd old proposition_layer
    for (const FactProxy &fact : to_restore) {
        PropID prop_id = get_prop_id(fact);
        map<PropID, int>::iterator it = proposition_layers.find(prop_id);
        if (it == proposition_layers.end()) {
            proposition_layers.insert({prop_id, level});
        }
    }
    //cout << "Finished restoring goal." << endl;

    return false;
}


bool RelaxationHeuristic::dead_ends_are_reliable() const {
    return !task_properties::has_axioms(task_proxy);
}

// Update both the is_goal member of propositions as well as the goal_propositions vector.
void RelaxationHeuristic::update_goal_propositions(const vector<FactProxy> &new_goal_facts, const vector<FactProxy> &old_goals) {
    for (const FactProxy &old_goal : old_goals) {
        PropID id = get_prop_id(old_goal);
        propositions[id].is_goal = false;
    }
    vector<PropID> new_goal_props;
    for (const FactProxy &fact : new_goal_facts) {
        PropID to_add = get_prop_id(fact);
        new_goal_props.push_back(to_add);
        propositions[to_add].is_goal = true;
    }
    goal_propositions = new_goal_props;
}

std::pair<int, int> RelaxationHeuristic::get_var_val(PropID id) const {
    size_t x;
    for (x = 0; x < proposition_offsets.size() - 1; ++x) {
        if (id < proposition_offsets[x + 1]) {
            break;
        }
    }
    return std::pair<int, int> (x, id - proposition_offsets[x]);
}

PropID RelaxationHeuristic::get_prop_id(int var, int value) const {
    return proposition_offsets[var] + value;
}

PropID RelaxationHeuristic::get_prop_id(const FactProxy &fact) const {
    return get_prop_id(fact.get_variable().get_id(), fact.get_value());
}

const Proposition *RelaxationHeuristic::get_proposition(
    int var, int value) const {
    return &propositions[get_prop_id(var, value)];
}

Proposition *RelaxationHeuristic::get_proposition(int var, int value) {
    return &propositions[get_prop_id(var, value)];
}

Proposition *RelaxationHeuristic::get_proposition(const FactProxy &fact) {
    return get_proposition(fact.get_variable().get_id(), fact.get_value());
}

void RelaxationHeuristic::build_unary_operators(const OperatorProxy &op) {
    int op_no = op.is_axiom() ? -1 : op.get_id();
    int base_cost = op.get_cost();
    vector<PropID> precondition_props;
    PreconditionsProxy preconditions = op.get_preconditions();
    precondition_props.reserve(preconditions.size());
    for (FactProxy precondition : preconditions) {
        precondition_props.push_back(get_prop_id(precondition));
    }
    for (EffectProxy effect : op.get_effects()) {
        PropID effect_prop = get_prop_id(effect.get_fact());
        EffectConditionsProxy eff_conds = effect.get_conditions();
        precondition_props.reserve(preconditions.size() + eff_conds.size());
        for (FactProxy eff_cond : eff_conds) {
            precondition_props.push_back(get_prop_id(eff_cond));
        }

        // The sort-unique can eventually go away. See issue497.
        vector<PropID> preconditions_copy(precondition_props);
        utils::sort_unique(preconditions_copy);
        array_pool::ArrayPoolIndex precond_index =
            preconditions_pool.append(preconditions_copy);
        unary_operators.emplace_back(
            preconditions_copy.size(), precond_index, effect_prop,
            op_no, base_cost);
        precondition_props.erase(precondition_props.end() - eff_conds.size(), precondition_props.end());
    }
}

void RelaxationHeuristic::simplify() {
    /*
      Remove dominated unary operators, including duplicates.

      Unary operators with more than MAX_PRECONDITIONS_TO_TEST
      preconditions are (mostly; see code comments below for details)
      ignored because we cannot handle them efficiently. This is
      obviously an inelegant solution.

      Apart from this restriction, operator o1 dominates operator o2 if:
      1. eff(o1) = eff(o2), and
      2. pre(o1) is a (not necessarily strict) subset of pre(o2), and
      3. cost(o1) <= cost(o2), and either
      4a. At least one of 2. and 3. is strict, or
      4b. id(o1) < id(o2).
      (Here, "id" is the position in the unary_operators vector.)

      This defines a strict partial order.
    */
#ifndef NDEBUG
    int num_ops = unary_operators.size();
    for (OpID op_id = 0; op_id < num_ops; ++op_id)
        assert(utils::is_sorted_unique(get_preconditions_vector(op_id)));
#endif

    const int MAX_PRECONDITIONS_TO_TEST = 5;

    cout << "Simplifying " << unary_operators.size() << " unary operators..." << flush;

    /*
      First, we create a map that maps the preconditions and effect
      ("key") of each operator to its cost and index ("value").
      If multiple operators have the same key, the one with lowest
      cost wins. If this still results in a tie, the one with lowest
      index wins. These rules can be tested with a lexicographical
      comparison of the value.

      Note that for operators sharing the same preconditions and
      effect, our dominance relationship above is actually a strict
      *total* order (order by cost, then by id).

      For each key present in the data, the map stores the dominating
      element in this total order.
    */
    using Key = pair<vector<PropID>, PropID>;
    using Value = pair<int, OpID>;
    using Map = utils::HashMap<Key, Value>;
    Map unary_operator_index;
    unary_operator_index.reserve(unary_operators.size());

    for (size_t op_no = 0; op_no < unary_operators.size(); ++op_no) {
        const UnaryOperator &op = unary_operators[op_no];
        /*
          Note: we consider operators with more than
          MAX_PRECONDITIONS_TO_TEST preconditions here because we can
          still filter out "exact matches" for these, i.e., the first
          test in `is_dominated`.
        */

        Key key(get_preconditions_vector(op_no), op.effect);
        Value value(op.base_cost, op_no);
        auto inserted = unary_operator_index.insert(
            make_pair(move(key), value));
        if (!inserted.second) {
            // We already had an element with this key; check its cost.
            Map::iterator iter = inserted.first;
            Value old_value = iter->second;
            if (value < old_value)
                iter->second = value;
        }
    }

    /*
      `dominating_key` is conceptually a local variable of `is_dominated`.
      We declare it outside to reduce vector allocation overhead.
    */
    Key dominating_key;

    /*
      is_dominated: test if a given operator is dominated by an
      operator in the map.
    */
    auto is_dominated = [&](const UnaryOperator &op) {
            /*
              Check all possible subsets X of pre(op) to see if there is a
              dominating operator with preconditions X represented in the
              map.
            */

            OpID op_id = get_op_id(op);
            int cost = op.base_cost;

            const vector<PropID> precondition = get_preconditions_vector(op_id);

            /*
              We handle the case X = pre(op) specially for efficiency and
              to ensure that an operator is not considered to be dominated
              by itself.

              From the discussion above that operators with the same
              precondition and effect are actually totally ordered, it is
              enough to test here whether looking up the key of op in the
              map results in an entry including op itself.
            */
            if (unary_operator_index[make_pair(precondition, op.effect)].second != op_id)
                return true;

            /*
              We now handle all cases where X is a strict subset of pre(op).
              Our map lookup ensures conditions 1. and 2., and because X is
              a strict subset, we also have 4a (which means we don't need 4b).
              So it only remains to check 3 for all hits.
            */
            if (op.num_preconditions > MAX_PRECONDITIONS_TO_TEST) {
                /*
                  The runtime of the following code grows exponentially
                  with the number of preconditions.
                */
                return false;
            }

            vector<PropID> &dominating_precondition = dominating_key.first;
            dominating_key.second = op.effect;

            // We subtract "- 1" to generate all *strict* subsets of precondition.
            int powerset_size = (1 << precondition.size()) - 1;
            for (int mask = 0; mask < powerset_size; ++mask) {
                dominating_precondition.clear();
                for (size_t i = 0; i < precondition.size(); ++i)
                    if (mask & (1 << i))
                        dominating_precondition.push_back(precondition[i]);
                Map::iterator found = unary_operator_index.find(dominating_key);
                if (found != unary_operator_index.end()) {
                    Value dominator_value = found->second;
                    int dominator_cost = dominator_value.first;
                    if (dominator_cost <= cost)
                        return true;
                }
            }
            return false;
        };

    unary_operators.erase(
        remove_if(
            unary_operators.begin(),
            unary_operators.end(),
            is_dominated),
        unary_operators.end());

    cout << " done! [" << unary_operators.size() << " unary operators]" << endl;
}
}
