#include "ff_heuristic.h"

#include "../global_state.h"
#include "../option_parser.h"
#include "../plugin.h"

#include "../task_utils/task_properties.h"

#include <cassert>
#include <algorithm>

using namespace std;

namespace ff_heuristic {
// construction and destruction
FFHeuristic::FFHeuristic(const Options &opts)
    : AdditiveHeuristic(opts),
      relaxed_plan(task_proxy.get_operators().size(), false) {
    cout << "Initializing FF heuristic..." << endl;
    RelaxationHeuristic::operator_order = opts.get<string>("operator_order");
    RelaxationHeuristic::state_decision = opts.get<string>("state_decision");
    RelaxationHeuristic::state_value = opts.get<int>("state_value");
    RelaxationHeuristic::percentage = opts.get<int>("percentage");

    cout << "SAGE Settings:" << endl;
    if (state_decision.compare("new_min") == 0) {
        cout << "Using NewMinimum" << endl;
    } else if (state_decision.compare("counter") == 0) {
        cout << "Using Counter, min_eval " << state_value << ", " << percentage << "%" << endl;;
    } else if (state_decision.compare("diff") == 0) {
        cout << "Using Accuracy, max_diff " << state_value << endl;
    } else {
        cout << "WARNING: Expanding on every call!" << endl;
    }

    if (operator_order.compare("most_sat") == 0) {
        cout << "MostSatisfied Operator first" << endl;
    } else if (operator_order.compare("lowest_layer") == 0) {
        cout << "LowestLayer Operator first" << endl;
    } else {
        cout << "WARNING: Using FIFO!" << endl;
    }

    vector<FactPair> original_goal;
    for (FactProxy fact : task_proxy.get_goals()) {
        original_goal.push_back(fact.get_pair());
    }
    ExGoal original_goal_ex(original_goal, -1);
    task_proxy.add_ex_goal(original_goal_ex);
}

// This is only used when recalculating h_initial.
void FFHeuristic::set_relaxed_plan(
    const State &state, PropID goal_id) {
    Proposition *goal = get_proposition(goal_id);
    if (!goal->marked) { // Only consider each subgoal once.
        goal->marked = true;
        OpID op_id = goal->reached_by;
        if (op_id != NO_OP) { // We have not yet chained back to a start node.
            UnaryOperator *unary_op = get_operator(op_id);
            for (PropID precond : get_preconditions(op_id)) {
                set_relaxed_plan(
                    state, precond);
            }
            int operator_no = unary_op->operator_no;
            if (operator_no != -1) {
                // This is not an axiom.
                relaxed_plan[operator_no] = true;
            }
        }
    }
}

void FFHeuristic::mark_preferred_operators_and_relaxed_plan(
    const State &state, PropID goal_id) {
    Proposition *goal = get_proposition(goal_id);
    if (!goal->marked) { // Only consider each subgoal once.
        goal->marked = true;
        OpID op_id = goal->reached_by;
        if (op_id != NO_OP) { // We have not yet chained back to a start node.
            UnaryOperator *unary_op = get_operator(op_id);
            bool is_preferred = true;
            for (PropID precond : get_preconditions(op_id)) {
                mark_preferred_operators_and_relaxed_plan(
                    state, precond);
                if (get_proposition(precond)->reached_by != NO_OP) {
                    is_preferred = false;
                }
            }
            int operator_no = unary_op->operator_no;
            if (operator_no != -1) {
                // This is not an axiom.
                relaxed_plan[operator_no] = true;
                if (is_preferred) {
                    OperatorProxy op = task_proxy.get_operators()[operator_no];
                    assert(task_properties::is_applicable(op, state));
                    set_preferred(op);
                }
            }
        }
    }
}

int FFHeuristic::compute_heuristic(const State &state, const bool for_h_initial) {
    int h_add = compute_add_and_ff(state);
    if (h_add == DEAD_END)
        return h_add;

    vector<OperatorProxy> ops;
    // Collecting the relaxed plan also sets the preferred operators.
    for (PropID goal_id : goal_propositions) {
        if (for_h_initial) {
            set_relaxed_plan(state, goal_id);
        } else {
            mark_preferred_operators_and_relaxed_plan(state, goal_id);
            
            OpID op = get_proposition(goal_id)->reached_by;
            if (op == -1) {
                // The proposition was already true.
            } else {
                UnaryOperator *unary = get_operator(op);
                OperatorProxy op_proxy = task_proxy.get_operators()[unary->operator_no];
                ops.push_back(op_proxy);
            }
        }
    }

    int h_ff = 0;
    for (size_t op_no = 0; op_no < relaxed_plan.size(); ++op_no) {
        if (relaxed_plan[op_no]) {
            relaxed_plan[op_no] = false; // Clean up for next computation.
            h_ff += task_proxy.get_operators()[op_no].get_cost();
        }
    }

    // 
    bool did_expand = false;
    if (RelaxationHeuristic::use_bgg) {
        did_expand = check_expansion(ops, h_ff);
    }

    // Initialize h_initial if not done already.
    if (RelaxationHeuristic::h_initial == -1) {
        if (state == task_proxy.get_initial_state()) {
            RelaxationHeuristic::h_initial = h_ff;
        }
    }

    // Recalculate h_initial.
    if (did_expand) {
        RelaxationHeuristic::use_bgg = false;
        RelaxationHeuristic::h_initial = compute_heuristic(task_proxy.get_initial_state(), true);
        RelaxationHeuristic::use_bgg = true;
        //cout << "Updated h_initial to " << RelaxationHeuristic::h_initial << endl;
    }
    
    return h_ff;
}


int FFHeuristic::compute_heuristic(const GlobalState &global_state) {
    State state = convert_global_state(global_state);
    return compute_heuristic(state, false);
}


int FFHeuristic::compute_heuristic(const GlobalState &global_state, const int current_cost, SearchStatistics* statistics) {
    if (RelaxationHeuristic::statistics == nullptr) {
        RelaxationHeuristic::statistics = statistics;
    }
    RelaxationHeuristic::cost = current_cost;
    RelaxationHeuristic::use_bgg = true;
    int result = compute_heuristic(global_state);
    return result;
}

static shared_ptr<Heuristic> _parse(OptionParser &parser) {
    parser.document_synopsis("FF heuristic", "");
    parser.document_language_support("action costs", "supported");
    parser.document_language_support("conditional effects", "supported");
    parser.document_language_support(
        "axioms",
        "supported (in the sense that the planner won't complain -- "
        "handling of axioms might be very stupid "
        "and even render the heuristic unsafe)");
    parser.document_property("admissible", "no");
    parser.document_property("consistent", "no");
    parser.document_property("safe", "yes for tasks without axioms");
    parser.document_property("preferred operators", "yes");

    parser.add_option<string>("operator_order", "Order in which the expansion operators are checked for legality.", "fifo");
    parser.add_option<int>("expand_goal_after", "Number of calls of the heuristic function until the goal is expanded.", "0");
    parser.add_option<string>("state_decision", "Criteria to decide what state to expand first.", "all");
    parser.add_option<int>("state_value", "Value for the state_decision.", "0");
    parser.add_option<int>("percentage", "Percentage above which an operator has to have appeared in the previous expansions.", "90");

    Heuristic::add_options_to_parser(parser);
    Options opts = parser.parse();
    if (parser.dry_run())
        return nullptr;
    else
        return make_shared<FFHeuristic>(opts);
}

static Plugin<Evaluator> _plugin("ff", _parse);
}
