#include "cycle_covering_heuristic.h"

#include "../option_parser.h"
#include "../plugin.h"

#include "../algorithms/johnson_cycle_detection.h"
#include "../landmarks/landmark_factory.h"

using namespace std;
using namespace landmarks;
using namespace operator_counting;

namespace cycle_covering_heuristic {
CycleCoveringHeuristic::CycleCoveringHeuristic(const Options &opts)
    : Heuristic(opts),
      cycle_constraints(opts.get<bool>("cycle_constraints")),
      ip(opts.get<bool>("ip")),
      recomp(opts.get<bool>("recomp")),
      lm(opts.get<shared_ptr<landmarks::LandmarkFactory>>("lm")),
      lp_solver(lp::LPSolverType(opts.get_enum("lpsolver"))) {

    if (!task_proxy.get_axioms().empty()) {
        cerr << "The cycle-covering heuristic does not support axioms." << endl;
        utils::exit_with(utils::ExitCode::SEARCH_UNSUPPORTED);
    }
    cout << "Preparing " << (ip ? "IP" : "LP") << "..." << endl;
    vector<lp::LPVariable> variables;
    double infinity = lp_solver.get_infinity();
    for (OperatorProxy op: task_proxy.get_operators()) {
        variables.emplace_back(0, infinity, op.get_cost(), ip);
    }
    vector<lp::LPConstraint> constraints;
    constraints_generator = make_shared<CycleConstraints>(CycleConstraints(opts));
    constraints_generator->initialize_constraints(task, constraints, infinity);
    lp_solver.load_problem(lp::LPObjectiveSense::MINIMIZE, variables, constraints);
}

CycleCoveringHeuristic::~CycleCoveringHeuristic() {
}

int CycleCoveringHeuristic::compute_heuristic(const GlobalState &global_state) {
    State state = convert_global_state(global_state);
    if ((recomp && constraints_generator->update_constraints(state, lp_solver))
    || (!recomp && constraints_generator->update_constraints(global_state, lp_solver))) {
        lp_solver.clear_temporary_constraints();
        return DEAD_END;
    }

    int result;
    lp_solver.solve();
    if (lp_solver.has_optimal_solution()) {
        result = ceil(lp_solver.get_objective_value() - 0.01);
        if (first) {
            size_t count = 0;
            for (auto s : lp_solver.extract_solution()) {
                if (ceil(s) - s != 0) ++count;
            }
            cout << "The LP solution has " << count << " non-integer variables." << endl;
            first = false;
        }
    } else {
        result = DEAD_END;
    }
    lp_solver.clear_temporary_constraints();
    return result;
}

static shared_ptr<Heuristic> _parse(OptionParser &parser) {
    parser.document_synopsis("Cycle-covering heuristic", "");
    parser.add_option<bool>("ip", "Constructs integer program instead of LP.", "false");
    parser.add_option<bool>("recomp", "Decides whether or not to recompute the LM graph for every state.", "false");

    // TODO: Hack, these are only here so they can be passed to the constructor of the cycle constraints.
    parser.add_option<shared_ptr<LandmarkFactory>>("lm", "");
    parser.add_option<bool>("cycle_constraints", "","true");
    parser.add_option<bool>("consider_orderings","","false");

    parser.document_language_support("action costs", "supported"); // TODO: check true?
    parser.document_language_support("conditional effects", "supported"); // TODO: check true?
    parser.document_language_support("axioms", "not supported"); // TODO: check true?
    parser.document_property("admissible", "yes"); // TODO: check true?
    parser.document_property("consistent", "?"); // TODO: check
    parser.document_property("safe", "?"); // TODO: check
    parser.document_property("preferred operators", "no"); // TODO: what does this mean?

    parser.document_synopsis("Cycle-covering heuristic", "");
    lp::add_lp_solver_option_to_parser(parser);
    Heuristic::add_options_to_parser(parser);
    Options opts = parser.parse();
    if (parser.dry_run()) return nullptr;
    return make_shared<CycleCoveringHeuristic>(opts);
}

static Plugin<Evaluator> _plugin("cycle", _parse);
}
