#include "implicit_fork_constraints.h"
//#include "constraint_generator.h"
#include "../plugins/plugin.h"
#include "../utils/logging.h"
#include <limits>

using namespace std;
using namespace lp;

namespace implicit {
    ImplicitForkConstraints::ImplicitForkConstraints(const LPSolver &lp_solver, const plugins::Options &opts)
        : log(utils::get_log_from_options(opts)),
        allow_negative_paths(opts.get<bool>("allow_negative_paths")),
        infinity(lp_solver.get_infinity()) {
    }

    ImplicitForkConstraints::ImplicitForkConstraints(const plugins::Options &opts)
            : log(utils::get_log_from_options(opts)),
              allow_negative_paths(opts.get<bool>("allow_negative_paths")),
            infinity(numeric_limits<double>::infinity()) {
    }

    void ImplicitForkConstraints::initialize_constraints(
            const shared_ptr<AbstractTask> &task, LinearProgram &lp) {
        infinity = lp.get_infinity();
        TaskProxy task_proxy(*task);
        const causal_graph::CausalGraph &cg = causal_graph::get_causal_graph(
                task.get());
        LPVariables &variables = lp.get_variables();
        //LPConstraints &constraints = lp.get_constraints();
        for (const VariableProxy &var: task_proxy.get_variables()) {
            forward_forks.emplace_back(cg, task_proxy, var.get_id());
        }
        int irrelevant_forks_removed = 0;
        for (ForwardFork &forward_fork: forward_forks) {
            fork_abstractions.emplace_back(forward_fork, task_proxy);
            if (fork_abstractions.back().abstract_goal.empty()) {
                fork_abstractions.pop_back();
                irrelevant_forks_removed++;
            }
        }
        log << "Removed " << irrelevant_forks_removed << " fork abstractions with no goal variables from "
        << forward_forks.size() << " total forks." << endl;
        //additivity variables already added (first variables added and therefore indexed 0-#operators-1)

        //Unary Constraints and H-constraints are temporary constraints as they are reliant on goal achieving root sequences
        //p and d constraints are also temporary constraints, as they are reliant on state values and root sequence
        //only add variables at initialization as all constraints are temporary
        for (ForkAbstraction const &fork_abstraction: fork_abstractions) {
            create_fixed_root_fork_variables(fork_abstraction, task_proxy, variables);//DONE
            create_root_seq_fork_variables(fork_abstraction, task_proxy, variables);//DONE
            create_h_variables(fork_abstraction, variables);//DONE
        }
        //potential to add inverted fork abstraction constraints here

        //dump_variables(variables);
    }

    void ImplicitForkConstraints::create_fixed_root_fork_variables(const ForkAbstraction &fork_abstraction,
                                                                     const TaskProxy &task_proxy,
                                                                     LPVariables &variables) {
        vector<int> fork_var_id(6, -1);
        fork_var_id[0] = fork_abstraction.root_id;
        for (const ForkSuccessor &succ: fork_abstraction.successors) {
            if (succ_is_goal_var(fork_abstraction, succ)) {
                int succ_domain_size = task_proxy.get_variables()[succ.var_id].get_domain_size();
                fork_var_id[1] = succ.var_id;
                //Firsts add all variables
                for (int dom_i = 0; dom_i < succ_domain_size; dom_i++) {
                    fork_var_id[2] = dom_i;
                    for (int dom_j = 0; dom_j < succ_domain_size; dom_j++) {
                        fork_var_id[3] = dom_j;
                        for (int root_value = 0; root_value < 2; root_value++) {
                            fork_var_id[4] = root_value;
                            if (dom_i == dom_j) {
                                //p(succ,dom,dom,0/1)
                                fork_var_id[5] = -1;
                                //vector<int> fork_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, dom_j, root_value};
                                fork_var_indices[fork_var_id] = variables.size();
                                variables.emplace_back(-infinity, infinity, 0, false);
                            } else {
                                for (const UnaryOperator &op: fork_abstraction.operators) {
                                    if (op.effect_id == succ.var_id
                                        && op.unary_effect.value == dom_j
                                        && (op.preconditions.empty()
                                            || op.preconditions.front().var_id != fork_abstraction.root_id
                                            || (op.preconditions.front().var_id == fork_abstraction.root_id
                                                && op.preconditions.front().value == root_value))) {
                                        //p(succ,dom_i,eff(op),root_value in Pre(op))
                                        fork_var_id[5] = op.original_op_id;
                                        //vector<int> fork_var_op_id = {fork_abstraction.root_id, succ.var_id, dom_i, dom_j, root_value, op.original_op_id};
                                        fork_var_indices[fork_var_id] = variables.size();
                                        variables.emplace_back(0, infinity, 0, false);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    void ImplicitForkConstraints::create_root_seq_fork_variables(const ForkAbstraction &fork_abstraction,
                                                        const TaskProxy &task_proxy, LPVariables &variables) {
        int largest_domain_size = 2;
        for (const ForkSuccessor &succ: fork_abstraction.successors) {
            int succ_domain_size = task_proxy.get_variables()[succ.var_id].get_domain_size();
            if (succ_is_goal_var(fork_abstraction, succ) && (succ_domain_size > largest_domain_size)) {
                largest_domain_size = succ_domain_size;
            }
        }
        largest_domain_sizes[fork_abstraction.root_id] = largest_domain_size;
        vector<int> root_seq_var_id(5,-1);
        root_seq_var_id[0] = fork_abstraction.root_id;
        for (const ForkSuccessor &succ: fork_abstraction.successors) {
            if (succ_is_goal_var(fork_abstraction, succ)) {
                int succ_domain_size = task_proxy.get_variables()[succ.var_id].get_domain_size();
                root_seq_var_id[1] = succ.var_id;
                for (int dom_i = 0; dom_i < succ_domain_size; dom_i++) {
                    root_seq_var_id[2] = dom_i;
                    //d(succ,dom_i,1)
                    root_seq_var_id[3] = 1;
                    root_seq_var_id[4] = -1;
                    //vector<int> root_seq_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, 1};
                    root_seq_var_indices[root_seq_var_id] = variables.size();
                    variables.emplace_back(0, infinity, 0, false);
                }
                for (int root_seq = 2; root_seq <= largest_domain_size + 1; root_seq++) {
                    root_seq_var_id[3] = root_seq;
                    for (int dom_i = 0; dom_i < succ_domain_size; dom_i++) {
                        root_seq_var_id[2] = dom_i;
                        for (int dom_j = 0; dom_j < succ_domain_size; dom_j++) {
                            //d(succ,dom_j -> dom_i,root_seq)
                            root_seq_var_id[4] = dom_j;
                            //vector<int> root_seq_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, root_seq, dom_j};
                            root_seq_var_indices[root_seq_var_id] = variables.size();
                            variables.emplace_back(0, infinity, 0, false);
                        }
                    }
                }
            }
        }
    }

    void ImplicitForkConstraints::create_h_variables(const ForkAbstraction &fork_abstraction, LPVariables &variables) {
        int largest_domain_size = largest_domain_sizes[fork_abstraction.root_id];
        vector<UnaryOperator> changing_root_to_zero;
        vector<UnaryOperator> changing_root_to_one;
        for (const UnaryOperator &op: fork_abstraction.operators) {
            if (op.effect_id == fork_abstraction.root_id) {
                if (op.unary_effect.value == 0) {
                    changing_root_to_zero.emplace_back(op);
                } else {
                    assert(op.unary_effect.value == 1);
                    changing_root_to_one.emplace_back(op);
                }
            }
        }
        vector<int> h_var_id(5,-1);
        h_var_id[0] = fork_abstraction.root_id;
        for (int root_seq = 1; root_seq <= largest_domain_size + 1; root_seq++) {
            h_var_id[1] = root_seq;
            //Find a better way to cover all those cases
            if (changing_root_to_zero.empty() && changing_root_to_one.empty()) {
                //ga_rs(1)
                //vector<int> h_var_id = {fork_abstraction.root_id, root_seq};
                h_var_indices[h_var_id] = variables.size();
                variables.emplace_back(0, infinity, 0, false);
                break;
            } else {
                if (changing_root_to_zero.empty()) {
                    assert(!changing_root_to_one.empty());
                    h_var_id[2] = 1;
                    for (const UnaryOperator &op1 : changing_root_to_one) {
                        //ga_rs(root_seq,op1)
                        h_var_id[3] = op1.original_op_id;
                        //vector<int> h_var_id = {fork_abstraction.root_id, root_seq, 1, op1.original_op_id};
                        h_var_indices[h_var_id] = variables.size();
                        variables.emplace_back(0, infinity, 0, false);
                    }
                    if(root_seq >= 2) {break;}
                }
                if (changing_root_to_one.empty()) {
                    assert(!changing_root_to_zero.empty());
                    h_var_id[2] = 0;
                    for (const UnaryOperator &op0 : changing_root_to_zero) {
                        //ga_rs(root_seq,op0)
                        h_var_id[3] = op0.original_op_id;
                        //vector<int> h_var_id = {fork_abstraction.root_id, root_seq, 0, op0.original_op_id};
                        h_var_indices[h_var_id] = variables.size();
                        variables.emplace_back(0, infinity, 0, false);
                    }
                    if(root_seq >= 2) {break;}
                }
                if (!changing_root_to_zero.empty() && !changing_root_to_one.empty()) {
                    h_var_id[2] = 2;
                    for (const UnaryOperator &op0 : changing_root_to_zero) {
                        h_var_id[3] = op0.original_op_id;
                        for (const UnaryOperator &op1 : changing_root_to_one) {
                            //ga_rs(root_seq,op0,op1)
                            h_var_id[4] = op1.original_op_id;
                            //vector<int> h_var_id = {fork_abstraction.root_id, root_seq, 2, op0.original_op_id, op1.original_op_id};
                            h_var_indices[h_var_id] = variables.size();
                            variables.emplace_back(0, infinity, 0, false);
                        }
                    }
                }
            }
        }
    }



    bool ImplicitForkConstraints::update_constraints(
            const State &state, LPSolver &lp_solver) {
        infinity = lp_solver.get_infinity();
        vector<lp::LPConstraint> temp_constraints;
        //unary_op_id = {fork_abstraction.root_id, op.original_op_id, op.effect_id}
        //fork_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, dom_j, root_value, (op.original_id)}
        //root_seq_var = {fork_abstraction.root_id, succ.var_id, dom_i, root_seq, (dom_j)}
        //h_var = {fork_abstraction.root_id}
        unordered_map<vector<int>, int, VectorHash> temp_constraint_indices;
        //Current idea add constraints in order w,p,d,h and add variables to constraints when adding their related constraints
        //modify w constraints later with p and h variables
        create_operator_cost_constraints(temp_constraints, temp_constraint_indices);
        for (const ForkAbstraction &fork_abstraction : fork_abstractions) {
            //modify p constraints later with p and d variables
            create_fixed_root_fork_constraints(state, fork_abstraction, temp_constraints,temp_constraint_indices);
            //modify d constraints later with d and h variables
            //just one h constraint; h vars are added to w and d
            create_temporary_fork_constraints(state, fork_abstraction, temp_constraints, temp_constraint_indices);
        }
        //dump_temp_constraints(temp_constraints);
        lp_solver.add_temporary_constraints(temp_constraints);
        return false;
    }

    void ImplicitForkConstraints::create_operator_cost_constraints(vector<lp::LPConstraint> &temp_constraints,
                                unordered_map<vector<int>, int, VectorHash> &temp_constraint_indices) {
        //reliant on p variables and h variables
        vector<int> unary_op_id(3,-1);
        for (const ForkAbstraction &fork_abstraction: fork_abstractions) {
            unary_op_id[0] = fork_abstraction.root_id;
            for (const UnaryOperator &op: fork_abstraction.operators) {
                unary_op_id[1] = op.original_op_id;
                unary_op_id[2] = op.effect_id;
                //vector<int> unary_op_id = {fork_abstraction.root_id, op.original_op_id, op.effect_id};
                temp_constraint_indices[unary_op_id] = (int) temp_constraints.size();
                temp_constraints.emplace_back(0, infinity);
                //occurrence in itself
                temp_constraints.back().insert(op.original_op_id, 1);
            }
        }
    }


    void ImplicitForkConstraints::create_fixed_root_fork_constraints(const State &state,
                                 const ForkAbstraction &fork_abstraction, vector<lp::LPConstraint> &temp_constraints,
                                 unordered_map<vector<int>, int, VectorHash> &temp_constraint_indices) {
        //reliant on p variables and d variables (occurrences in w)
        vector<int> fork_var_id(6, -1);
        vector<int> reaching_pre_var_id(6,-1);
        vector<int> unary_op_id(3, -1);
        fork_var_id[0] = fork_abstraction.root_id;
        reaching_pre_var_id[0] = fork_abstraction.root_id;
        unary_op_id[0] = fork_abstraction.root_id;
        for (const ForkSuccessor &succ: fork_abstraction.successors) {
            if (succ_is_goal_var(fork_abstraction, succ)) {
                int succ_domain_size = state.get_task().get_variables()[succ.var_id].get_domain_size();
                fork_var_id[1] = succ.var_id;
                reaching_pre_var_id[1] = succ.var_id;
                //Firsts add all constraints
                for (int dom_i = 0; dom_i < succ_domain_size; dom_i++) {
                    fork_var_id[2] = dom_i;
                    for (int dom_j = 0; dom_j < succ_domain_size; dom_j++) {
                        fork_var_id[3] = dom_j;
                        for (int root_value = 0; root_value < 2; root_value++) {
                            fork_var_id[4] = root_value;
                            fork_var_id[5] = -1;
                            //vector<int> fork_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, dom_j, root_value};
                            temp_constraint_indices[fork_var_id] = (int)temp_constraints.size();
                            if (allow_negative_paths) {
                                temp_constraints.emplace_back(0,0);
                            } else {
                                temp_constraints.emplace_back(0, infinity);
                            }
                        }
                    }
                }
                //Then add all variables to constraints
                for (int dom_i = 0; dom_i < succ_domain_size; dom_i++) {
                    fork_var_id[2] = dom_i;
                    reaching_pre_var_id[2] = dom_i;
                    for (int dom_j = 0; dom_j < succ_domain_size; dom_j++) {
                        fork_var_id[3] = dom_j;
                        for (int root_value = 0; root_value < 2; root_value++) {//000,001,010,011,100
                            fork_var_id[4] = root_value;
                            reaching_pre_var_id[4] = root_value;
                            //we need to differentiate between dom_i == dom_j vars and op induced vars
                            //so we add the correct vars to constraints instead of always the same
                            fork_var_id[5] = -1;
                            //vector<int> fork_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, dom_j, root_value};
                            int fork_var_index = temp_constraint_indices[fork_var_id];
                            if (dom_i == dom_j) {
                                //p(succ,dom,dom,0/1)
                                //occurrence in itself and in d
                                temp_constraints[fork_var_index].insert(fork_var_indices[fork_var_id], 1);
                            } else {
                                //Add all p(succ,dom_i,dom_j,0/1) constraints not trivial/inf, which are:
                                //p(succ,dom,eff(op)[v],pre(op)[r]_1)
                                //p(succ,dom,eff(op)[v],pre(op)[r]_2)
                                for (const UnaryOperator &op: fork_abstraction.operators) {
                                    //fixed issue by adding specifier to vars from which operator they came from
                                    if (op.effect_id == succ.var_id
                                        && op.unary_effect.value == dom_j
                                        && (op.preconditions.empty()
                                            || op.preconditions.front().var_id != fork_abstraction.root_id
                                            || (op.preconditions.front().var_id == fork_abstraction.root_id
                                                && op.preconditions.front().value == root_value))) {
                                        fork_var_id[5] = op.original_op_id;
                                        //vector<int> fork_var_op_id = {fork_abstraction.root_id, succ.var_id, dom_i, dom_j, root_value, op.original_op_id};
                                        unary_op_id[1] = op.original_op_id;
                                        unary_op_id[2] = op.effect_id;
                                        //vector<int> unary_op_id = {fork_abstraction.root_id, op.original_op_id, op.effect_id};
                                        int unary_op_index = temp_constraint_indices[unary_op_id];
                                        //occurrence in w
                                        temp_constraints[unary_op_index].insert(fork_var_indices[fork_var_id], -1);
                                        //occurrence in itself
                                        temp_constraints[fork_var_index].insert(fork_var_indices[fork_var_id], 1);
                                        if ((!op.preconditions.empty()) &&
                                            op.preconditions.back().var_id == succ.var_id) {
                                            reaching_pre_var_id[3] = op.preconditions.back().value;
                                            //p(succ,dom_i,eff(op)[v],pre(op)[r]) <= p(succ,dom_i,pre(op)[v]pre(op)[r]) + wa
                                            //vector<int> reaching_pre_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, op.preconditions.back().value, root_value};
                                            int pre_var_index = temp_constraint_indices[reaching_pre_var_id];
                                            //occurrence in (previous/precondition) p
                                            temp_constraints[pre_var_index].insert(fork_var_indices[fork_var_id], -1);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }


    void ImplicitForkConstraints::create_temporary_fork_constraints(const State &state,
                                const ForkAbstraction &fork_abstraction, vector<lp::LPConstraint> &temp_constraints,
                                unordered_map<vector<int>, int, VectorHash> &temp_constraint_indices) {
        //reliant on d variables and h variables (occurrences in p)
        int largest_domain_size = largest_domain_sizes[fork_abstraction.root_id];
        int initial_root_value = fork_abstraction.root_dom_mapping[state[fork_abstraction.root_id].get_value()];//0 or 1
        vector<int> root_sequence;
        root_sequence.reserve(largest_domain_size + 1);
        int root_value = initial_root_value;
        for (int i = 0; i < largest_domain_size + 1; i++) {
            root_sequence.emplace_back(root_value);
            root_value = 1 - root_value;
        }
        //Creating d constraints
        vector<int> root_seq_var_id(5, -1);
        vector<int> pre_seq_var_id(5,-1);
        vector<int> fork_var_id(6, -1);
        root_seq_var_id[0] = fork_abstraction.root_id;
        pre_seq_var_id[0] = fork_abstraction.root_id;
        fork_var_id[0] = fork_abstraction.root_id;
        for (const ForkSuccessor &succ: fork_abstraction.successors) {
            if (succ_is_goal_var(fork_abstraction, succ)) {
                int succ_domain_size =  state.get_task().get_variables()[succ.var_id].get_domain_size();
                root_seq_var_id[1] = succ.var_id;
                pre_seq_var_id[1] = succ.var_id;
                fork_var_id[1] = succ.var_id;
                fork_var_id[2] = state[succ.var_id].get_value();
                for (int dom_i = 0; dom_i < succ_domain_size; dom_i++) {
                    root_seq_var_id[2] = dom_i;
                    //d(succ,dom_i,1) <= p(succ,s[succ],dom_i,root_seq[1])
                    root_seq_var_id[3] = 1;
                    root_seq_var_id[4] = -1;
                    //vector<int> root_seq_var = {fork_abstraction.root_id, succ.var_id, dom_i, 1};
                    temp_constraint_indices[root_seq_var_id] = (int)temp_constraints.size();
                    if (allow_negative_paths) {
                        temp_constraints.emplace_back(0,0);
                    } else {
                        temp_constraints.emplace_back(0, infinity);
                    }
                    fork_var_id[3] = dom_i;
                    fork_var_id[4] = initial_root_value;
                    //vector<int> fork_var_id = {fork_abstraction.root_id, succ.var_id, state[succ.var_id].get_value(), dom_i, initial_root_value};
                    int fork_var_index = temp_constraint_indices[fork_var_id];
                    //occurrence in itself
                    temp_constraints.back().insert(root_seq_var_indices[root_seq_var_id], 1);
                    //occurrence in p
                    temp_constraints[fork_var_index].insert(root_seq_var_indices[root_seq_var_id], -1);
                }
                //TODO Difficult to find and take out the root sequences that are unachievable and won't be used in h
                for (int root_seq = 2; root_seq <= largest_domain_size + 1; root_seq++) {
                    root_seq_var_id[3] = root_seq;
                    pre_seq_var_id[3] = root_seq - 1;
                    fork_var_id[4] = root_sequence[root_seq - 1];
                    for (int dom_i = 0; dom_i < succ_domain_size; dom_i++) {
                        root_seq_var_id[2] = dom_i;
                        root_seq_var_id[4] = -1;
                        fork_var_id[3] = dom_i;
                        //vector<int> root_seq_var = {fork_abstraction.root_id, succ.var_id, dom_i, root_seq};
                        temp_constraint_indices[root_seq_var_id] = (int)temp_constraints.size();
                        if (allow_negative_paths) {
                            temp_constraints.emplace_back(0,0);
                        } else {
                            temp_constraints.emplace_back(0, infinity);
                        }
                        for (int dom_j = 0; dom_j < succ_domain_size; dom_j++) {
                            root_seq_var_id[4] = dom_j;
                            pre_seq_var_id[2] = dom_j;
                            fork_var_id[2] = dom_j;
                            //d(succ,dom_i,i) <= d(succ,dom_j,i-1) + p(succ,dom_j,dom_i,root_seq[i])
                            //vector<int> root_seq_var_id = {fork_abstraction.root_id, succ.var_id, dom_i, root_seq, dom_j};
                            //vector<int> fork_var_id = {fork_abstraction.root_id, succ.var_id, dom_j, dom_i, root_sequence[root_seq - 1]};
                            int fork_var_index = temp_constraint_indices[fork_var_id];
                            //vector<int> pre_seq_var_id = {fork_abstraction.root_id, succ.var_id, dom_j, root_seq - 1};
                            int pre_seq_var_index = temp_constraint_indices[pre_seq_var_id];
                            //occurrence in itself
                            temp_constraints.back().insert(root_seq_var_indices[root_seq_var_id], 1);
                            //occurrence in (previous) d
                            temp_constraints[pre_seq_var_index].insert(root_seq_var_indices[root_seq_var_id], -1);
                            //occurrence in p
                            temp_constraints[fork_var_index].insert(root_seq_var_indices[root_seq_var_id], -1);
                        }
                    }
                }
            }
        }
        create_h_constraints(root_sequence, fork_abstraction, temp_constraints, temp_constraint_indices);
    }

    void ImplicitForkConstraints::create_h_constraints(vector<int> &root_sequence,
                                   const ForkAbstraction &fork_abstraction, vector<lp::LPConstraint> &temp_constraints,
                                   unordered_map<vector<int>, int, VectorHash> &temp_constraint_indices) {
        //singular constraint reliant on h variables (occurrences in w and d)
        bool root_is_specified_in_goal = false;
        if ((!fork_abstraction.abstract_goal.empty())
            && fork_abstraction.abstract_goal.front().var_id == fork_abstraction.root_id) {
            root_is_specified_in_goal = true;
        }
        vector<UnaryOperator> changing_root_to_zero;
        vector<UnaryOperator> changing_root_to_one;
        for (const UnaryOperator &op: fork_abstraction.operators) {
            if (op.effect_id == fork_abstraction.root_id) {
                if (op.unary_effect.value == 0) {
                    //cout << "operator changing root to 0 found" << endl;
                    changing_root_to_zero.emplace_back(op);
                } else {
                    assert(op.unary_effect.value == 1);
                    //cout << "operator changing root to 1 found" << endl;
                    changing_root_to_one.emplace_back(op);
                }
            }
        }
        int initial_root_value = root_sequence[0];
        //vector<int> partial_root_sequence;
        //partial_root_sequence.reserve(root_sequence.size());
        //add singular h constraint
        temp_constraint_indices[{fork_abstraction.root_id}] = (int)temp_constraints.size();
        if (allow_negative_paths) {
            temp_constraints.emplace_back(1,1);
        } else {
            temp_constraints.emplace_back(1, infinity);
        }
        //add reliance on h_vars and occurrences in w and d
        //Find a better way to cover all those cases
        vector<int> h_var_id(5,-1);
        vector<int> unary_op_id(3,-1);
        h_var_id[0] = fork_abstraction.root_id;
        unary_op_id[0] = fork_abstraction.root_id;
        int root_seq_size = 0;
        for (int r_value: root_sequence) {
            //partial_root_sequence.emplace_back(r_value);
            //int root_seq_size = (int)partial_root_sequence.size();
            root_seq_size++;
            if (root_is_specified_in_goal && fork_abstraction.abstract_goal.front().value != r_value) {
                continue;
            } else {
                assert((root_is_specified_in_goal && fork_abstraction.abstract_goal.front().value == r_value) ||
                       !root_is_specified_in_goal);
                h_var_id[1] = root_seq_size;
                if (changing_root_to_zero.empty()) {
                    if (changing_root_to_one.empty()) {
                        if (root_seq_size > 1) {
                            //Achieving this root_sequence will always be impossible if we cannot change the root
                            break;
                        }
                        //vector<int> h_var_id = {fork_abstraction.root_id, root_seq_size};
                        int h_var_index = h_var_indices[h_var_id];
                        //reliance on h_vars
                        temp_constraints.back().insert(h_var_index, 1);
                        //no occurrence in w
                        //add goal variables i.e. occurrences in d
                        add_goal_var_constraints(root_seq_size, h_var_index, fork_abstraction, temp_constraints, temp_constraint_indices);
                        //no more root sequences possible as we cannot change the root
                        break;
                    } else {
                        assert(changing_root_to_zero.empty());
                        assert(!changing_root_to_one.empty());
                        if (root_seq_size > 2) {
                            //Achieving this root_sequence will always be impossible if we cannot set root to 0
                            break;
                        }
                        h_var_id[2] = 1;
                        for (const UnaryOperator &op1: changing_root_to_one) {
                            h_var_id[3] = op1.original_op_id;
                            //vector<int> h_var_id = {fork_abstraction.root_id, root_seq_size, 1, op1.original_op_id};
                            int h_var_index = h_var_indices[h_var_id];
                            temp_constraints.back().insert(h_var_index, 1);
                            //occurrences in w
                            unary_op_id[1] = op1.original_op_id;
                            unary_op_id[2] = op1.effect_id;
                            //vector<int> unary_op_id = {fork_abstraction.root_id, op1.original_op_id, op1.effect_id};
                            int unary_op_index = temp_constraint_indices[unary_op_id];
                            if (initial_root_value == 0) { //-> 1 once more often than 0 effect (op1)
                                temp_constraints[unary_op_index].insert(h_var_index, -ceil(static_cast<float>(root_seq_size-1)/2));
                            } else {
                                assert(initial_root_value == 1);
                                temp_constraints[unary_op_index].insert(h_var_index, -floor(static_cast<float>(root_seq_size-1)/2));
                            }
                            //add goal variables i.e. occurrences in d
                            add_goal_var_constraints(root_seq_size, h_var_index, fork_abstraction, temp_constraints, temp_constraint_indices);
                        }
                    }
                    if (r_value == 1) {
                        //no more root sequences possible as we cannot set root to 0
                        break;
                    }
                } else {
                    assert(!changing_root_to_zero.empty());
                    if (changing_root_to_one.empty()) {
                        if (root_seq_size > 2) {
                            //Achieving this root_sequence will always be impossible if we cannot set root to 1
                            break;
                        }
                        h_var_id[2] = 0;
                        for (const UnaryOperator &op0: changing_root_to_zero) {
                            h_var_id[3] = op0.original_op_id;
                            //vector<int> h_var_id = {fork_abstraction.root_id, root_seq_size, 0, op0.original_op_id};
                            int h_var_index = h_var_indices[h_var_id];
                            //reliance on h_vars
                            temp_constraints.back().insert(h_var_index, 1);
                            //occurrences in w
                            unary_op_id[1] = op0.original_op_id;
                            unary_op_id[2] = op0.effect_id;
                            //vector<int> unary_op_id = {fork_abstraction.root_id, op0.original_op_id, op0.effect_id};
                            int unary_op_index = temp_constraint_indices[unary_op_id];
                            if (initial_root_value == 0) { //-> 1 once more often than 0 effect (op0)
                                temp_constraints[unary_op_index].insert(h_var_index, -floor(static_cast<float>(root_seq_size-1)/2));
                            } else {
                                assert(initial_root_value == 1);
                                temp_constraints[unary_op_index].insert(h_var_index, -ceil(static_cast<float>(root_seq_size-1)/2));
                            }
                            //add goal variables i.e. occurrences in d
                            add_goal_var_constraints(root_seq_size, h_var_index, fork_abstraction, temp_constraints, temp_constraint_indices);
                        }
                        if (r_value == 0) {
                            //no more root sequences possible as we cannot set root to 1
                            break;
                        }
                    } else {
                        assert(!changing_root_to_zero.empty());
                        assert(!changing_root_to_one.empty());
                        h_var_id[2] = 2;
                        for (const UnaryOperator &op0: changing_root_to_zero) {
                            h_var_id[3] = op0.original_op_id;
                            unary_op_id[1] = op0.original_op_id;
                            unary_op_id[2] = op0.effect_id;
                            int op0_index = temp_constraint_indices[unary_op_id];
                            for (const UnaryOperator &op1: changing_root_to_one) {
                                h_var_id[4] = op1.original_op_id;
                                //vector<int> h_var_id = {fork_abstraction.root_id, root_seq_size, 2, op0.original_op_id, op1.original_op_id};
                                int h_var_index = h_var_indices[h_var_id];
                                //reliance on h_vars
                                temp_constraints.back().insert(h_var_index, 1);
                                //occurrences in w
                                unary_op_id[1] = op1.original_op_id;
                                unary_op_id[2] = op1.effect_id;
                                int op1_index = temp_constraint_indices[unary_op_id];
                                //vector<int> op0_id = {fork_abstraction.root_id, op0.original_op_id, op0.effect_id};
                                //vector<int> op1_id = {fork_abstraction.root_id, op1.original_op_id, op1.effect_id};
                                if (initial_root_value == 0) { //-> 1 once more often than 0 effect
                                    temp_constraints[op0_index].insert(h_var_index, -floor(static_cast<float>(root_seq_size-1)/2));
                                    temp_constraints[op1_index].insert(h_var_index, -ceil(static_cast<float>(root_seq_size-1)/2));
                                } else {
                                    assert(initial_root_value == 1);
                                    temp_constraints[op0_index].insert(h_var_index, -ceil(static_cast<float>(root_seq_size-1)/2));
                                    temp_constraints[op1_index].insert(h_var_index, -floor(static_cast<float>(root_seq_size-1)/2));
                                }
                                //add goal variables i.e. occurrences in d
                                add_goal_var_constraints(root_seq_size, h_var_index, fork_abstraction, temp_constraints, temp_constraint_indices);
                            }
                        }
                    }
                }
            }
        }
    }

    void ImplicitForkConstraints::add_goal_var_constraints(int root_seq_size, int h_var_index,const ForkAbstraction &fork_abstraction,
                                                           vector<lp::LPConstraint> &temp_constraints,
                                                           unordered_map<vector<int>, int, VectorHash> &temp_constraint_indices) {
        vector<int> root_seq_goal_id(5,-1);
        root_seq_goal_id[0] = fork_abstraction.root_id;
        root_seq_goal_id[3] = root_seq_size;
        for (const ForkSuccessor &succ: fork_abstraction.successors) {
            for (const AbstractFact &goal_var: fork_abstraction.abstract_goal) {
                if (goal_var.var_id == succ.var_id) {
                    root_seq_goal_id[1] = succ.var_id;
                    root_seq_goal_id[2] = goal_var.value;
                    //vector<int> root_seq_goal_id = {fork_abstraction.root_id, succ.var_id, goal_var.value, root_seq_size};
                    int root_seq_goal_index = temp_constraint_indices[root_seq_goal_id];
                    temp_constraints[root_seq_goal_index].insert(h_var_index, -1);
                    break;
                }
            }
        }
    }

    bool ImplicitForkConstraints::succ_is_goal_var(const ForkAbstraction &fork_abstraction, const ForkSuccessor &succ) {
        for (const AbstractFact &goal_var: fork_abstraction.abstract_goal) {
            if (goal_var.var_id == succ.var_id) {
                return true;
            }
        }
        return false;
    }

    void ImplicitForkConstraints::dump_variables(LPVariables &variables) {
        cout << endl;
        cout << variables.size() << " LPVariables:" << endl;
        for (LPVariable lpvar: variables) {
            cout << lpvar.lower_bound << "<=" << lpvar.upper_bound << ", obj_coeff:" << lpvar.objective_coefficient
                 << endl;
        }
        cout << endl;
    }

    void ImplicitForkConstraints::dump_temp_constraints(vector<lp::LPConstraint> &temp_constraints) {
        cout << temp_constraints.size() << " Temporary Constraints:" << endl;
        for (LPConstraint &lp_constraint : temp_constraints) {
            lp_constraint.dump(cout);
            cout << endl;
            /*cout << "Constraint: " << lp_constraint.get_lower_bound() << " <= " << lp_constraint.get_upper_bound();
            cout << ", Variables: ";
            for (int var: lp_constraint.get_variables()) {
                cout << var << ", ";
            }
            cout << "Coefficients: ";
            for (double coeff: lp_constraint.get_coefficients()) {
                cout << coeff << ", ";
            }
            cout << endl;*/
        }
    }
}

namespace operator_counting {
class ImplicitForkConstraintsFeature : public plugins::TypedFeature<ConstraintGenerator, implicit::ImplicitForkConstraints> {
public:
    ImplicitForkConstraintsFeature() : TypedFeature("implicit_constraints") {
        document_title("Implicit fork constraints");
        document_synopsis(
                "Computes implicit fork constraints");
        utils::add_log_options_to_feature(*this);
        add_option<bool>(
                "allow_negative_paths",
                "allows negative paths. Meaning the primal view variables can be negative,which leads to more"
                "general constraints. We note that unary_operator costs in the primal view always have to be restricted"
                "to non-negative values to achieve bounded objective values.",
                "false");
    }

    virtual shared_ptr<ImplicitForkConstraints> create_component(const plugins::Options &opts, const utils::Context &) const override {
        return make_shared<ImplicitForkConstraints>(opts);
    }
};

    static plugins::FeaturePlugin<ImplicitForkConstraintsFeature> _plugin;
}

