#pragma once

#include "landmark_factory.h"
#include "../utils/logging.h"

namespace landmarks
{
    // class LandmarkMapper is used to map values, as they are not unique. It's an easy
    // pre processing structure which can be used for other domains as well, but needs to be changed accordingly
    class LandmarkMapper98
    {
    public:
        enum class ObjectType98
        {
            Airplane,
            Airport,
            City,
            Location,
            Package,
            Truck
        };

        struct ObjectLocation98
        {
            int index;

            struct ObjHash98
            {
                size_t operator()(const ObjectLocation98 &loc) const
                {
                    return std::hash<int>()(loc.index);
                }
            };

            bool operator==(const ObjectLocation98 &other) const
            {
                return index == other.index;
            }

            bool operator!=(const ObjectLocation98 &other) const
            {
                return !(index == other.index);
            }
        };

        // Important:
        struct MapperPair98
        {
            ObjectLocation98 key;
            ObjectLocation98 val;
        };

        struct CityInfo98
        {
            ObjectLocation98 city;
            std::vector<ObjectLocation98> position;
            ObjectLocation98 airport;
            std::vector<ObjectLocation98> trucks;
        };

    private:
        // Linke Seite: Unique [ID] --> aka variable id
        // Rechte Seite: Unique pro Linke Seite --> aka value

        struct ObjectDescriptor
        {
            std::vector<ObjectType98> types;
            std::string name;

            bool has_type(ObjectType98 type) const { return std::find(types.begin(), types.end(), type) != types.end(); }
            bool is_vehicle() const { return has_type(ObjectType98::Airplane) || has_type(ObjectType98::Truck); }
        };

        // All objects
        std::vector<ObjectDescriptor> allObjects;

        std::vector<ObjectDescriptor> vehicles;

        std::vector<ObjectLocation98> positions;

        // Mapper FactPair --> Object Index
        //      Key: Fact-Var-ID; Value Vector[Fact-Value-ID (per Var-ID)] = obj_location
        std::unordered_map<int, std::vector<int>> value_locations;

        std::unordered_map<int, std::vector<int>> value_locations_truck;
        std::unordered_map<int, std::vector<int>> value_locations_air;

        // Mapper Key: Fact-Var-ID: Obj-Location
        std::unordered_map<int, int> variable_id_locations;
        std::unordered_map<int, int> locations_to_variable_id;

        std::vector<CityInfo98> city_infos;
        // std::unordered_map<ObjectLocation, ObjectLocation, ObjectLocation::ObjHash> is_in_city;

        // Helper Functions
        int find_object(const std::string &name);
        const char *get_type_name(ObjectType98 type);

    public:
        void init(const char *path, const TaskProxy &taskProxy);
        void parse_inCity(const char *path);

        MapperPair98 from_fact_pair(const FactPair &fact);
        FactPair to_fact_pair(const MapperPair98 &mapperPair);
        int to_state_id(ObjectLocation98 location);

        const std::string &get_name(ObjectLocation98 location);
        std::string get_name(MapperPair98 mapperPair);

        bool has_type(ObjectLocation98 location, ObjectType98 type);
        bool is_vehicle(ObjectLocation98 location);

        std::vector<ObjectDescriptor> get_vehicles()
        {
            return vehicles;
        };

        // ObjectLocation get_in_city(ObjectLocation location);
        CityInfo98 *get_city_info_ptr(ObjectLocation98 location);
        CityInfo98 get_city_info(ObjectLocation98 location);

        bool has_value_with_type(ObjectLocation98 key, ObjectType98 type);

        FactPair get_vehicle_fact_pair_from_loc(const MapperPair98 &mapperPair, const ObjectType98 objType);

        template <typename FuncType>
        void get_by(ObjectType98 type, FuncType func)
        {
            for (size_t i = 0; i < allObjects.size(); ++i)
            {
                const ObjectDescriptor &desc = allObjects[i];
                if (desc.has_type(type))
                    func(ObjectLocation98{static_cast<int>(i)});
            }
        }
    };

    class LandmarkFactoryLogistics98 : public LandmarkFactory
    {

        struct HashFunction
        {
            size_t operator()(const std::pair<int, int> &pos) const
            {
                size_t rowHash = std::hash<int>()(pos.first);
                size_t colHash = std::hash<int>()(pos.second) << 1;
                return rowHash ^ colHash;
            }
        };

        struct NextHash
        {
            size_t operator()(const FactProxy &pos) const
            {
                size_t rowHash = std::hash<int>()(pos.get_value());
                size_t colHash = std::hash<int>()(pos.get_pair().var) << 1;
                return rowHash ^ colHash;
            }
        };

    public:
        bool integrated = false;
        explicit LandmarkFactoryLogistics98(const options::Options &opts);

        virtual bool supports_conditional_effects() const override;

        virtual void
        generate_landmarks(const std::shared_ptr<AbstractTask> &task) override;

        void generate_truck_graph(TaskProxy &task);

        void add_landmarks_for_location_based(const LandmarkMapper98::MapperPair98 &mapperPair, const State &state);
        void add_landmarks_for_truck_based(const LandmarkMapper98::MapperPair98 &mapperPair, const State &state);
        void add_landmarks_for_airplane_based(const LandmarkMapper98::MapperPair98 &mapperPair, const State &state);

        void print(const TaskProxy &taskProxy);

        bool initialized = false;

        std::unordered_map<LandmarkMapper98::ObjectLocation98, LandmarkMapper98::MapperPair98, LandmarkMapper98::ObjectLocation98::ObjHash98> goalLocations2;

        void parse_logistics(const TaskProxy task_proxy);

        bool is_package_in_airplane(LandmarkMapper98::MapperPair98 mapperPair)
        {
            return mapper.has_value_with_type(mapperPair.key, LandmarkMapper98::ObjectType98::Airplane);
        }
        // package at destination city checks the goal location and the package city
        bool pack_at_destination_city(LandmarkMapper98::MapperPair98 pack)
        {

            LandmarkMapper98::CityInfo98 info1 = mapper.get_city_info(pack.val);

            LandmarkMapper98::CityInfo98 info2 = mapper.get_city_info(goalLocations2.at(pack.key).val);
            return info1.city == info2.city;
        }
        // is package at goal returns true, if not then false
        bool pack_at_goal(const LandmarkMapper98::MapperPair98 &pack)
        {
            /*auto it = find_if(goalLocations2.begin(), goalLocations2.end(),[&](){
                return goalLocations2.at(pack.key);
            });
            if(it == goalLocations2.end()){
                return true; 
            }*/
            return pack.val == goalLocations2.at(pack.key).val;
        }

        LandmarkMapper98 mapper;

        void calc_achievers(const TaskProxy &task_proxy);
    };
}