#include <fstream>
#include "landmark_factory_logistics98.h"
#include "../utils/logging.h"
#include "landmark_graph.h"
#include "../plugin.h"
#include "../option_parser.h"

namespace landmarks
{
    //finds in all object the description and assigns a value
    int LandmarkMapper98::find_object(const std::string &name)
    {
        auto it = find_if(allObjects.begin(), allObjects.end(),
                          [&](const ObjectDescriptor &desc)
                          { return desc.name == name; });

        if (it == allObjects.end())
            return -1;

        return std::distance(allObjects.begin(), it);
    }
    // used to debug, receiving an enum, returning a char pointer
    const char *LandmarkMapper98::get_type_name(ObjectType98 type)
    {
        switch (type)
        {
        case ObjectType98::Airplane:
            return "Airplane";
        case ObjectType98::Airport:
            return "Airport";
        case ObjectType98::City:
            return "City";
        case ObjectType98::Location:
            return "Location";
        case ObjectType98::Package:
            return "Package";
        case ObjectType98::Truck:
            return "Truck";
        default:
            return "Unknown";
        }
    }
    // used with the static-information.out file, otherwise doesn't work. Will perform
    // substring search and parse everything from the file except in-city
    void LandmarkMapper98::init(const char *path, const TaskProxy &taskProxy)
    {
        std::ifstream static_properties(path);
        assert(static_properties.is_open());

        std::string property;
        while (static_properties)
        {
            getline(static_properties, property);
            int begin = property.find(' ');
            int center = property.find('(');
            int end = property.find(')');

            // airplane
            std::string type = property.substr(begin + 1, center - begin - 1);

            // apn1
            std::string name = property.substr(center + 1, end - center - 1);

            ObjectType98 objType = ObjectType98::Airplane;
            if (type == "airplane")
            {
                objType = ObjectType98::Airplane;
            }
            else if (type == "airport")
            {
                objType = ObjectType98::Airport;
            }
            else if (type == "city")
            {
                objType = ObjectType98::City;
            }
            else if (type == "location")
            {
                objType = ObjectType98::Location;
            }
            else if (type == "obj")
            {
                objType = ObjectType98::Package;
            }
            else if (type == "truck")
            {
                objType = ObjectType98::Truck;
            }
            else
            {
                continue;
            }

            int opt_idx = find_object(name);
            if (opt_idx == -1)
            {
                allObjects.push_back({{objType}, name});
                if (objType == ObjectType98::Truck || objType == ObjectType98::Airplane)
                {
                    vehicles.push_back({{objType}, name});
                }
            }
            else
            {
                allObjects[opt_idx].types.push_back(objType);
            }
        }

        // Lookup for state variables
        for (auto var : taskProxy.get_variables())
        {
            variable_id_locations[var.get_id()] = 0;
            value_locations[var.get_id()].resize(var.get_domain_size());
            value_locations_truck[var.get_id()].resize(var.get_domain_size());
            value_locations_air[var.get_id()].resize(var.get_domain_size());

            for (int i = 0; i < var.get_domain_size(); ++i)
            {
                FactProxy fact = var.get_fact(i);
                std::string fact_name = fact.get_name();

                // Atom at(tru2, pos2)
                int center = fact_name.find(',');
                int sub = fact_name.find('(');
                int end = fact_name.find(')');

                // tru2
                std::string variable_name = fact_name.substr(sub + 1, center - sub - 1);

                // tru1, pos2 -> tru1 kann nicht zu pos2 -> nicht gleiche city
                // tru2, pos2

                // pos2
                std::string location = fact_name.substr(center + 2, end - center - 2);

                //utils::g_log << "Mapping [" << variable_name << "] -> "
                //         << "[" << location << "]" << std::endl;

                // indexOf(pos2)
                int obj_index = find_object(location);
                assert(obj_index != -1);
                //utils::g_log << "Obj Index: " << obj_index << std::endl;

                int var_index = find_object(variable_name);
                assert(var_index != -1);
                //utils::g_log << "Var Index Index: " << var_index << std::endl;
                //utils::g_log << "Fact Value: " << fact.get_value() << std::endl;

                variable_id_locations[var.get_id()] = var_index; // Note: is written multiple times
                locations_to_variable_id[var_index] = var.get_id();

                value_locations[var.get_id()][fact.get_value()] = obj_index;
            }
        }
//for debugging
#if 0
        for (size_t i = 0; i < allObjects.size(); ++i)
        {
            const ObjectDescriptor &desc = allObjects[i];
            utils::g_log << "Index: " << i << " Name: " << desc.name << std::endl;
            utils::g_log << "Types: " << std::endl;
            for (ObjectType98 type : desc.types)
            {
                utils::g_log << " >> " << get_type_name(type) << std::endl;
            }
        }

        for (auto x : locations_to_variable_id)
        {

            utils::g_log << "LOC-Left: " << x.first << " --> FactID: " << x.second << std::endl;
        }
        for (auto x : value_locations)
        {
            for (size_t i = 0; i < x.second.size(); ++i)
            {
                utils::g_log << "FactID: " << x.first << " Fact-ID-Right: " << i << " LOC-Right: " << x.second[i] << std::endl;
            }
        }

#endif
    }
    // parses arguements from in city, uses the static information out file
    void LandmarkMapper98::parse_inCity(const char *path)
    {

        std::ifstream static_properties(path);
        assert(static_properties.is_open());

        std::string property;
        while (static_properties)
        {
            getline(static_properties, property);
            int begin = property.find(' ');
            int center = property.find('(');
            int end = property.find(')');

            // Atom airplane(apn1)
            // Atom in-city(apt1, cit1)

            // airplane
            std::string type = property.substr(begin + 1, center - begin - 1);

            // apn1
            std::string name = property.substr(center + 1, end - center - 1);

            if (type == "in-city")
            {
                int center = name.find(", ");

                // apt1
                std::string location = name.substr(0, center);
                // cit1
                std::string city = name.substr(center + 2);

                int loc_index = find_object(location);
                assert(loc_index != -1);

                int city_index = find_object(city);
                assert(city_index != -1);

                auto it = std::find_if(city_infos.begin(), city_infos.end(),
                                       [city_index](const CityInfo98 &info)
                                       { return info.city.index == city_index; });
                if (it == city_infos.end())
                {
                    city_infos.push_back({ObjectLocation98{city_index}, {}, -1, {}});
                    it = city_infos.end() - 1;
                }

                if (allObjects[loc_index].has_type(ObjectType98::Airport))
                    it->airport = ObjectLocation98{loc_index};
                else
                    it->position.push_back(ObjectLocation98{loc_index});

                // idx_loc(apt1) -> idx_cit(cit1)
                // is_in_city[ObjectLocation98{loc_index}] = ObjectLocation98{city_index};
            }
        }
        // get information from the problem file here
        for (const auto &pair : variable_id_locations)
        {
            if (allObjects[pair.second].has_type(ObjectType98::Truck))
            {
                auto const &values = value_locations[pair.first];
                for (int value_index : values)
                {
                    // utils::g_log << "Get city info from: " << get_name(ObjectLocation98{value_index}) << " for truck: " << get_name(ObjectLocation98{pair.second}) << std::endl;

                    CityInfo98 *info = get_city_info_ptr(ObjectLocation98{value_index});
                    assert(info != nullptr);

                    info->trucks.push_back(ObjectLocation98{pair.second});
                    break;
                }
            }
        }
//again debugging methods
#if 0
        utils::g_log << "BANANE!!!!!" << std::endl;

        for(auto x : city_infos){
            utils::g_log << "Airport: "<< get_name(x.airport) << " City: " << get_name(x.city) << std::endl; 
            utils::g_log << "Locations: " << std::endl;
            for (size_t i = 0; i < x.position.size(); ++i)
            {
                utils::g_log << " >> " << get_name(x.position[i]) << std::endl;
            }
            utils::g_log << "Trucks: " << std::endl;
            for (size_t i = 0; i < x.trucks.size(); ++i)
            {
                utils::g_log << " >> " << get_name(x.trucks[i]) << std::endl;
            }            

        }
#endif
    }
    // from a fact pair create the according MapperPair
    LandmarkMapper98::MapperPair98 LandmarkMapper98::from_fact_pair(const FactPair &fact)
    {
        auto var_it = variable_id_locations.find(fact.var);
        assert(var_it != variable_id_locations.end());

        auto val_it = value_locations.find(fact.var);
        assert(val_it != value_locations.end());
        int var_loc = var_it->second;
        int val_loc = val_it->second[fact.value];

        return {{var_loc}, {val_loc}};
    }
    // from a MapperPair create according fact pair
    FactPair LandmarkMapper98::to_fact_pair(const MapperPair98 &MapperPair98)
    {
        //utils::g_log << "to_fact_pair(" << MapperPair98.key.index << "," << MapperPair98.val.index << ")" << std::endl;

        auto var_it = locations_to_variable_id.find(MapperPair98.key.index);
        assert(var_it != locations_to_variable_id.end());

        int fact_var = var_it->second;
        //utils::g_log << fact_var << std::endl;
        auto val_it = value_locations.find(fact_var);
        //utils::g_log << "Mapper pair value index >> " << MapperPair98.val.index << std::endl;
        int fact_value = -1;
        for (size_t i = 0; i < val_it->second.size(); ++i)
        {
            //utils::g_log << "Val iterator second index " << val_it -> second[i] << std::endl;
            if (val_it->second[i] == MapperPair98.val.index)
            {
                fact_value = i;
                break;
            }
        }

        assert(fact_value != -1);

        return FactPair(fact_var, fact_value);
    }
    // gets the state id, used for search in the state structure
    int LandmarkMapper98::to_state_id(ObjectLocation98 location)
    {
        auto var_it = locations_to_variable_id.find(location.index);
        assert(var_it != locations_to_variable_id.end());

        return var_it->second;
    }
    // searches value with a given type
    bool LandmarkMapper98::has_value_with_type(ObjectLocation98 key, ObjectType98 type)
    {
        auto var_it = locations_to_variable_id.find(key.index);
        if (var_it == locations_to_variable_id.end())
            return false;

        int fact_var = var_it->second;
        auto val_it = value_locations.find(fact_var);

        const auto &values = val_it->second;
        for (size_t i = 0; i < values.size(); ++i)
        {
            if (allObjects[values[i]].has_type(type))
                return true;
        }
        return false;
    }
    // for debugging, getting the name
    const std::string &LandmarkMapper98::get_name(ObjectLocation98 location)
    {
        return allObjects[location.index].name;
    }
    // for debugging
    std::string LandmarkMapper98::get_name(MapperPair98 MapperPair98)
    {
        return allObjects[MapperPair98.key.index].name + ", " + allObjects[MapperPair98.val.index].name;
    }
    // identifies type
    bool LandmarkMapper98::has_type(ObjectLocation98 location, ObjectType98 type)
    {
        return allObjects[location.index].has_type(type);
    }
    // identifies vehicle
    bool LandmarkMapper98::is_vehicle(ObjectLocation98 location)
    {
        return allObjects[location.index].is_vehicle();
    }
    // searches in all cities the according city with the objectlocation provided
    LandmarkMapper98::CityInfo98 *LandmarkMapper98::get_city_info_ptr(ObjectLocation98 location)
    {
        if (has_type(location, ObjectType98::City))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo98 &info)
                              { return info.city == location; });
            assert(it != city_infos.end());
            return &(*it);
        }
        else if (has_type(location, ObjectType98::Airport))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo98 &info)
                              { return info.airport == location; });
            assert(it != city_infos.end());
            return &(*it);
        }
        else if (has_type(location, ObjectType98::Truck))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo98 &info)
                              {
                                  for (size_t i = 0; i < info.trucks.size(); ++i)
                                  {
                                      if (info.trucks[i] == location)
                                          return true;
                                  }
                                  return false;
                              });
            assert(it != city_infos.end());
            return &(*it);
        }
        else if (has_type(location, ObjectType98::Location))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo98 &info)
                              {
                                  for (size_t i = 0; i < info.position.size(); ++i)
                                  {
                                      if (info.position[i] == location)
                                          return true;
                                  }
                                  return false;
                              });
            assert(it != city_infos.end());
            return &(*it);
        }
        else
        {
            assert(false);
        }
        return nullptr;
    }
    // creates a copy of the city if found, if not returns dummy city which does not exist
    LandmarkMapper98::CityInfo98 LandmarkMapper98::get_city_info(ObjectLocation98 location)
    {
        LandmarkMapper98::CityInfo98 *optLoc = get_city_info_ptr(location);
        if (optLoc != nullptr)
            return *optLoc;
        return {{-1}, {}, {-1}, {}};
    }

    LandmarkFactoryLogistics98::LandmarkFactoryLogistics98(
        const options::Options &options)
        : LandmarkFactory(options),
          integrated(options.get<bool>("integrated"))
    {
    }
    // generating the lookup maps, city infos and mapper pairs
    void LandmarkFactoryLogistics98::generate_landmarks(
        const std::shared_ptr<AbstractTask> &task)
    {

        TaskProxy taskProxy(*task);

        // initialize the lookup table with all facts. Using a hashmap is better here.
        if (!initialized)
        {
            initialized = true;
            mapper.init("static-information.out", taskProxy);
            mapper.parse_inCity("static-information.out");

            //print(taskProxy);
        }
        generate_truck_graph(taskProxy);
        calc_achievers(taskProxy);
    }
    // calculates achievers, is copied from other file
    void LandmarkFactoryLogistics98::calc_achievers(const TaskProxy &task_proxy)
    {
        VariablesProxy variables = task_proxy.get_variables();
        for (auto &lmn : lm_graph->get_nodes())
        {
            for (const FactPair &lm_fact : lmn->facts)
            {
                const std::vector<int> &ops = get_operators_including_eff(lm_fact);
                lmn->possible_achievers.insert(ops.begin(), ops.end());

                if (variables[lm_fact.var].is_derived())
                    lmn->is_derived = true;
            }
        }
    }
    // refactor this. This generates all landmarks for the goal state and iterates through all packages, calling different functions
    void LandmarkFactoryLogistics98::generate_truck_graph(TaskProxy &taskProxy)
    {

        State state = taskProxy.get_initial_state();
        for (FactProxy goal : taskProxy.get_goals())
        {

            if (state[goal.get_variable()].get_value() != goal.get_value())
            {
                LandmarkNode &lmNode = lm_graph->add_simple_landmark(
                    goal.get_pair());
                lmNode.is_true_in_goal = true;
                //Atom at(obj23, pos2)
                //in-city(pos2, cit2)
            }
            LandmarkMapper98::MapperPair98 MapperPair98 = mapper.from_fact_pair(goal.get_pair());
            goalLocations2.emplace(MapperPair98.key, MapperPair98);
        }

        for (FactProxy fact : state)
        {

            // utils::g_log << "Or here? " << std::endl;
            LandmarkMapper98::MapperPair98 MapperPair98 = mapper.from_fact_pair(fact.get_pair());
            //Atom at(obj23, pos2) -> goalLocations position
            //utils::g_log << mapper.get_name(MapperPair98) << std::endl;

            //Atom in(obj13, tru2)
            //utils::g_log << fact.get_name() << std::endl;
            if (mapper.has_type(MapperPair98.key, LandmarkMapper98::ObjectType98::Package) && mapper.has_type(MapperPair98.val, LandmarkMapper98::ObjectType98::Location))
            {

                if (pack_at_goal(MapperPair98))
                {
                    continue;
                }
                add_landmarks_for_location_based(MapperPair98, state);
            }

            else if (mapper.has_type(MapperPair98.key, LandmarkMapper98::ObjectType98::Package) && mapper.has_type(MapperPair98.val, LandmarkMapper98::ObjectType98::Truck))
            {
                add_landmarks_for_truck_based(MapperPair98, state);
            }
            else if (mapper.has_type(MapperPair98.key, LandmarkMapper98::ObjectType98::Package) && mapper.has_type(MapperPair98.val, LandmarkMapper98::ObjectType98::Airplane))
            {
                add_landmarks_for_airplane_based(MapperPair98, state);
            }
        }
        //add_order();

        lm_graph->set_landmark_ids();
    }
    // if the location is either a position or airport, different landmarks are added. Corresponding landmarks are from the paper 2017 Paul et al.
    void LandmarkFactoryLogistics98::add_landmarks_for_location_based(const LandmarkMapper98::MapperPair98 &MapperPair98, const State &state)
    {
        if (pack_at_destination_city(MapperPair98))
        {
            //utils::g_log << "Processing: " << mapper.get_name(MapperPair98) << std::endl;

            //apt2 oder pos2
            // if true, pack already at destination
            // if not true -> truck goes to pack.val -> location  --> einladen --> zur goallocation fahren, fertig
            LandmarkMapper98::CityInfo98 cityInfo = mapper.get_city_info(MapperPair98.val); // city infos
            std::vector<LandmarkMapper98::ObjectLocation98> trucks = cityInfo.trucks;

            std::set<FactPair> truck_at_origin{};
            std::set<FactPair> load_in_truck{};
            std::set<FactPair> truck_at_destination{};

            for (LandmarkMapper98::ObjectLocation98 truck : trucks)
            {
                truck_at_origin.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, MapperPair98.val}));
                load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
            }

            lm_graph->add_disjunctive_landmark(load_in_truck);                           // einladen
            LandmarkNode &to = lm_graph->add_disjunctive_landmark(truck_at_destination); // zur goalLocation fahren

            if (std::none_of(trucks.begin(), trucks.end(), [&](LandmarkMapper98::ObjectLocation98 truck)
                             { return mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair()).val == MapperPair98.val; }))
            {
                LandmarkNode &from = lm_graph->add_disjunctive_landmark(truck_at_origin); // zum paket fahren

                edge_add(from, to, EdgeType::REASONABLE);
            }
        }
        else
        {
            LandmarkMapper98::CityInfo98 cityInfo = mapper.get_city_info(MapperPair98.val);
            std::vector<LandmarkMapper98::ObjectLocation98> trucks = cityInfo.trucks;

            std::set<FactPair> truck_at_origin{};
            std::set<FactPair> load_in_truck{};
            std::set<FactPair> truck_at_airport{};
            for (LandmarkMapper98::ObjectLocation98 truck : trucks)
            {
                truck_at_origin.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, MapperPair98.val}));
                load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, cityInfo.airport}));
            }

            if (cityInfo.airport != MapperPair98.val)
            {
                //utils::g_log << "Crash here?" << std::endl;

                if (std::none_of(trucks.begin(), trucks.end(), [&](LandmarkMapper98::ObjectLocation98 truck)
                                 { return mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair()).val == MapperPair98.val; }))
                {
                    //utils::g_log << "Crash here? right????" << std::endl;
                    LandmarkNode &from = lm_graph->add_disjunctive_landmark(truck_at_origin); //zum paket fahren
                    //utils::g_log << "Crash here?" << std::endl;
                    lm_graph->add_disjunctive_landmark(load_in_truck);                                                                      // paket einladen
                    LandmarkNode &to = lm_graph->add_disjunctive_landmark(truck_at_airport);                                                // zum flughafen fahren
                    edge_add(from, to, EdgeType::REASONABLE);                                                                               // kante zuerst beim paket -> beim flughafen
                    lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, cityInfo.airport})); // paket abladen

                    std::vector<LandmarkMapper98::MapperPair98> airplanes{};
                    mapper.get_by(LandmarkMapper98::ObjectType98::Airplane,
                                  [&](LandmarkMapper98::ObjectLocation98 loc)
                                  {
                                      airplanes.push_back(mapper.from_fact_pair(state[mapper.to_state_id(loc)].get_pair()));
                                  });
                    LandmarkMapper98::CityInfo98 otherCity = mapper.get_city_info(goalLocations2.at(MapperPair98.key).val);

                    std::set<FactPair> plane_at_origin{};
                    std::set<FactPair> load_in_plane{};
                    std::set<FactPair> plane_at_destination{};
                    for (LandmarkMapper98::MapperPair98 airplane : airplanes)
                    {
                        plane_at_origin.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, cityInfo.airport}));
                        load_in_plane.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, airplane.key}));
                        plane_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, otherCity.airport}));
                    }

                    lm_graph->add_disjunctive_landmark(load_in_plane);                            // in flugzeug laden
                    LandmarkNode &to1 = lm_graph->add_disjunctive_landmark(plane_at_destination); // zur destination fliegen

                    if (pack_at_goal(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}))
                    {
                        if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                         { return airplane.val == cityInfo.airport; }))
                        {
                            LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                            edge_add(from1, to1, EdgeType::REASONABLE);
                        }
                        return;
                    }
                    std::set<FactPair> other_load_in_truck{};
                    std::set<FactPair> other_truck_at_airport{};
                    std::set<FactPair> other_truck_at_destination{};
                    for (LandmarkMapper98::ObjectLocation98 truck : otherCity.trucks)
                    {
                        other_truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, otherCity.airport}));
                        other_load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                        other_truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
                    }

                    LandmarkNode &lmTo = lm_graph->add_disjunctive_landmark(other_truck_at_destination); // der andere truck zum zielort

                    if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                     { return airplane.val == cityInfo.airport; }))
                    {
                        LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                        edge_add(from1, to1, EdgeType::REASONABLE);
                        if (integrated)
                            if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                                edge_add(from1, lmTo, EdgeType::REASONABLE);
                    }

                    lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}));
                    lm_graph->add_disjunctive_landmark(other_load_in_truck);
                    if (std::none_of(other_truck_at_airport.begin(), other_truck_at_airport.end(), [&](FactPair truck)
                                     { return mapper.from_fact_pair(state[truck.var].get_pair()).val == otherCity.airport; }))
                    {
                        LandmarkNode &lmFrom = lm_graph->add_disjunctive_landmark(other_truck_at_airport);
                        edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
                    }

                    if (integrated)
                    {
                        if (MapperPair98.val != cityInfo.airport)
                        {

                            edge_add(from, to1, EdgeType::REASONABLE); // if pos(p) not airport and truck not there then, truck at pos before airplane in dest city
                            if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                            {
                                edge_add(to, lmTo, EdgeType::REASONABLE);   // if pos(p) nor dest(p) an airport and truck not there, then truck at pos of pack before truck at goal
                                edge_add(from, lmTo, EdgeType::REASONABLE); // if pos(p) no truck and dest not in airport nor pos(p) in airport, then truck at pos before truck in dest
                            }
                            edge_add(to, to1, EdgeType::REASONABLE); // if pos(p) not an airport truck has to drive to airport before airplane in destination city
                            if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                            {
                                edge_add(to1, lmTo, EdgeType::REASONABLE);
                            }
                        }
                    }
                }
                else
                {
                    //utils::g_log << "Crash here?" << std::endl;

                    lm_graph->add_disjunctive_landmark(load_in_truck);
                    LandmarkNode &lmTruck = lm_graph->add_disjunctive_landmark(truck_at_airport);

                    lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, cityInfo.airport}));

                    std::vector<LandmarkMapper98::MapperPair98> airplanes{};
                    mapper.get_by(LandmarkMapper98::ObjectType98::Airplane,
                                  [&](LandmarkMapper98::ObjectLocation98 loc)
                                  {
                                      airplanes.push_back(mapper.from_fact_pair(state[mapper.to_state_id(loc)].get_pair()));
                                  });
                    LandmarkMapper98::CityInfo98 otherCity = mapper.get_city_info(goalLocations2.at(MapperPair98.key).val);

                    std::set<FactPair> plane_at_origin{};
                    std::set<FactPair> load_in_plane{};
                    std::set<FactPair> plane_at_destination{};
                    for (LandmarkMapper98::MapperPair98 airplane : airplanes)
                    {
                        plane_at_origin.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, cityInfo.airport}));
                        load_in_plane.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, airplane.key}));
                        plane_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, otherCity.airport}));
                    }

                    lm_graph->add_disjunctive_landmark(load_in_plane);
                    //utils::g_log << "Then here?" << std::endl;

                    LandmarkNode &to1 = lm_graph->add_disjunctive_landmark(plane_at_destination); // CRASH

                    if (pack_at_goal(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}))
                    {
                        if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                         { return airplane.val == cityInfo.airport; }))
                        {
                            LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                            edge_add(from1, to1, EdgeType::REASONABLE);
                        }

                        return;
                    }
                    std::set<FactPair> other_load_in_truck{};
                    std::set<FactPair> other_truck_at_airport{};
                    std::set<FactPair> other_truck_at_destination{};
                    for (LandmarkMapper98::ObjectLocation98 truck : otherCity.trucks)
                    {
                        //utils::g_log <<"Truck: " << truck.index << " MapperPair.key: " << MapperPair98.key.index << " othercity.airport: " << otherCity.airport.index << std::endl;

                        //utils::g_log << "to_fact_pair_1" << std::endl;
                        other_truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, otherCity.airport}));

                        //utils::g_log << "to_fact_pair_2" << std::endl;
                        other_load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));

                        //utils::g_log << "to_fact_pair_3" << std::endl;
                        other_truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
                    }

                    LandmarkNode &lmTo = lm_graph->add_disjunctive_landmark(other_truck_at_destination);

                    if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                     { return airplane.val == cityInfo.airport; }))
                    {
                        LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                        edge_add(from1, to1, EdgeType::REASONABLE);
                        if (integrated)
                            if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                                edge_add(from1, lmTo, EdgeType::REASONABLE);
                    }
                    lm_graph->add_disjunctive_landmark(other_load_in_truck);
                    lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}));
                    //utils::g_log << "Crashed &&&&&&&&&&&&&&&&&&& "<<std::endl;
                    if (std::none_of(other_truck_at_airport.begin(), other_truck_at_airport.end(), [&](FactPair truck)
                                     { return mapper.from_fact_pair(state[truck.var].get_pair()).val == otherCity.airport; }))
                    {
                        LandmarkNode &lmFrom = lm_graph->add_disjunctive_landmark(other_truck_at_airport);
                        edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
                    }
                    if (integrated)
                    {
                        if (MapperPair98.val != cityInfo.airport)
                        {
                            if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                            {
                                edge_add(lmTruck, lmTo, EdgeType::REASONABLE);
                            }
                            edge_add(lmTruck, to1, EdgeType::REASONABLE);
                        }
                        if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                        {
                            edge_add(to1, lmTo, EdgeType::REASONABLE);
                        }
                    }
                }
            }
            else
            {

                std::vector<LandmarkMapper98::MapperPair98> airplanes{};
                LandmarkMapper98::CityInfo98 otherCity = mapper.get_city_info(goalLocations2.at(MapperPair98.key).val);

                mapper.get_by(LandmarkMapper98::ObjectType98::Airplane,
                              [&](LandmarkMapper98::ObjectLocation98 loc)
                              {
                                  airplanes.push_back(mapper.from_fact_pair(state[mapper.to_state_id(loc)].get_pair()));
                              });
                std::set<FactPair> plane_at_origin{};
                std::set<FactPair> load_in_plane{};
                std::set<FactPair> plane_at_destination{};
                for (LandmarkMapper98::MapperPair98 airplane : airplanes)
                {
                    plane_at_origin.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, cityInfo.airport}));
                    load_in_plane.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, airplane.key}));
                    plane_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, otherCity.airport}));
                }

                lm_graph->add_disjunctive_landmark(load_in_plane);
                LandmarkNode &to1 = lm_graph->add_disjunctive_landmark(plane_at_destination);
                if (pack_at_goal(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}))
                {
                    if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                     { return airplane.val == cityInfo.airport; }))
                    {
                        LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                        edge_add(from1, to1, EdgeType::REASONABLE);
                    }

                    return;
                }

                std::set<FactPair> other_load_in_truck{};
                std::set<FactPair> other_truck_at_airport{};
                std::set<FactPair> other_truck_at_destination{};
                for (LandmarkMapper98::ObjectLocation98 truck : otherCity.trucks)
                {
                    other_truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, otherCity.airport}));
                    other_load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                    other_truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
                }

                LandmarkNode &lmTo = lm_graph->add_disjunctive_landmark(other_truck_at_destination); // truck zum zielort

                if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                 { return airplane.val == cityInfo.airport; }))
                {
                    LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                    edge_add(from1, to1, EdgeType::REASONABLE);
                    if (integrated)
                        edge_add(from1, lmTo, EdgeType::REASONABLE);
                }
                lm_graph->add_disjunctive_landmark(other_load_in_truck);                                                                 // in anderen truck einladen
                lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport})); // am anderen airport ausladen

                if (std::none_of(other_truck_at_airport.begin(), other_truck_at_airport.end(), [&](FactPair truck)
                                 { return mapper.from_fact_pair(state[truck.var].get_pair()).val == otherCity.airport; }))
                {
                    LandmarkNode &lmFrom = lm_graph->add_disjunctive_landmark(other_truck_at_airport); // truck zum flughafen
                    edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
                }

                if (integrated)
                    edge_add(to1, lmTo, EdgeType::REASONABLE);
            }
        }
    }
    // the package is in a truck, therefore needs different landmarks
    void LandmarkFactoryLogistics98::add_landmarks_for_truck_based(const LandmarkMapper98::MapperPair98 &MapperPair98, const State &state)
    {
        if (pack_at_destination_city(MapperPair98))
        {
            LandmarkMapper98::ObjectLocation98 truck = MapperPair98.val; // truck zur city
            LandmarkMapper98::MapperPair98 truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair());
            // utils::g_log << "Processing: " << mapper.get_name(truckMapper) << std::endl;
            if (pack_at_goal(LandmarkMapper98::MapperPair98{MapperPair98.key, truckMapper.val}))
            {
                return;
            }
            else
            {
                lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
            }
        }
        else
        {
            LandmarkMapper98::ObjectLocation98 truck = MapperPair98.val;
            // LandmarkMapper98::MapperPair98 truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair());
            LandmarkMapper98::CityInfo98 cityInfo = mapper.get_city_info(truck);
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, cityInfo.airport})); // paket at airport

            std::vector<LandmarkMapper98::MapperPair98> airplanes{};
            mapper.get_by(LandmarkMapper98::ObjectType98::Airplane,
                          [&](LandmarkMapper98::ObjectLocation98 loc)
                          {
                              airplanes.push_back(mapper.from_fact_pair(state[mapper.to_state_id(loc)].get_pair()));
                          });
            LandmarkMapper98::CityInfo98 otherCity = mapper.get_city_info(goalLocations2.at(MapperPair98.key).val);

            std::set<FactPair> plane_at_origin{};
            std::set<FactPair> load_in_plane{};
            std::set<FactPair> plane_at_destination{};
            for (LandmarkMapper98::MapperPair98 airplane : airplanes)
            {
                plane_at_origin.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, cityInfo.airport}));
                load_in_plane.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, airplane.key}));
                plane_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, otherCity.airport}));
            }

            lm_graph->add_disjunctive_landmark(load_in_plane);
            LandmarkNode &to1 = lm_graph->add_disjunctive_landmark(plane_at_destination);

            if (truck != cityInfo.airport)
            {

                LandmarkNode &truckAtAirport = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, cityInfo.airport})); //truck drives to airport

                if (pack_at_goal(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}))
                {
                    if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                     { return airplane.val == cityInfo.airport; }))
                    {
                        LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                        edge_add(from1, to1, EdgeType::REASONABLE);
                    }
                    return;
                }

                std::vector<LandmarkMapper98::ObjectLocation98> otherTrucks = otherCity.trucks;
                std::set<FactPair> other_load_in_truck{};
                std::set<FactPair> other_truck_at_airport{};
                std::set<FactPair> other_truck_at_destination{};
                for (LandmarkMapper98::ObjectLocation98 truck : otherTrucks)
                {
                    other_truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, otherCity.airport}));
                    other_load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                    other_truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
                }

                LandmarkNode &to = lm_graph->add_disjunctive_landmark(other_truck_at_destination);
                if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                 { return airplane.val == cityInfo.airport; }))
                {
                    LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin); //plane at origin
                    edge_add(from1, to1, EdgeType::REASONABLE);
                    if (integrated)
                    {
                        edge_add(from1, to, EdgeType::REASONABLE);
                    }
                }

                lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}));
                lm_graph->add_disjunctive_landmark(other_load_in_truck);

                if (std::none_of(other_truck_at_airport.begin(), other_truck_at_airport.end(), [&](FactPair truck)
                                 { return mapper.from_fact_pair(state[truck.var].get_pair()).val == otherCity.airport; }))
                {
                    LandmarkNode &lmFrom = lm_graph->add_disjunctive_landmark(other_truck_at_airport);
                    edge_add(lmFrom, to, EdgeType::REASONABLE);
                }

                if (integrated)
                {

                    if (truck != cityInfo.airport)
                    {
                        if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                        {
                            edge_add(truckAtAirport, to, EdgeType::REASONABLE);
                        }
                        edge_add(truckAtAirport, to1, EdgeType::REASONABLE);
                    }
                    if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                    {
                        edge_add(to1, to, EdgeType::REASONABLE);
                    }
                }
            }
            else
            {
                if (pack_at_goal(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}))
                {
                    if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                     { return airplane.val == cityInfo.airport; }))
                    {
                        LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                        edge_add(from1, to1, EdgeType::REASONABLE);
                    }
                    return;
                }

                std::vector<LandmarkMapper98::ObjectLocation98> otherTrucks = otherCity.trucks;
                std::set<FactPair> other_load_in_truck{};
                std::set<FactPair> other_truck_at_airport{};
                std::set<FactPair> other_truck_at_destination{};
                for (LandmarkMapper98::ObjectLocation98 truck : otherTrucks)
                {
                    other_truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, otherCity.airport}));
                    other_load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                    other_truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
                }

                LandmarkNode &to = lm_graph->add_disjunctive_landmark(other_truck_at_destination);
                if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper98::MapperPair98 airplane)
                                 { return airplane.val == cityInfo.airport; }))
                {
                    LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                    edge_add(from1, to1, EdgeType::REASONABLE);
                    if (integrated)
                    {
                        edge_add(from1, to, EdgeType::REASONABLE);
                    }
                }

                lm_graph->add_disjunctive_landmark(other_load_in_truck);
                lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, otherCity.airport}));

                if (std::none_of(other_truck_at_airport.begin(), other_truck_at_airport.end(), [&](FactPair truck)
                                 { return mapper.from_fact_pair(state[truck.var].get_pair()).val == otherCity.airport; }))
                {
                    LandmarkNode &from = lm_graph->add_disjunctive_landmark(other_truck_at_airport);
                    edge_add(from, to, EdgeType::REASONABLE);
                }
                if (integrated)
                {
                    if (goalLocations2.at(MapperPair98.key).val != otherCity.airport)
                    {
                        edge_add(to1, to, EdgeType::REASONABLE);
                    }
                }
            }
        }
    }
    // the truck is in an airplane, therefore distinguish two cases again
    void LandmarkFactoryLogistics98::add_landmarks_for_airplane_based(const LandmarkMapper98::MapperPair98 &MapperPair98, const State &state)
    {
        LandmarkMapper98::MapperPair98 airplane = mapper.from_fact_pair(state[mapper.to_state_id(MapperPair98.val)].get_pair());
        LandmarkMapper98::MapperPair98 airplaneVal{MapperPair98.key, airplane.val};
        LandmarkMapper98::CityInfo98 cityInfo = mapper.get_city_info(goalLocations2.at(MapperPair98.key).val);

        if (pack_at_destination_city(airplaneVal))
        {

            //utils::g_log << "Does it crash here?" << std::endl;

            if (pack_at_goal(airplaneVal))
            {
                return;
            }
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, cityInfo.airport})); //unload
            // city infos
            std::vector<LandmarkMapper98::ObjectLocation98> trucks = cityInfo.trucks;
            std::set<FactPair> truck_at_airport{};
            std::set<FactPair> truck_at_destination;
            std::set<FactPair> load_in_truck{};
            for (LandmarkMapper98::ObjectLocation98 truck : trucks)
            {
                truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, cityInfo.airport}));
                load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
            }

            lm_graph->add_disjunctive_landmark(load_in_truck); // einladen

            LandmarkNode &lmTo = lm_graph->add_disjunctive_landmark(truck_at_destination); // zur goalLocation fahren

            // truck zur city
            if (std::none_of(truck_at_airport.begin(), truck_at_airport.end(), [&](FactPair truck)
                             { return mapper.from_fact_pair(state[truck.var].get_pair()).val == cityInfo.airport; }))
            {
                LandmarkNode &lmFrom = lm_graph->add_disjunctive_landmark(truck_at_airport); // zum paket fahren
                edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
            }
        }
        else
        {

            /*utils::g_log << "Crash key: " << toAdd1.key.index << " crash val: " << toAdd1.val.index << std::endl;
                    utils::g_log << "Airplane key: " << airplane.key.index << " crash val: " << cityInfo.airport.index << std::endl;*/

            LandmarkNode &lmAirport = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{airplane.key, cityInfo.airport})); //fly to airport

            if (pack_at_goal(LandmarkMapper98::MapperPair98{MapperPair98.key, cityInfo.airport}))
            {
                return;
            }

            std::vector<LandmarkMapper98::ObjectLocation98> trucks = cityInfo.trucks;
            std::set<FactPair> truck_at_airport{};
            std::set<FactPair> truck_at_destination;
            std::set<FactPair> load_in_truck{};
            for (LandmarkMapper98::ObjectLocation98 truck : trucks)
            {
                truck_at_airport.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, cityInfo.airport}));
                load_in_truck.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, truck}));
                truck_at_destination.insert(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{truck, goalLocations2.at(MapperPair98.key).val}));
            }

            LandmarkNode &lmTo = lm_graph->add_disjunctive_landmark(truck_at_destination);

            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper98::MapperPair98{MapperPair98.key, cityInfo.airport}));
            lm_graph->add_disjunctive_landmark(load_in_truck);

            if (std::none_of(truck_at_airport.begin(), truck_at_airport.end(), [&](FactPair truck)
                             { return mapper.from_fact_pair(state[truck.var].get_pair()).val == cityInfo.airport; }))
            {
                LandmarkNode &lmFrom = lm_graph->add_disjunctive_landmark(truck_at_airport);
                edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
            }

            if (integrated)
            {
                if (goalLocations2.at(MapperPair98.key).val != cityInfo.airport)
                {
                    edge_add(lmAirport, lmTo, EdgeType::REASONABLE);
                }
            }
        }
    }
    // has to be added because pure virtual function
    bool LandmarkFactoryLogistics98::supports_conditional_effects() const
    {
        return true;
    }
    // not needed anymore
    void LandmarkFactoryLogistics98::print(const TaskProxy &taskProxy)
    {

        for (FactProxy allFacts : taskProxy.get_variables().get_facts())
        {
            LandmarkMapper98::MapperPair98 MapperPair98 = mapper.from_fact_pair(allFacts.get_pair());
            utils::g_log << "Key: " << MapperPair98.key.index << " value: " << MapperPair98.val.index << std::endl;
            utils::g_log << "Name: " << mapper.get_name(MapperPair98) << std::endl;
        }
    }
    // parses command line arguments, need to change documentation notes sometime
    static std::shared_ptr<LandmarkFactory> _parse(OptionParser &parser)
    {
        parser.document_synopsis(
            "Logistics Landmarks",
            "Only for logistics task checks for each fact if it is a landmark."
            "This check is for Logistics Task");
        parser.document_note(
            "No relevant options right now",
            "reasonable_ordersddddddddddd, only_causal_landmarks");
        _add_options_to_parser(parser);
        parser.add_option<bool>("integrated",
                                "use integrated logistics landmark",
                                "false");

        Options opts = parser.parse();

        if (parser.dry_run())
            return nullptr;
        else
            return std::make_shared<LandmarkFactoryLogistics98>(opts);
    }
    // the lm logistics98 plugin
    static Plugin<LandmarkFactory> _plugin("lm_logistics98", _parse);
}
