#pragma once

#include "landmark_factory.h"
#include "../utils/logging.h"

namespace landmarks
{

    class LandmarkMapper
    {
public:
        enum class ObjectType
        {
            Airplane,
            Airport,
            City,
            Location,
            Package,
            Truck
        };

        struct ObjectLocation 
        { 
            int index; 

            struct ObjHash
            {
                size_t operator()(const ObjectLocation& loc) const
                {
                    return std::hash<int>()(loc.index);
                }
            };

            bool operator==(const ObjectLocation& other) const
            {
                return index == other.index;
            }

            bool operator!=(const ObjectLocation& other) const
            {
                return !(index == other.index);
            }
        };

        // Important: 
        struct MapperPair 
        {
            ObjectLocation key;
            ObjectLocation val;
        };

        struct CityInfo
        {
            ObjectLocation city;
            std::vector<ObjectLocation> position;
            ObjectLocation airport;
            ObjectLocation truck;
        };
private:
        // Linke Seite: Unique [ID] --> aka variable id
        // Rechte Seite: Unique pro Linke Seite --> aka value



        struct ObjectDescriptor
        {
            std::vector<ObjectType> types;
            std::string name;

            bool has_type(ObjectType type) const { return std::find(types.begin(), types.end(), type) != types.end(); }
            bool is_vehicle() const { return has_type(ObjectType::Airplane) || has_type(ObjectType::Truck); }
        };

        // All objects
        std::vector<ObjectDescriptor> allObjects;
        
        std::vector<ObjectDescriptor> vehicles; 

        std::vector<ObjectLocation> positions;

        // Mapper FactPair --> Object Index
        //      Key: Fact-Var-ID; Value Vector[Fact-Value-ID (per Var-ID)] = obj_location
        std::unordered_map<int, std::vector<int>> value_locations;

        std::unordered_map<int, std::vector<int>> value_locations_truck;
        std::unordered_map<int, std::vector<int>> value_locations_air; 

        // Mapper Key: Fact-Var-ID: Obj-Location
        std::unordered_map<int, int> variable_id_locations;
        std::unordered_map<int, int> locations_to_variable_id;

        std::vector<CityInfo> city_infos;
        // std::unordered_map<ObjectLocation, ObjectLocation, ObjectLocation::ObjHash> is_in_city;

        // Helper Functions
        int find_object(const std::string& name);
        const char* get_type_name(ObjectType type);

    public:

        
        void init(const char *path, const TaskProxy& taskProxy);
        void parse_inCity(const char *path);

        MapperPair from_fact_pair(const FactPair& fact);
        FactPair to_fact_pair(const MapperPair& mapperPair);
        int to_state_id(ObjectLocation location);

        const std::string& get_name(ObjectLocation location);
        std::string get_name(MapperPair mapperPair);
        
        bool has_type(ObjectLocation location, ObjectType type);
        bool is_vehicle(ObjectLocation location);

        std::vector<ObjectDescriptor> get_vehicles() {
            return vehicles;
        };

        // ObjectLocation get_in_city(ObjectLocation location);
        CityInfo* get_city_info_ptr(ObjectLocation location);
        CityInfo get_city_info(ObjectLocation location);

        bool has_value_with_type(ObjectLocation key, ObjectType type);
        
        FactPair get_vehicle_fact_pair_from_loc(const MapperPair& mapperPair,const ObjectType objType);

        template<typename FuncType>
        void get_by(ObjectType type, FuncType func)
        {
            for(size_t i = 0; i < allObjects.size(); ++i)
            {
                const ObjectDescriptor& desc = allObjects[i];
                if(desc.has_type(type))
                    func(ObjectLocation{static_cast<int>(i)});
            }
        }
    };

    class LandmarkFactoryLogistics : public LandmarkFactory
    {

        struct HashFunction
        {
            size_t operator()(const std::pair<int, int> &pos) const
            {
                size_t rowHash = std::hash<int>()(pos.first);
                size_t colHash = std::hash<int>()(pos.second) << 1;
                return rowHash ^ colHash;
            }
        };

        struct NextHash
        {
            size_t operator()(const FactProxy &pos) const
            {
                size_t rowHash = std::hash<int>()(pos.get_value());
                size_t colHash = std::hash<int>()(pos.get_pair().var) << 1;
                return rowHash ^ colHash;
            }
        };

    public:
        bool integrated = false;
        explicit LandmarkFactoryLogistics(const options::Options &opts);

        virtual bool supports_conditional_effects() const override;

        virtual void
        generate_landmarks(const std::shared_ptr<AbstractTask> &task) override;

        void generate_truck_graph(TaskProxy &task);



        void add_landmarks_for_location_based(const LandmarkMapper::MapperPair &mapperPair, const State &state);
        void add_landmarks_for_truck_based(const LandmarkMapper::MapperPair &mapperPair, const State &state);
        void add_landmarks_for_airplane_based(const LandmarkMapper::MapperPair &mapperPair, const State &state);



        void print(const TaskProxy &taskProxy); 

        bool initialized = false;
   
        std::unordered_map<LandmarkMapper::ObjectLocation, LandmarkMapper::MapperPair, LandmarkMapper::ObjectLocation::ObjHash> goalLocations2;


        void parse_logistics(const TaskProxy task_proxy);

        bool is_package_in_airplane(LandmarkMapper::MapperPair mapperPair)
        {     
            return mapper.has_value_with_type(mapperPair.key, LandmarkMapper::ObjectType::Airplane); 
        }
        
        bool pack_at_destination_city(LandmarkMapper::MapperPair pack)
        {

            LandmarkMapper::CityInfo info1 = mapper.get_city_info(pack.val);

            LandmarkMapper::CityInfo info2 = mapper.get_city_info(goalLocations2.at(pack.key).val);
            return info1.city == info2.city;
            
        }

        bool pack_at_goal(const LandmarkMapper::MapperPair &pack)
        {
            /*auto it = find_if(goalLocations2.begin(), goalLocations2.end(),[&](){
                return goalLocations2.at(pack.key);
            });
            if(it == goalLocations2.end()){
                return true; 
            }*/
            return pack.val == goalLocations2.at(pack.key).val;
        }

        LandmarkMapper mapper;

        void calc_achievers(const TaskProxy &task_proxy);
    };
}