#include <fstream>
#include "landmark_factory_logistics.h"
#include "../utils/logging.h"
#include "landmark_graph.h"
#include "../plugin.h"
#include "../option_parser.h"

namespace landmarks
{
    int LandmarkMapper::find_object(const std::string &name)
    {
        auto it = find_if(allObjects.begin(), allObjects.end(),
                          [&](const ObjectDescriptor &desc)
                          { return desc.name == name; });

        if (it == allObjects.end())
            return -1;

        return std::distance(allObjects.begin(), it);
    }

    const char *LandmarkMapper::get_type_name(ObjectType type)
    {
        switch (type)
        {
        case ObjectType::Airplane:
            return "Airplane";
        case ObjectType::Airport:
            return "Airport";
        case ObjectType::City:
            return "City";
        case ObjectType::Location:
            return "Location";
        case ObjectType::Package:
            return "Package";
        case ObjectType::Truck:
            return "Truck";
        default:
            return "Unknown";
        }
    }

    void LandmarkMapper::init(const char *path, const TaskProxy &taskProxy)
    {
        std::ifstream static_properties(path);
        assert(static_properties.is_open());

        std::string property;
        while (static_properties)
        {
            getline(static_properties, property);
            int begin = property.find(' ');
            int center = property.find('(');
            int end = property.find(')');

            // airplane
            std::string type = property.substr(begin + 1, center - begin - 1);

            // apn1
            std::string name = property.substr(center + 1, end - center - 1);

            ObjectType objType = ObjectType::Airplane;
            if (type == "airplane")
            {
                objType = ObjectType::Airplane;
            }
            else if (type == "airport")
            {
                objType = ObjectType::Airport;
            }
            else if (type == "city")
            {
                objType = ObjectType::City;
            }
            else if (type == "location")
            {
                objType = ObjectType::Location;
            }
            else if (type == "package")
            {
                objType = ObjectType::Package;
            }
            else if (type == "truck")
            {
                objType = ObjectType::Truck;
            }
            else
            {
                continue;
            }

            int opt_idx = find_object(name);
            if (opt_idx == -1)
            {
                allObjects.push_back({{objType}, name});
                if (objType == ObjectType::Truck || objType == ObjectType::Airplane)
                {
                    vehicles.push_back({{objType}, name});
                }
            }
            else
            {
                allObjects[opt_idx].types.push_back(objType);
            }
        }

        // Lookup for state variables
        for (auto var : taskProxy.get_variables())
        {
            variable_id_locations[var.get_id()] = 0;
            value_locations[var.get_id()].resize(var.get_domain_size());
            value_locations_truck[var.get_id()].resize(var.get_domain_size());
            value_locations_air[var.get_id()].resize(var.get_domain_size());

            for (int i = 0; i < var.get_domain_size(); ++i)
            {
                FactProxy fact = var.get_fact(i);
                std::string fact_name = fact.get_name();

                // Atom at(tru2, pos2)
                int center = fact_name.find(',');
                int sub = fact_name.find('(');
                int end = fact_name.find(')');

                // tru2
                std::string variable_name = fact_name.substr(sub + 1, center - sub - 1);

                // tru1, pos2 -> tru1 kann nicht zu pos2 -> nicht gleiche city
                // tru2, pos2

                // pos2
                std::string location = fact_name.substr(center + 2, end - center - 2);

                //utils::g_log << "Mapping [" << variable_name << "] -> "
                //         << "[" << location << "]" << std::endl;

                // indexOf(pos2)
                int obj_index = find_object(location);
                assert(obj_index != -1);
                //utils::g_log << "Obj Index: " << obj_index << std::endl;

                int var_index = find_object(variable_name);
                assert(var_index != -1);
                //utils::g_log << "Var Index Index: " << var_index << std::endl;
                //utils::g_log << "Fact Value: " << fact.get_value() << std::endl;

                variable_id_locations[var.get_id()] = var_index; // Note: is written multiple times
                locations_to_variable_id[var_index] = var.get_id();

                value_locations[var.get_id()][fact.get_value()] = obj_index;
            }
        }

#if 0
        for (size_t i = 0; i < allObjects.size(); ++i)
        {
            const ObjectDescriptor &desc = allObjects[i];
            utils::g_log << "Index: " << i << " Name: " << desc.name << std::endl;
            utils::g_log << "Types: " << std::endl;
            for (ObjectType type : desc.types)
            {
                utils::g_log << " >> " << get_type_name(type) << std::endl;
            }
        }


        for (auto x : locations_to_variable_id)
        {

            utils::g_log << "LOC-Left: " << x.first << " --> FactID: " << x.second << std::endl;
        }
        for (auto x : value_locations)
        {
            for (size_t i = 0; i < x.second.size(); ++i)
            {
                utils::g_log << "FactID: " << x.first << " Fact-ID-Right: " << i << " LOC-Right: " << x.second[i] << std::endl;
            }
        }

#endif
    }

    void LandmarkMapper::parse_inCity(const char *path)
    {

        std::ifstream static_properties(path);
        assert(static_properties.is_open());

        // Infos from static information
        std::string property;
        while (static_properties)
        {
            getline(static_properties, property);
            int begin = property.find(' ');
            int center = property.find('(');
            int end = property.find(')');

            // Atom airplane(apn1)
            // Atom in-city(apt1, cit1)

            // airplane
            std::string type = property.substr(begin + 1, center - begin - 1);

            // apn1
            std::string name = property.substr(center + 1, end - center - 1);

            if (type == "in-city")
            {
                int center = name.find(", ");

                // apt1
                std::string location = name.substr(0, center);
                // cit1
                std::string city = name.substr(center + 2);

                int loc_index = find_object(location);
                assert(loc_index != -1);

                int city_index = find_object(city);
                assert(city_index != -1);

                auto it = std::find_if(city_infos.begin(), city_infos.end(),
                                       [city_index](const CityInfo &info)
                                       { return info.city.index == city_index; });
                if (it == city_infos.end())
                {
                    city_infos.push_back({ObjectLocation{city_index}, {}, -1, -1});
                    it = city_infos.end() - 1;
                }

                if (allObjects[loc_index].has_type(ObjectType::Airport))
                    it->airport = ObjectLocation{loc_index};
                else
                    it->position.push_back(ObjectLocation{loc_index});

                // idx_loc(apt1) -> idx_cit(cit1)
                // is_in_city[ObjectLocation{loc_index}] = ObjectLocation{city_index};
            }
        }

        // Infos from problem file
        for (const auto &pair : variable_id_locations)
        {
            if (allObjects[pair.second].has_type(ObjectType::Truck))
            {
                auto const &values = value_locations[pair.first];
                for (int value_index : values)
                {
                    CityInfo *info = get_city_info_ptr(ObjectLocation{value_index});
                    assert(info != nullptr);
                    info->truck = ObjectLocation{pair.second};
                }
            }
        }
    }

    LandmarkMapper::MapperPair LandmarkMapper::from_fact_pair(const FactPair &fact)
    {
        auto var_it = variable_id_locations.find(fact.var);
        assert(var_it != variable_id_locations.end());

        auto val_it = value_locations.find(fact.var);
        assert(val_it != value_locations.end());
        int var_loc = var_it->second;
        int val_loc = val_it->second[fact.value];

        return {{var_loc}, {val_loc}};
    }

    FactPair LandmarkMapper::to_fact_pair(const MapperPair &mapperPair)
    {
        //utils::g_log << "to_fact_pair(" << mapperPair.key.index << "," << mapperPair.val.index << ")" << std::endl;

        auto var_it = locations_to_variable_id.find(mapperPair.key.index);
        assert(var_it != locations_to_variable_id.end());

        int fact_var = var_it->second;
        //utils::g_log << fact_var << std::endl;
        auto val_it = value_locations.find(fact_var);
        //utils::g_log << "Mapper pair value index >> " << mapperPair.val.index << std::endl;
        int fact_value = -1;
        for (size_t i = 0; i < val_it->second.size(); ++i)
        {
            //utils::g_log << "Val iterator second index " << val_it -> second[i] << std::endl;
            if (val_it->second[i] == mapperPair.val.index)
            {
                fact_value = i;
                break;
            }
        }

        assert(fact_value != -1);

        return FactPair(fact_var, fact_value);
    }

    int LandmarkMapper::to_state_id(ObjectLocation location)
    {
        auto var_it = locations_to_variable_id.find(location.index);
        assert(var_it != locations_to_variable_id.end());

        return var_it->second;
    }

    bool LandmarkMapper::has_value_with_type(ObjectLocation key, ObjectType type)
    {
        auto var_it = locations_to_variable_id.find(key.index);
        if (var_it == locations_to_variable_id.end())
            return false;

        int fact_var = var_it->second;
        auto val_it = value_locations.find(fact_var);

        const auto &values = val_it->second;
        for (size_t i = 0; i < values.size(); ++i)
        {
            if (allObjects[values[i]].has_type(type))
                return true;
        }
        return false;
    }

    const std::string &LandmarkMapper::get_name(ObjectLocation location)
    {
        return allObjects[location.index].name;
    }

    std::string LandmarkMapper::get_name(MapperPair mapperPair)
    {
        return allObjects[mapperPair.key.index].name + ", " + allObjects[mapperPair.val.index].name;
    }

    bool LandmarkMapper::has_type(ObjectLocation location, ObjectType type)
    {
        return allObjects[location.index].has_type(type);
    }

    bool LandmarkMapper::is_vehicle(ObjectLocation location)
    {
        return allObjects[location.index].is_vehicle();
    }

    LandmarkMapper::CityInfo *LandmarkMapper::get_city_info_ptr(ObjectLocation location)
    {
        if (has_type(location, ObjectType::City))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo &info)
                              { return info.city == location; });
            assert(it != city_infos.end());
            return &(*it);
        }
        else if (has_type(location, ObjectType::Airport))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo &info)
                              { return info.airport == location; });
            assert(it != city_infos.end());
            return &(*it);
        }
        else if (has_type(location, ObjectType::Truck))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo &info)
                              { return info.truck == location; });
            assert(it != city_infos.end());
            return &(*it);
        }
        else if (has_type(location, ObjectType::Location))
        {
            auto it = find_if(city_infos.begin(), city_infos.end(),
                              [location](const CityInfo &info)
                              {
                                  for (size_t i = 0; i < info.position.size(); ++i)
                                  {
                                      if (info.position[i] == location)
                                          return true;
                                  }
                                  return false;
                              });
            assert(it != city_infos.end());
            return &(*it);
        }
        else
        {
            assert(false); // CRASH
        }
        return nullptr;
    }

    LandmarkMapper::CityInfo LandmarkMapper::get_city_info(ObjectLocation location)
    {
        LandmarkMapper::CityInfo *optLoc = get_city_info_ptr(location);
        if (optLoc != nullptr)
            return *optLoc;
        return {{-1}, positions, {-1}, {-1}};
    }

    LandmarkFactoryLogistics::LandmarkFactoryLogistics(
        const options::Options &options)
        : LandmarkFactory(options),
          integrated(options.get<bool>("integrated"))
    {
    }

    void LandmarkFactoryLogistics::generate_landmarks(
        const std::shared_ptr<AbstractTask> &task)
    {

        TaskProxy taskProxy(*task);

        if (!initialized)
        {
            initialized = true;
            mapper.init("static-information.out", taskProxy);
            mapper.parse_inCity("static-information.out");

            //print(taskProxy);
        }
        generate_truck_graph(taskProxy);
        calc_achievers(taskProxy);
    }

    void LandmarkFactoryLogistics::calc_achievers(const TaskProxy &task_proxy)
    {
        VariablesProxy variables = task_proxy.get_variables();
        for (auto &lmn : lm_graph->get_nodes())
        {
            for (const FactPair &lm_fact : lmn->facts)
            {
                const std::vector<int> &ops = get_operators_including_eff(lm_fact);
                lmn->possible_achievers.insert(ops.begin(), ops.end());

                if (variables[lm_fact.var].is_derived())
                    lmn->is_derived = true;
            }
        }
    }

    void LandmarkFactoryLogistics::generate_truck_graph(TaskProxy &taskProxy)
    {

        State state = taskProxy.get_initial_state();
        for (FactProxy goal : taskProxy.get_goals())
        {

            if (state[goal.get_variable()].get_value() != goal.get_value())
            {
                LandmarkNode &lmNode = lm_graph->add_simple_landmark(
                    goal.get_pair());
                lmNode.is_true_in_goal = true;
                //Atom at(obj23, pos2)
                //in-city(pos2, cit2)
            }
            LandmarkMapper::MapperPair mapperPair = mapper.from_fact_pair(goal.get_pair());
            goalLocations2.emplace(mapperPair.key, mapperPair);
        }

        for (FactProxy fact : state)
        {

            LandmarkMapper::MapperPair mapperPair = mapper.from_fact_pair(fact.get_pair());
            if (mapper.has_type(mapperPair.key, LandmarkMapper::ObjectType::Package) && mapper.has_type(mapperPair.val, LandmarkMapper::ObjectType::Location))
            {

                if (pack_at_goal(mapperPair))
                {
                    continue;
                }
                add_landmarks_for_location_based(mapperPair, state);
            }

            else if (mapper.has_type(mapperPair.key, LandmarkMapper::ObjectType::Package) && mapper.has_type(mapperPair.val, LandmarkMapper::ObjectType::Truck))
            {
                add_landmarks_for_truck_based(mapperPair, state);
            }
            else if (mapper.has_type(mapperPair.key, LandmarkMapper::ObjectType::Package) && mapper.has_type(mapperPair.val, LandmarkMapper::ObjectType::Airplane))
            {
                add_landmarks_for_airplane_based(mapperPair, state);
            }
        }
        //add_order();

        lm_graph->set_landmark_ids();
    }

    void LandmarkFactoryLogistics::add_landmarks_for_location_based(const LandmarkMapper::MapperPair &mapperPair, const State &state)
    {
        if (pack_at_destination_city(mapperPair))
        {
            //utils::g_log << "Processing: " << mapper.get_name(mapperPair) << std::endl;

            //apt2 oder pos2
            // if true, pack already at destination
            if (pack_at_goal(mapperPair))
            {
                return;
            }
            // if not true -> truck goes to pack.val -> location  --> einladen --> zur goallocation fahren, fertig
            //
            LandmarkMapper::CityInfo cityInfo = mapper.get_city_info(mapperPair.val);                                    // city infos
            LandmarkMapper::ObjectLocation truck = cityInfo.truck;                                                       // truck zur city
            LandmarkMapper::MapperPair truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair()); // position des trucks
            LandmarkMapper::MapperPair to_add2{mapperPair.key, truck};
            LandmarkMapper::MapperPair to_add3{truck, goalLocations2.at(mapperPair.key).val};
            lm_graph->add_simple_landmark(mapper.to_fact_pair(to_add2));                    // einladen
            LandmarkNode &to = lm_graph->add_simple_landmark(mapper.to_fact_pair(to_add3)); // zur goalLocation fahren
            if (mapperPair.val != truckMapper.val)
            {
                LandmarkMapper::MapperPair to_add1{truck, mapperPair.val};
                LandmarkNode &from = lm_graph->add_simple_landmark(mapper.to_fact_pair(to_add1)); // zum paket fahren

                edge_add(from, to, EdgeType::REASONABLE);
            }
        }
        else
        {
            LandmarkMapper::CityInfo cityInfo = mapper.get_city_info(mapperPair.val);

            LandmarkMapper::MapperPair truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(cityInfo.truck)].get_pair());
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, truckMapper.key}));                      // paket einladen
            LandmarkNode &to = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{truckMapper.key, cityInfo.airport})); // zum flughafen fahren
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, cityInfo.airport}));                     // paket abladen
            std::vector<LandmarkMapper::MapperPair> airplanes{};
            mapper.get_by(LandmarkMapper::ObjectType::Airplane,
                          [&](LandmarkMapper::ObjectLocation loc)
                          {
                              airplanes.push_back(mapper.from_fact_pair(state[mapper.to_state_id(loc)].get_pair()));
                          });
            LandmarkMapper::CityInfo otherCity = mapper.get_city_info(goalLocations2.at(mapperPair.key).val);

            std::set<FactPair> plane_at_origin{};
            std::set<FactPair> load_in_plane{};
            std::set<FactPair> plane_at_destination{};
            for (LandmarkMapper::MapperPair airplane : airplanes)
            {
                plane_at_origin.insert(mapper.to_fact_pair(LandmarkMapper::MapperPair{airplane.key, cityInfo.airport}));
                load_in_plane.insert(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, airplane.key}));
                plane_at_destination.insert(mapper.to_fact_pair(LandmarkMapper::MapperPair{airplane.key, otherCity.airport}));
            }

            lm_graph->add_disjunctive_landmark(load_in_plane);                            // in flugzeug laden
            LandmarkNode &to1 = lm_graph->add_disjunctive_landmark(plane_at_destination); // zur destination fliegen

            if (pack_at_goal(LandmarkMapper::MapperPair{mapperPair.key, otherCity.airport}))
            {
                if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper::MapperPair airplane)
                                 { return airplane.val == cityInfo.airport; }))
                {
                    LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                    edge_add(from1, to1, EdgeType::REASONABLE);
                }
                return;
            }
            LandmarkNode &lmTo = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{otherCity.truck, goalLocations2.at(mapperPair.key).val})); // der andere truck zum zielort
            if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper::MapperPair airplane)
                             { return airplane.val == cityInfo.airport; }))
            {
                LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                edge_add(from1, to1, EdgeType::REASONABLE);
                if (integrated)
                    if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                        edge_add(from1, lmTo, EdgeType::REASONABLE);
            }

            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, otherCity.airport}));
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, otherCity.truck}));

            if (otherCity.truck != otherCity.airport)
            {
                LandmarkNode &lmFrom = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{otherCity.truck, otherCity.airport}));
                edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
            }

            if (truckMapper.val != mapperPair.val)
            {
                LandmarkNode &from = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{truckMapper.key, mapperPair.val})); //zum paket fahren

                edge_add(from, to, EdgeType::REASONABLE); // kante zuerst beim paket -> beim flughafen

                if (integrated)
                {
                    if (mapperPair.val != cityInfo.airport)
                    {

                        edge_add(from, to1, EdgeType::REASONABLE); // if pos(p) not airport and truck not there then, truck at pos before airplane in dest city
                        if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                        {
                            edge_add(to, lmTo, EdgeType::REASONABLE);   // if pos(p) nor dest(p) an airport and truck not there, then truck at pos of pack before truck at goal
                            edge_add(from, lmTo, EdgeType::REASONABLE); // if pos(p) no truck and dest not in airport nor pos(p) in airport, then truck at pos before truck in dest
                        }
                        edge_add(to, to1, EdgeType::REASONABLE); // if pos(p) not an airport truck has to drive to airport before airplane in destination city
                        if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                        {
                            edge_add(to1, lmTo, EdgeType::REASONABLE);
                        }
                    }
                }
            }
            else
            {

                if (integrated)
                {
                    if (mapperPair.val != cityInfo.airport)
                    {
                        if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                        {
                            edge_add(to, lmTo, EdgeType::REASONABLE);
                        }
                        edge_add(to, to1, EdgeType::REASONABLE);
                    }
                    if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                    {
                        edge_add(to1, lmTo, EdgeType::REASONABLE);
                    }
                }
            }
        }
    }

    void LandmarkFactoryLogistics::add_landmarks_for_truck_based(const LandmarkMapper::MapperPair &mapperPair, const State &state)
    {
        LandmarkMapper::ObjectLocation truck = mapperPair.val; // truck zur city

        if (pack_at_destination_city(mapperPair))
        {
            LandmarkMapper::MapperPair truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair());
            // utils::g_log << "Processing: " << mapper.get_name(truckMapper) << std::endl;
            if (pack_at_goal(LandmarkMapper::MapperPair{mapperPair.key, truckMapper.val}))
            {
                return;
            }
            else
            {
                lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{truck, goalLocations2.at(mapperPair.key).val}));
            }
        }
        else
        {
            // LandmarkMapper::MapperPair truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair());
            LandmarkMapper::CityInfo cityInfo = mapper.get_city_info(truck);
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, cityInfo.airport}));
            std::vector<LandmarkMapper::MapperPair> airplanes{};
            mapper.get_by(LandmarkMapper::ObjectType::Airplane,
                          [&](LandmarkMapper::ObjectLocation loc)
                          {
                              airplanes.push_back(mapper.from_fact_pair(state[mapper.to_state_id(loc)].get_pair()));
                          });
            LandmarkMapper::CityInfo otherCity = mapper.get_city_info(goalLocations2.at(mapperPair.key).val);

            std::set<FactPair> plane_at_origin{};
            std::set<FactPair> load_in_plane{};
            std::set<FactPair> plane_at_destination{};
            for (LandmarkMapper::MapperPair airplane : airplanes)
            {
                plane_at_origin.insert(mapper.to_fact_pair(LandmarkMapper::MapperPair{airplane.key, cityInfo.airport}));
                load_in_plane.insert(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, airplane.key}));
                plane_at_destination.insert(mapper.to_fact_pair(LandmarkMapper::MapperPair{airplane.key, otherCity.airport}));
            }

            lm_graph->add_disjunctive_landmark(load_in_plane);
            LandmarkNode &to1 = lm_graph->add_disjunctive_landmark(plane_at_destination);

            if (pack_at_goal(LandmarkMapper::MapperPair{mapperPair.key, otherCity.airport}))
            {
                if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper::MapperPair airplane)
                                 { return airplane.val == cityInfo.airport; }))
                {
                    LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                    edge_add(from1, to1, EdgeType::REASONABLE);
                }
                return;
            }

            LandmarkNode &to = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{otherCity.truck, goalLocations2.at(mapperPair.key).val}));
            if (std::none_of(airplanes.begin(), airplanes.end(), [&](LandmarkMapper::MapperPair airplane)
                             { return airplane.val == cityInfo.airport; }))
            {
                LandmarkNode &from1 = lm_graph->add_disjunctive_landmark(plane_at_origin);
                edge_add(from1, to1, EdgeType::REASONABLE);
                if (integrated)
                {
                    edge_add(from1, to, EdgeType::REASONABLE);
                }
            }

            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, otherCity.truck}));
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, otherCity.airport}));

            if (otherCity.truck != otherCity.airport)
            {
                LandmarkNode &from = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{otherCity.truck, otherCity.airport}));
                edge_add(from, to, EdgeType::REASONABLE);
            }
            if (truck != cityInfo.airport)
            {

                LandmarkNode &truckAtAirport = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{truck, cityInfo.airport})); //truck drives to airport
                if (integrated)
                {

                    if (cityInfo.truck != cityInfo.airport)
                    {
                        if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                        {
                            edge_add(truckAtAirport, to, EdgeType::REASONABLE);
                        }
                        edge_add(truckAtAirport, to1, EdgeType::REASONABLE);
                    }
                    if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                    {
                        edge_add(to1, to, EdgeType::REASONABLE);
                    }
                }
            }
            else
            {

                if (integrated)
                {
                    if (goalLocations2.at(mapperPair.key).val != otherCity.airport)
                    {
                        edge_add(to1, to, EdgeType::REASONABLE);
                    }
                }
            }
        }
    }

    void LandmarkFactoryLogistics::add_landmarks_for_airplane_based(const LandmarkMapper::MapperPair &mapperPair, const State &state)
    {
        LandmarkMapper::MapperPair airplane = mapper.from_fact_pair(state[mapper.to_state_id(mapperPair.val)].get_pair());
        LandmarkMapper::MapperPair airplaneVal{mapperPair.key, airplane.val};

        if (pack_at_destination_city(airplaneVal))
        {

            //utils::g_log << "Does it crash here?" << std::endl;

            if (pack_at_goal(airplaneVal))
            {
                return;
            }
            LandmarkMapper::CityInfo cityInfo = mapper.get_city_info(airplane.val);
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, cityInfo.airport})); //ausladen
            // city infos
            LandmarkMapper::ObjectLocation truck = cityInfo.truck;                                                       // truck zur city
            LandmarkMapper::MapperPair truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(truck)].get_pair()); // position des trucks
            LandmarkMapper::MapperPair to_add2{mapperPair.key, truck};
            LandmarkMapper::MapperPair to_add3{truck, goalLocations2.at(mapperPair.key).val};
            lm_graph->add_simple_landmark(mapper.to_fact_pair(to_add2)); // einladen

            LandmarkNode &lmTo = lm_graph->add_simple_landmark(mapper.to_fact_pair(to_add3)); // zur goalLocation fahren
            if (cityInfo.airport != truckMapper.val)
            {
                LandmarkMapper::MapperPair to_add1{truck, cityInfo.airport};

                LandmarkNode &lmFrom = lm_graph->add_simple_landmark(mapper.to_fact_pair(to_add1)); // zum paket fahren

                edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
            }

            
        }
        else
        {
            LandmarkMapper::CityInfo cityInfo = mapper.get_city_info(goalLocations2.at(mapperPair.key).val);
            /*utils::g_log << "Crash key: " << toAdd1.key.index << " crash val: " << toAdd1.val.index << std::endl;
                    utils::g_log << "Airplane key: " << airplane.key.index << " crash val: " << cityInfo.airport.index << std::endl;*/

            LandmarkNode &lmAirport = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{airplane.key, cityInfo.airport}));

            LandmarkMapper::MapperPair truckMapper = mapper.from_fact_pair(state[mapper.to_state_id(cityInfo.truck)].get_pair());

            if (pack_at_goal(LandmarkMapper::MapperPair{mapperPair.key, cityInfo.airport}))
            {
                return;
            }
            LandmarkNode &lmTo = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{cityInfo.truck, goalLocations2.at(mapperPair.key).val}));

            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, cityInfo.airport}));
            lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{mapperPair.key, cityInfo.truck}));

            if (truckMapper.val != cityInfo.airport)
            {
                LandmarkNode &lmFrom = lm_graph->add_simple_landmark(mapper.to_fact_pair(LandmarkMapper::MapperPair{cityInfo.truck, cityInfo.airport}));
                edge_add(lmFrom, lmTo, EdgeType::REASONABLE);
            }

            if (integrated)
            {
                if (goalLocations2.at(mapperPair.key).val != cityInfo.airport)
                {
                    edge_add(lmAirport, lmTo, EdgeType::REASONABLE);
                }
            }
        }
    }

    bool LandmarkFactoryLogistics::supports_conditional_effects() const
    {
        return true;
    }

    void LandmarkFactoryLogistics::print(const TaskProxy &taskProxy)
    {

        for (FactProxy allFacts : taskProxy.get_variables().get_facts())
        {
            LandmarkMapper::MapperPair mapperPair = mapper.from_fact_pair(allFacts.get_pair());
            utils::g_log << "Key: " << mapperPair.key.index << " value: " << mapperPair.val.index << std::endl;
            utils::g_log << "Name: " << mapper.get_name(mapperPair) << std::endl;
        }
    }

    static std::shared_ptr<LandmarkFactory> _parse(OptionParser &parser)
    {
        parser.document_synopsis(
            "Logistics Landmarks",
            "Only for logistics task checks for each fact if it is a landmark."
            "This check is for Logistics Task");
        parser.document_note(
            "No relevant options right now",
            "reasonable_ordersddddddddddd, only_causal_landmarks");
        _add_options_to_parser(parser);
        parser.add_option<bool>("integrated",
                                "use integrated logistics landmark",
                                "false");

        Options opts = parser.parse();

        if (parser.dry_run())
            return nullptr;
        else
            return std::make_shared<LandmarkFactoryLogistics>(opts);
    }

    static Plugin<LandmarkFactory> _plugin("lm_logistics", _parse);
}
