#include "landmark_constraint_handler.h"

#include "disj_action_landmark_graph.h"

#include "../landmarks/landmark_factory.h"
#include "../landmarks/util.h"
#include "../task_utils/task_properties.h"
#include "../utils/logging.h"
#include "../landmarks/landmark_factory_logistics.h"
#include "unistd.h"

using namespace std;

namespace landmarks {
long LandmarkConstraintHandler::num_cycles = 0;

LandmarkConstraintHandler &LandmarkConstraintHandler::get_instance() {
    static LandmarkConstraintHandler instance;
    return instance;
}

void LandmarkConstraintHandler::initialize(
    const options::Options &opts,
    const std::shared_ptr<AbstractTask> &original_task) {

    if (!initialized) {
        lm = opts.get<shared_ptr<LandmarkFactory>>("lm");
        task = original_task;
        task_proxy = make_shared<TaskProxy>(*task);
        task_properties::verify_no_axioms(*task_proxy);
        task_properties::verify_no_conditional_effects(*task_proxy);
        initialized = true;
    }
}

shared_ptr<DisjActionLandmarkGraph> LandmarkConstraintHandler::get_lm_graph(
    const State &state) {

    state.unpack();
    if (state.get_id() != last_state_id) {
        compute_lm_graph(state);
    }
    return dalm_graph;
}

void LandmarkConstraintHandler::compute_lm_graph(const State &state) {
    assert(state.get_id() != last_state_id);

    if (last_state_id == StateID::no_state) {
        LandmarkGraph &lm_graph = *lm->compute_lm_graph(task);
        landmarks::dump_landmark_graph(*task_proxy, lm_graph);
        dalm_graph = make_shared<DisjActionLandmarkGraph>(
            lm_graph, state);

        num_landmarks = dalm_graph->get_number_of_landmarks();
        num_orderings = dalm_graph->get_number_of_orderings();
        num_weak_orderings = dalm_graph-> get_number_of_weak_orderings();
        num_lm_graphs = 1;

        utils::g_log << "Landmark graph contains " << num_landmarks << " landmarks." << endl;
        utils::g_log << "Landmark graph contains " << num_orderings
                     << " orderings of which " << dalm_graph->get_number_of_strong_orderings()
                     << " are strong and " << num_weak_orderings
                     << " are weak." << endl;
    } else {
        LandmarkGraph &lm_graph= *lm->recompute_lm_graph(state);
        dalm_graph = make_shared<DisjActionLandmarkGraph>(
            lm_graph,state);
        
        // landmarks::dump_landmark_graph(*task_proxy,lm_graph);

        num_landmarks += dalm_graph->get_number_of_landmarks();
        num_orderings += dalm_graph->get_number_of_orderings();
        num_weak_orderings += dalm_graph-> get_number_of_weak_orderings();
        ++num_lm_graphs;
    }
    last_state_id = state.get_id();
}

void LandmarkConstraintHandler::print_statistics() const {
    cout << "Number of computed LM Graphs: " << num_lm_graphs << endl;
    cout << "Average number of landmarks: " << (static_cast<double>(num_landmarks) / num_lm_graphs) << endl;
    cout << "Average number of orderings: " << (static_cast<double>(num_orderings) / num_lm_graphs) << endl;
    cout << "Average number of weak orderings: " << (static_cast<double>(num_weak_orderings) / num_lm_graphs) << endl;
    cout << "Average number of strong orderings: " << (static_cast<double>(num_orderings - num_weak_orderings) / num_lm_graphs) << endl;

    cout << "Average number of cycle constraints: " << (static_cast<double>(num_cycles) / num_lm_graphs) << endl;
}
}
