import java.io.File;
import java.io.FileNotFoundException; 
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Scanner;

public class Graph {
  private HashMap<String, Vertex> vertices;
  
  Graph(String filename) {
    vertices = new HashMap<String, Vertex>();
    try {
      File inputFile = new File(filename);
      Scanner scanner = new Scanner(inputFile);
      String[] verticesArray = scanner.nextLine().split(" ");
      for (String vertexName : verticesArray) {
        addVertex(vertexName);
      }
      
      Integer edgeCount = 0;
      while(scanner.hasNextLine()) {
        String[] edge = scanner.nextLine().split(" ");
        assert edge.length == 2;
        addEdge(edge[0], edge[1]);
        edgeCount++;
      }
      scanner.close();
      
      // Check if the graph is a tree by testing if it has |V|-1 edges and is 
      // connected.
      if (edgeCount != verticesArray.length-1) {
        throw new IllegalArgumentException("The graph is not a tree since it does not have |V|-1 edges.");
      }
      LinkedList<String> queue = new LinkedList<String>();
      HashSet<String> closed = new HashSet<String>();
      queue.add(verticesArray[0]);
      while(!queue.isEmpty()) {
        String vertexName = queue.pop();
        closed.add(vertexName);
        for (Vertex neighbour : getVertex(vertexName).getNeighbours()) {
          String neighbourName = neighbour.getName();
          if (!closed.contains(neighbourName)) {
            closed.add(neighbourName);
            queue.add(neighbourName);
          }
        }
      }
      if (closed.size() != verticesArray.length) {
        throw new IllegalArgumentException("The graph is not a tree since it is not connected.");
      }
      
    } catch (FileNotFoundException e) {
      System.out.println("Could not find file " + filename);
      System.exit(1);
    } catch (IllegalArgumentException e) {
      System.out.println(e.getMessage());
      System.exit(1);
    }
  }
  
  public void addVertex(String name) {
    vertices.put(name, new Vertex(name));
  }
  
  public void addEdge(String name1, String name2) {
    Vertex v1 = vertices.get(name1);
    Vertex v2 = vertices.get(name2);
    
    if (!vertices.containsKey(name1)) {
      throw new IllegalArgumentException("The graph has an edge involving vertex " + name1 + ", but this vertex does not exist.");
    } else if (!vertices.containsKey(name2)) {
      throw new IllegalArgumentException("The graph has an edge involving vertex " + name2 + ", but this vertex does not exist.");
    }
    
    v1.addNeighbour(v2);
    v2.addNeighbour(v1);
  }
  
  public Vertex getVertex(String name) {
    return vertices.get(name);
  }
  
  public HashMap<String, Vertex> getVertices() {
    return vertices;
  }
  
  public Integer get_size() {
    return vertices.size();
  }
}
