package logic.proofs;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import logic.formulas.Formula;
import logic.proofs.rules.InferenceRule;

/**
 * Checks proofs for logical inference in propositional calculus for validity.
 */
public class ProofChecker {
    /**
     * A proof is correct if all of its steps are correct.
     * While checking each step, we collect the assumptions that are made along the way.
     * @param steps list of proof steps to check
     * @param assumptions list of assumptions that are made in the proof (filled while checking the proof).
     * @return True if every step in the derivation is valid.
     */
    public static boolean checkProof(List<ProofStep> steps, List<Formula> assumptions) {
        assumptions.clear();
        for (int i = 0; i < steps.size(); i++) {
            ProofStep step = steps.get(i);
            if (!checkProofStep(i, step, assumptions))
                return false;
        }
        return true;
    }

    /**
     * A step in the proof asserts that a formula can be derived as an assumption or by applying a rule
     * to previously derived formulas.
     * Deriving a formula as an assumption is always valid but adds a new assumption to the list of
     * assumptions we made so far.
     * Deriving a formula with a rule is valid if the rule can be applied to its inputs and results in
     * the formula that should be derived.
     * @param stepId current step in the proof (used for error messages to identify the invalid step)
     * @param step the proof step to verify.
     * @param assumptions list of assumptions made so far. If the step derives a formula as
     * an assumption, the new assumption is added to this list.
     * @return true iff the step is valid.
     */
    public static boolean checkProofStep(int stepId, ProofStep step, List<Formula> assumptions) {
        InferenceRule rule = step.getRule();
        Formula[] conditions = step.getRuleInputs();
        Formula result = step.getFormula();
        if (rule == null) {
            assumptions.add(result);
        } else {
            HashMap<String, Formula> replacements = new HashMap<String, Formula>();
            Formula[] conditionPatterns = rule.getConditions();
            Formula resultPattern = rule.getResult();
            if (conditions.length != conditionPatterns.length) {
                System.err.println("Rule " + rule.getName() + " used in step " + stepId +
                                   " has " + conditionPatterns.length + " conditions " +
                                   "but " + conditions.length + " are given.");
                PrettyPrinter.reportMatchFailure(conditions, result,
                                                 conditionPatterns, resultPattern,
                                                 replacements);
                return false;
            }
            for (int i = 0; i < conditionPatterns.length; i++) {
                Formula conditionPattern = conditionPatterns[i];
                Formula condition = conditions[i];
                if (!conditionPattern.match(condition, replacements)) {
                    System.err.println("Could not match condition " + (i+1) +
                                       " of rule " + rule.getName() + " in step " + stepId + ".");
                    PrettyPrinter.reportMatchFailure(conditions, result,
                                                     conditionPatterns, resultPattern,
                                                     replacements);
                    return false;
                }
            }
            if (!resultPattern.match(result, replacements)) {
                System.err.println("Could not match result of rule " + rule.getName() +
                                   " in step " + stepId + ".");
                PrettyPrinter.reportMatchFailure(conditions, result,
                                                 conditionPatterns, resultPattern,
                                                 replacements);
                return false;
            }
        }
        return true;
    }

    /**
     * Command line interface: call with a path to a file that contains a proof as the only argument
     * to check the validity of the proof. Prints the assumptions and derived formula if the derivation is valid
     * or an error message if it isn't. See documentation in logic.proofs.Parser.java for the file format. 
     * @param args should contain the path to a proof file.
     */
    public static void main(String[] args) {
        if (args.length != 1) {
            System.out.println("Usage: java -jar ProofChecker.jar <path-to-proof-file>");
            System.exit(1);
        }
        List<ProofStep> proofSteps = Parser.parseFile(args[0]);
        if (proofSteps == null) {
            System.err.println("Parsing failed.");
        } else {
            List<Formula> assumptions = new ArrayList<Formula>();
            if (checkProof(proofSteps, assumptions)) {
                System.out.println("Proof is valid and shows that");
                Formula lastFormula = proofSteps.get(proofSteps.size() - 1).getFormula();
                System.out.println("    " + lastFormula);
                if (assumptions.isEmpty()) {
                    System.out.println("is a tautology.");
                } else {
                    System.out.println("follows logically from the following assumptions:");
                    for (Formula assumption : assumptions)
                        System.out.println("    " + assumption);
                }
            } else {
                System.err.println("Proof invalid.");
            }
        }
    }

}
