package logic.formulas;

import java.util.HashMap;

/** 
 * A negation is a formula expressing that something is not the case.
 * It is written without parentheses and a "~" in front, e.g., "~phi".
 */
public class Negation extends Formula {
    // The subformula that is negated by this formula (phi in the example above)
    private Formula negatedFormula;

    public Negation(Formula negatedFormula) {
        this.negatedFormula = negatedFormula;
    }

    public Formula getNegatedFormula() {
        return negatedFormula;
    }

    @Override
    public String toString() {
        return "~" + negatedFormula;
    }

    /**
     * A formula can only be matched with a negation if it also is a negation and
     * the subformulas can be matched.
     * For example, ~(A /\ B) can be matched with ~phi iff (A /\ B) can be matched with phi.
     */
    @Override
    public boolean match(Formula formula, HashMap<String, Formula> metaVariables) {
        if (!(formula instanceof Negation))
            return false;
        Negation negation = (Negation) formula;
        return negatedFormula.match(negation.getNegatedFormula(), metaVariables);
    }
}
