package logic.formulas;

import java.util.HashMap;

public abstract class Formula {

    /*
     * NOTE:
     * Equality on formulas is equality on the syntactic level.
     * To emphasize this, we implement hashing and equality as
     * functions of the string representation. In general this
     * is not good practice because the string concatenation used
     * in toString() makes the code unnecessarily slow.
     */

    /**
     * hashCode() must be implemented if equals() has a custom implementation.
     * Here we just delegate the calculation of the hash code to the String
     * class and return the string representation's hash code.
     */
    @Override
    public int hashCode() {
        String s = this.toString();
        return s == null ? 0 : s.hashCode();
    }

    /**
     * Two formulas are equal iff their string representation is the same.
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (!(obj instanceof Formula))
            return false;
        String s1 = this.toString();
        String s2 = obj.toString();
        if (s1 == null || s2 == null)
            return false;
        return s1.equals(s2);
    }

    /**
     * Match a given formula with this formula.
     * To match a formula f1 with a formula f2, there must be some assignment of
     * the atom names in f2 (called meta variables here) to (sub)formulas in f1.
     * 
     * For example, matching ((A /\ B) \/ ~C) with (phi \/ ~psi) is possible by
     * mapping phi to (A /\ B) and psi to C.
     * 
     * Matching ((A /\ B) \/ ~C) with (phi /\ ~psi) is not possible, because
     * the first formula is a disjunction and the second one is a conjunction.
     * 
     * Matching ((A /\ B) \/ ~C) with ((phi /\ psi) \/ ~psi) is not possible,
     * because psi cannot be mapped to B and C at the same time.
     * 
     * @param formula the formula we want to match.
     * @param metaVariables the assignment of atom names in the given formula
     * to (sub)formulas of this formula. Created during the call to unify.
     * @return whether it is possible to match the given formula with this formula.
     */
    public abstract boolean match(Formula formula, HashMap<String, Formula> metaVariables);
}
