package logic.formulas;

import java.util.HashMap;

/*
 * NOTE:
 * We could have implemented all (binary) formulas with inheritance
 * without so much duplicated code. While this would be a better design,
 * we hope that this more specific approach is easier to understand.
 */

/** 
 * A disjunction is a formula expressing that at least one of two subformulas is true.
 * It is written with parentheses and a "\/" between the subformulas,
 * e.g., "(phi \/ psi)".
 */
public class Disjunction extends Formula {
    // Left-hand side (phi in the example above)
    private Formula lhs;
    // Right-hand side (psi in the example above)
    private Formula rhs;

    public Disjunction(Formula lhs, Formula rhs) {
        this.lhs = lhs;
        this.rhs = rhs;
    }

    public Formula getLhs() {
        return lhs;
    }

    public Formula getRhs() {
        return rhs;
    }

    @Override
    public String toString() {
        return "(" + lhs + " \\/ " + rhs + ")";
    }

    /**
     * A formula can only be matched with a disjunction if it also is a disjunction and
     * the respective subformulas can be matched.
     * For example, ((A /\ B) \/ C) can be matched with (phi \/ psi) iff
     * (A /\ B) can be matched with phi and C can be matched with psi.
     */
    @Override
    public boolean match(Formula formula, HashMap<String, Formula> metaVariables) {
        if (!(formula instanceof Disjunction))
            return false;
        Disjunction disjunction = (Disjunction) formula;
        return lhs.match(disjunction.getLhs(), metaVariables) &&
               rhs.match(disjunction.getRhs(), metaVariables);
    }
}
