package logic.proofs.rules;

import java.util.HashMap;

import logic.formulas.Formula;
import logic.formulas.Parser;
import logic.formulas.Parser.ParseError;

/**
 * A calculus for propositional logic is a set of named inference rules.
 */
public class Calculus {
    // Set of rules in the calculus indexed by their names.
    static HashMap<String, InferenceRule> rules =
        new HashMap<String, InferenceRule>();

    // We parse all rules from strings because this makes it easier to specify them.
    static {
        // Basic rules
        addRule("ModusPonens",                     "X, (X -> Y) |- Y");
        addRule("ModusTollens",                   "~Y, (X -> Y) |- ~X");
        addRule("AndElimLeft",                       "(X /\\ Y) |- Y");
        addRule("AndElimRight",                      "(X /\\ Y) |- X");
        addRule("AndIntro",                               "X, Y |- (X /\\ Y)");
        addRule("OrIntroLeft",                               "X |- (X \\/ Y)");
        addRule("OrIntroRight",                              "Y |- (X \\/ Y)");
        // Rules to introduce or remove abbreviations
        addRule("BiimplicationIntro",       "(X -> Y), (Y -> X) |- (X <-> Y)");
        addRule("BiimplicationElimLeft",             "(X <-> Y) |- (Y -> X)");
        addRule("BiimplicationElimRight",            "(X <-> Y) |- (X -> Y)");
        addRule("ImplicationIntro",                 "(~X \\/ Y) |- (X -> Y)");
        addRule("ImplicationElim",                    "(X -> Y) |- (~X \\/ Y)");
        // Rules derived from equivalences
        addRule("DeMorgan1LeftToRight",             "~(X \\/ Y) |- (~X /\\ ~Y)");
        addRule("DeMorgan1RightToLeft",            "(~X /\\ ~Y) |- ~(X \\/ Y)");
        addRule("DeMorgan2LeftToRight",             "~(X /\\ Y) |- (~X \\/ ~Y)");
        addRule("DeMorgan2RightToLeft",            "(~X \\/ ~Y) |- ~(X /\\ Y)");
    }

    /**
     * Add a rule to the calculus.
     * @param name Name that is used to identify the rule in proofs.
     * @param ruleString Rule definition in the format "<conditions> |- <result>"
     * Where <conditions> is a comma-separated list of formula strings and result
     * is a formula string. Formula strings have the format described in logic.formulas.*
     */
    static void addRule(String name, String ruleString) {
        // Split the rule into a list of conditions and a result.
        String[] ruleComponents = ruleString.split("\\|-");
        String[] conditionStrings = ruleComponents[0].trim().split(",");
        String resultString = ruleComponents[1].trim();
        try {
            // Parse the result and all conditions of the rule.
            Formula result = Parser.parse(resultString);
            Formula[] conditions = new Formula[conditionStrings.length];
            for (int i = 0; i < conditionStrings.length; i++) {
                conditions[i] = Parser.parse(conditionStrings[i]);
            }
            // Add the new rule to the calculus.
            rules.put(name, new InferenceRule(name, conditions, result));
        } catch (ParseError e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    public static InferenceRule getRule(String name) {
        return rules.get(name);
    }
}
