package logic.proofs;

import java.util.HashMap;
import java.util.TreeMap;
import java.util.Map.Entry;

import logic.formulas.Formula;

/**
 * Formated output for some cases where the proof is not valid.
 */
public class PrettyPrinter {

    /**
     * We want to print the left and right hand side of a rule and the
     * formulas the rule is matched with and align corresponding formulas.
     * We also want to print the assignment of meta variables that are
     * already assigned.
     * @param conditions list of formulas that are used as rule inputs
     * @param result formula that should be derived
     * @param conditionPatterns list of patterns that should match conditions
     * @param resultPattern pattern that should match result
     * @param replacements assignment to meta variables matched so far
     */
    static void reportMatchFailure(Formula[] conditions, Formula result,
                                   Formula[] conditionPatterns, Formula resultPattern,
                                   HashMap<String, Formula> replacements) {
        // We first put strings in a table that is later printed with aligned columns
        // In the table, we have one column for each condition, one column for the result
        // and four columns for symbols between columns:
        // "Formulas: ", "{", condition[0],         ..., condition[n],         "}", " |-/- ", result
        // "Patterns: ", "{", conditionPatterns[0], ..., conditionPatterns[n], "}",  " |- ",  resultPattern
        String[][] table = new String[conditions.length + 5][2];
        table[0][0] = "  Formulas: ";
        table[1][0] = "{";
        for (int i = 0; i < conditions.length; i++) {
            table[i+2][0] = conditions[i] + (i < conditions.length - 1? ", " : ""); 
        }
        table[conditions.length + 2][0] = "}";
        table[conditions.length + 3][0] = " |-/- ";
        table[conditions.length + 4][0] = result.toString();

        table[0][1] = "  Patterns: ";
        table[1][1] = "{";
        for (int i = 0; i < conditionPatterns.length; i++) {
            table[i+2][1] = conditionPatterns[i] + (i < conditionPatterns.length - 1? ", " : ""); 
        }
        table[conditionPatterns.length + 2][1] = "}";
        table[conditionPatterns.length + 3][1] = " |- ";
        table[conditionPatterns.length + 4][1] = resultPattern.toString();

        // Print the table aligned by columns.
        printAligned(table);

        // Print the replacement made so far.
        if (!replacements.isEmpty()) {
            // Sort entries by name
            TreeMap<String, Formula> sortedReplacements = new TreeMap<String, Formula>(replacements);
            System.err.println("  Replacements so far:");
            for(Entry<String, Formula> entry : sortedReplacements.entrySet()) {
                System.err.println("    " + entry.getKey() + ": " + entry.getValue());
            }
        }
    }

    /**
     * Print a 2D array of strings such that columns are aligned.
     * @param table 2D array (first index: columns, second index: rows)
     */
    private static void printAligned(String[][] table) {
        // Calculate the length needed to print each column.
        int[] columnLengths = new int[table.length];
        for (int x = 0; x < table.length; x++) {
            String[] column = table[x];
            for (int y = 0; y < column.length; y++) {
                columnLengths[x] = Math.max(columnLengths[x], column[y].length());
            }
        }

        // Print each row using the length columnLengths[x] for each column x.
        for (int y = 0; y < table[0].length; y++) {
            for (int x = 0; x < table.length; x++) {
                System.err.printf("%-" + columnLengths[x] + "s", table[x][y]);
            }
            System.err.println();
        }
    }
}
