package logic.proofs;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import logic.formulas.Formula;
import logic.formulas.Parser.ParseError;
import logic.proofs.rules.Calculus;
import logic.proofs.rules.InferenceRule;

/**
 * Parser for proof files.
 * The proof file must have one proof step per line.
 * Each step must have the following format:
 * <Formula> | <Rulename> | <RuleInputs>
 * 
 * <Formula> must be a valid formula (see parser for logic formulas for format)
 * <Rulename> must be either "Assumption" or a name of the rule in the calculus
 *            (see Calculus.java)
 * <RuleInputs> is a comma separated list of line numbers whose formula should
 *              be used as as input for the rule.
 *
 * Example:
 * A                | Assumption      |
 * B                | Assumption      |
 * (A /\ B)         | AndIntro        | 1, 2
 * (~C \/ (A /\ B)) | OrIntroRight    | 3
 * (C -> (A /\ B))  | ImplicationIntro| 4
 *
 * Explanation of example:
 * Line 1 adds A as an assumption, no rule inputs are required.
 * Line 2 adds B as an assumption.
 * Line 3 derives the formula (A /\ B) from the two assumptions.
 *     This derivation uses the rule AndIntro (phi, psi |- (phi /\ psi))
 *     which has two parameters (phi and psi). We match phi with A (line 1)
 *     and psi with B (line 2).
 * Line 4 derives the formula (~C \/ (A /\ B)) from (A /\ B) in line 3
 *     with the rule OrIntroRight (phi |- (psi \\/ phi)). This rule only
 *     has one parameter on the left side (phi). The other parameter is
 *     automatically matched with ~C.
 * Line 5 applies ImplicationIntro to line 4 and derives (C -> (A /\ B)).
 * 
 * All in all the proof shows that (C -> (A /\ B)) can be derived from the
 * assumptions A and B.
 */
public class Parser {

    /**
     * Parse a file containing one proof (see above for format)
     * @param filename path to the file containing the proof
     * @return List of proof steps or null if the file cannot be parsed.
     */
    public static List<ProofStep> parseFile(String filename) {
        ArrayList<ProofStep> result = new ArrayList<ProofStep>();
        try {
            // Read in the file
            BufferedReader br = new BufferedReader(new FileReader(filename));
            try {
                // Read the file line by line and parse one proof step per line
                String line;
                while ((line = br.readLine()) != null) {
                    result.add(parseProofStep(line, result));
                }
            } catch (logic.formulas.Parser.ParseError e) {
                e.printStackTrace();
                System.err.println(e.getMessage());
                return null;
            } finally {
                br.close();
            }
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
        return result;
    }

    /**
     * Parse one line from a proof file (see above for format).
     * @param line one line from a proof file (see above for format).
     * @param stepsSoFar previously parsed steps. numbers in rule input lists refer to formulas in this list
     * @return the next step in the proof
     * @throws ParseError if the line has invalid format or uses a line in the proof that does not exist yet.
     */
    private static ProofStep parseProofStep(String line, ArrayList<ProofStep> stepsSoFar) throws ParseError {
        // Split the line into formula, rule name and rule inputs
        String[] tokens = line.split("\\|", -1);
        if (tokens.length != 3) {
            throw new ParseError(
                "Unexpected format in proof file.\n" +
                "Expected '<Formula> | <Rulename> | <RuleInputs>' but got '" + line +"'.");
        }
        Formula formula = logic.formulas.Parser.parse(tokens[0].trim());
        String ruleName = tokens[1].trim();
        String ruleInputString = tokens[2].trim();

        // Parse rule inputs and fetch rule from calculus.
        InferenceRule rule = null;
        Formula[] ruleInputs = null;
        // Assumptions are no inference rules, we leave the rule empty in this case.
        // Assumptions also do not use rule inputs.
        if (!ruleName.toLowerCase().equals("assumption")) {
            // Fetch the rule from the calculus
            rule = Calculus.getRule(ruleName);
            if (rule == null) {
                throw new ParseError("Failed to create rule with name '" + ruleName + "'");
            }
            // Parse the rule inputs.
            int[] ruleInputIndices = parseIntArray(ruleInputString);
            // Get the corresponding formula for each index.
            ruleInputs = new Formula[ruleInputIndices.length];
            for (int i = 0; i < ruleInputIndices.length; i++) {
                // Subtract 1 because internal indices start at 0.
                int ruleInputIndex = ruleInputIndices[i] - 1;
                if (ruleInputIndex >= stepsSoFar.size()) {
                    throw new ParseError(
                        "Cannot use index " + ruleInputIndex + " as input " +
                        "for rule in step " + (stepsSoFar.size() + 1) + ". " +
                        "Only indices from previous steps can be used.");
                }
                ruleInputs[i] = stepsSoFar.get(ruleInputIndex).getFormula();
            }
        }
        return new ProofStep(formula, rule, ruleInputs);
    }

    /**
     * Helper function to parse the rule inputs.
     * @param input comma/semicolon separated list of numbers as a string
     * @return integer array containing the numbers
     */
    private static int[] parseIntArray(String input) {
        // Split the string into a list of strings, each containing one number.
        String[] numbers = input.split("[ ,;]+");
        // Parse one number from each string in the list.
        int[] result = new int[numbers.length];
        for (int i = 0; i < numbers.length; i++) {
            String number = numbers[i].trim();
            result[i] = Integer.parseInt(number);
        }
        return result;
    }
}
