package logic.formulas;

import java.util.HashMap;

/*
 * NOTE:
 * We could have implemented all (binary) formulas with inheritance
 * without so much duplicated code. While this would be a better design,
 * we hope that this more specific approach is easier to understand.
 */

/** 
 * An implication is a formula with two subformulas expressing that
 * if the left subformula is true then the right subformula is true as well.
 * It is written with parentheses and a "->" between the subformulas,
 * e.g., "(phi -> psi)".
 */
public class Implication extends Formula {
    // Left-hand side (phi in the example above)
    private Formula lhs;
    // Right-hand side (psi in the example above)
    private Formula rhs;

    public Implication(Formula lhs, Formula rhs) {
        this.lhs = lhs;
        this.rhs = rhs;
    }

    public Formula getLhs() {
        return lhs;
    }

    public Formula getRhs() {
        return rhs;
    }

    @Override
    public String toString() {
        return "(" + lhs + " -> " + rhs + ")";
    }

    /**
     * A formula can only be matched with an implication if it also is an implication and
     * the respective subformulas can be matched.
     * For example, ((A /\ B) -> C) can be matched with (phi -> psi) iff
     * (A /\ B) can be matched with phi and C can be matched with psi.
     */
    @Override
    public boolean match(Formula formula, HashMap<String, Formula> metaVariables) {
        if (!(formula instanceof Implication))
            return false;
        Implication implication = (Implication) formula;
        return lhs.match(implication.getLhs(), metaVariables) &&
               rhs.match(implication.getRhs(), metaVariables);
    }
}
