package logic.formulas;

import java.util.HashMap;

/*
 * NOTE:
 * We could have implemented all (binary) formulas with inheritance
 * without so much duplicated code. While this would be a better design,
 * we hope that this more specific approach is easier to understand.
 */

/** 
 * A biimplication is a formula expressing that
 * one subformula is true if and only if another subformula is true.
 * It is written with parentheses and a "<->" between the subformulas,
 * e.g., "(phi <-> psi)".
 */
public class Biimplication extends Formula {
    // Left-hand side (phi in the example above)
    private Formula lhs;
    // Right-hand side (psi in the example above)
    private Formula rhs;

    public Biimplication(Formula lhs, Formula rhs) {
        this.lhs = lhs;
        this.rhs = rhs;
    }

    public Formula getLhs() {
        return lhs;
    }

    public Formula getRhs() {
        return rhs;
    }

    @Override
    public String toString() {
        return "(" + lhs + " <-> " + rhs + ")";
    }

    /**
     * A formula can only be matched with a biimplication if it also is a biimplication and
     * the respective subformulas can be matched.
     * For example, ((A /\ B) <-> C) can be matched with (phi <-> psi) iff
     * (A /\ B) can be matched with phi and C can be matched with psi.
     */
    @Override
    public boolean match(Formula formula, HashMap<String, Formula> metaVariables) {
        if (!(formula instanceof Biimplication))
            return false;
        Biimplication biimplication = (Biimplication) formula;
        return lhs.match(biimplication.getLhs(), metaVariables) &&
               rhs.match(biimplication.getRhs(), metaVariables);
    }
}
