package logic.formulas;

import java.util.HashMap;

/**
 * An atom is the base case of the inductive definition of formulas
 * and represents a statement that can be true or false.
 * Here we represent an atom as an alpha-numeric string.
 */
public class Atom extends Formula {
    private String name;

    public Atom(String name) {
        this.name = name;
    }

    @Override
    public String toString() {
        return name;
    }

    /**
     * A formula can be matched with a (meta-)atom, by mapping the atom name to the complete formula.
     * For example, (A /\ B) is matched with phi by mapping phi to (A /\ B).
     * This is only possible if we did not previously map the name to some other formula.
     */
    @Override
    public boolean match(Formula formula, HashMap<String, Formula> metaVariables) {
        // Check if there already is a mapping for this name.
        Formula previousValue = metaVariables.get(name);
        if (previousValue == null) {
            // No previous mapping. Store the mapping and report success.
            metaVariables.put(name, formula);
            return true;
        } else {
            // If there is a previous mapping, matching is successful iff the name was
            // previously mapped to the same formula.
            return formula.equals(previousValue);
        }
    }
}
