import Statement
import cplex
import copy
import re


class Verifier(object):

    def __init__(self, pre, post, variables, program, onlyStrings=False):
        self.givenPre = pre
        self.givenPost = post
        self.derivedPre = ''
        self.variables = variables
        self.program = program
        self.intermediatePreconditions = []
        self.invertedDerivedPreconditions = []
        self.onlyStrings = onlyStrings

    def derivePrecondition(self):
        '''
        go through the statements starting at the bottom and always derive a weakest
        liberal precondition. Then use the derived precondition as the postcondition
        for the next statement.

        provides a list of derived preconditions and all intermediate conditions
        '''
        newPrecondition = self.givenPost
        for statement in reversed(self.program):
            self.intermediatePreconditions.append(newPrecondition)
            if isinstance(statement, Statement.Statement):
                newPrecondition = statement.wlp(newPrecondition)

        self.derivedPre = newPrecondition
        if not self.onlyStrings:
            # rearrange derived precondition and invert them
            self.invertedDerivedPreconditions = self.rearrangePreconditions(self.derivedPre, True)
            # rearrange derived preconditions and do not invert them
            self.derivedPre = self.rearrangePreconditions(self.derivedPre, False)
            # rearrange given preconditions and not inverting them
            self.givenPre = self.rearrangePreconditions(self.givenPre, False)

        # reverse the intermediate conditions so the order matches the statements
        self.intermediatePreconditions = list(reversed(self.intermediatePreconditions))

    def printDerivedPreconditions(self):
        self.printListOfConditions(self.derivedPre)
        for i in range(len(self.program)):
            self.program[i].print()
            self.printListOfConditions(self.intermediatePreconditions[i])

    def printResults(self):
        print('Given preconditions:')
        self.printListOfConditions(self.givenPre)
        print('\nDerived preconditions')
        self.printListOfConditions(self.derivedPre)
        print('\nGiven postconditions')
        self.printListOfConditions(self.givenPost)
        print('\nDerivation tree:')
        self.printDerivedPreconditions()

    def printListOfConditions(self, conditions):
        if self.onlyStrings:
            # add white spaces
            c = re.sub("(\\<\\=|\\>\\=|\\<|\\>|\\=\\=|\\!\\=|\\&\\&|\\|\\|)", " \\1 ", conditions)
            c = c.replace(',', ', ')
            print('{' + c + '}')
            return

        # if PPConditions are used we convert them to strings to print
        conditionsString = ''
        for c in conditions:
            if not c == conditions[-1]:
                conditionsString += c.toString() + ', '
            else:
                conditionsString += c.toString()
        print('{' + conditionsString + '}')

    def rearrangePreconditions(self, listOfConditions, invert):
        '''
        If invert is set to true: invert each precondition
        then rearrange it (move all variables to the left, all scalars to the right)
        and finally remove strict inequalities
        '''

        # deep copy for the inverted conditions since we do not want to replace the
        # non-inverted conditions
        result = copy.deepcopy(listOfConditions)
        for condition in result:
            if invert:
                condition.invertCondition()
            condition.rearrangeCondition()
            condition.removeStrictInequality()

            # only for inverted conditions use removeUnequal
            # since the inverted conditions are in disjunctive normal form
            if condition.comp == '!=' and invert:
                a = condition.removeUnequal()
                result = result + [a]
        return result

    def getConstraints(self, listOfConstraints):
        '''
        Prepare a list of preconditions to be used with cplex.
        The list will be seen as a conjunction of conditions.
        returns a list of linear expressions, what each expression senses (greater, less than, equals),
        the right hand side of the expressions, the constraint names and the variable names.
        '''
        lin_expr = []
        sensesList = []
        rhsList = []
        constraintNames = []
        variableNamesTemp = []
        i = 0
        for condition in listOfConstraints:
            ind, val, senses, rhs = condition.prepareForCplex()
            lin_expr.append(cplex.SparsePair(ind=ind, val=val))
            sensesList.append(senses)
            rhsList.append(rhs)
            constraintNames.append('c' + str(i))
            variableNamesTemp += ind
            i += 1

        # remove duplicates from variabelNames
        variableNames = list(dict.fromkeys(variableNamesTemp))
        return lin_expr, sensesList, rhsList, constraintNames, variableNames


    def testCplex(self):
        '''
        Use cplex to verify the given Hoare triple
        Try to find a solution which satisfies (givenPre and not derivedPre)
        If we find a solution this means that the given preconditions do not
        imply the derived preconditions and therefore the Hoare triple is not valid

        Finally print the if the triple is valid and provide a counter example if not
        '''
        valid = True

        for invDerivedPre in self.invertedDerivedPreconditions:
            # initialize cplex optimizer
            solver = cplex.Cplex()

            # deactivate cplex output
            solver.set_log_stream(None)
            solver.set_error_stream(None)
            solver.set_warning_stream(None)
            solver.set_results_stream(None)

            # conjunction of given preconditions and one inverted derived precondition
            currentConstraints = self.givenPre + [invDerivedPre]

            # obtain constraints and variable names
            lin_expr, sensesList, rhsList, constraintNames, variableNames = self.getConstraints(currentConstraints)
            # adding variables
            solver.variables.add(names=variableNames)
            # adding linear constraints
            solver.linear_constraints.add(
                lin_expr=lin_expr,
                senses=sensesList,
                rhs=rhsList,
                names=constraintNames
            )

            # solve the current linear problem
            solver.solve()

            # try to retrieve the solution found by cplex
            # if no solution exists, a CplexSolverError will be raised
            try:
                '''
                There is a solution which fulfills:
                GivenPre && !invDerivedPre
                '''
                result = solver.solution.get_values()
                valid = False
                break
            except :
                '''
                There is no solution
                '''
                valid = True

        if valid:
            print('\n####-####-####-#### \n')
            print('The Hoare triple is valid!')
            print('\n####-####-####-####')
        else:
            print('\n####-####-####-#### \n')
            print('The Hoare triple is not valid!')
            counterExampleList = list(zip(solver.variables.get_names(), result))
            counterExampleString = ""
            for v in counterExampleList:
                counterExampleString += v[0] + ' = ' + str(v[1]) + '; '

            print('Counter example: ' + counterExampleString)
            print('\n####-####-####-####')



