import sympy
import re
import cplex


class PPCondition(object):
    def __init__(self, left, comp, right):
        '''
        Pre- and Postconditions of the form {left comp right}

        allowed comparators (comp): <, <=, >, >=, ==, !=
        Do not use != when writing preconditions as input to the verifier since cplex does not allow this comparator
        '''
        self.left = left
        self.comp = comp
        self.right = right

    def simplifyCondition(self):
        '''
        simplify the left and the right hand side of a condition
        '''
        self.left = str(sympy.simplify(self.left))
        self.right = str(sympy.simplify(self.right))


    def rearrangeCondition(self):
        '''
        rearrange the condition, such that the left hand side contains all variables
        and the right hand side only contains a constant
        '''
        leftConstant = ''
        rightConstant = ''
        leftVariables = ''
        rightVariables = ''

        '''
        use the arguments of a sympy expression to extract constants and variables and 
        their respective sign
        '''
        leftSymExpression = sympy.simplify(self.left)
        rightSymExpression = sympy.simplify(self.right)
        for a in leftSymExpression.args:
            if (isinstance(a, sympy.core.numbers.Integer) or isinstance(a, sympy.core.numbers.Rational)
                or isinstance(a, sympy.core.numbers.Half)) and not leftSymExpression.is_Mul:
                leftConstant += '-(' + str(a) + ')'
            elif isinstance(a, sympy.core.mul.Mul) or isinstance(a, sympy.core.symbol.Symbol):
                leftVariables += '-(' + str(a) + ')'
            else:
                print('unexpected argument in sympy expression')

        for a in rightSymExpression.args:
            if isinstance(a, sympy.core.numbers.Integer):
                rightConstant += '-(' +  str(a) + ')'
            elif rightSymExpression.is_zero:
                rightConstant += '0'
            elif isinstance(a, sympy.core.mul.Mul) or isinstance(a, sympy.core.symbol.Symbol):
                rightVariables += '-(' + str(a) + ')'
            else:
                print('unexpected argument in sympy expression')

        '''
        shift constants to the right and variables to the left
        '''
        self.left = str(sympy.simplify(self.left + leftConstant + rightVariables))
        self.right = str(sympy.simplify(self.right + leftConstant + rightVariables))


    def invertCondition(self):
        '''
        replace the current comparator with the inverted one
        '''
        c = self.comp

        if c == '<':
            self.comp = '>='
            return
        elif c == '>':
            self.comp = '<='
            return
        elif c == '<=':
            self.comp = '>'
            return
        elif c == '>=':
            self.comp = '<'
            return
        elif c == '==':
            self.comp = '!='
            return
        elif c == '!=':
            self.comp = '=='
            return
        else:
            print("not a valid comparator")


    def print(self):
        print('{' + self.toString() + '}')

    def toString(self) -> str:
        return self.left + ' ' + self.comp + ' ' + self.right

    def removeStrictInequality(self):
        '''
        remove < and > and replace with <= and >= respectively
        Since we deal with integers we can either add or subtract 1 from the right hand side to
        maintain the same inequality
        '''
        if self.comp == '<':
            self.comp = '<='
            self.right += '-1'
            self.right = str(sympy.simplify(self.right))
        elif self.comp == '>':
            self.comp = '>='
            self.right += '+1'
            self.right = str(sympy.simplify(self.right))

    def removeUnequal(self):
        '''
        replace != with two inequalities. This should only be applied to the inverted derived preconditions
        since the two inequalities are disjunct
        '''
        if self.comp == '!=':
            a = PPCondition(self.left, '<=', self.right + '-1')
            a.simplifyCondition()
            self.comp = '>='
            self.right = self.right + '+1'
            self.simplifyCondition()
            return a

    def prepareForCplex(self):
        '''
        Assuming the condition has already been rearranged:

        left contains only linear combination of variables
        right only contains an integer or a division
        '''
        senses = ''
        if self.comp == '<=':
            senses = 'L'
        elif self.comp == '>=':
            senses = 'G'
        elif self.comp == '==':
            senses = 'E'
        else:
            print("not a valid sense")

        ind = []
        val = []
        leftSym = sympy.simplify(self.left)

        if leftSym.is_Add:
            for a in leftSym.args:
                if a.is_symbol:
                    ind.append(str(a))
                    val.append(1.0)
                elif a.is_Mul:
                    ind.append(str(a.args[1]))
                    val.append(float(a.args[0]))

        elif leftSym.is_Mul:
            for a in leftSym.args:
                if a.is_Integer or isinstance(a, sympy.core.numbers.Rational) or isinstance(a, sympy.core.numbers.Half):
                    val.append(float(a))
                elif a.is_symbol:
                    ind.append(str(a))

        elif leftSym.is_symbol:
            ind.append(str(leftSym))
            val.append(1.0)

        # right hand side contains division:
        # extract dividend and divisor and perform division
        # we can either use an integer division or floating point arithmetic. Both work with cplex
        if '/' in self.right:
            a = self.right.split('/')[0]
            b = self.right.split('/')[1]
            # this could be used for integer division
            # rhs = float((int(a) // int(b)))

            # this could be used for floating point precision
            rhs = float((float(a) / float(b)))
        else:
            rhs = float(self.right)

        return ind, val, senses, rhs
