/*
  This file contains the state space definition for Sokoban.

  These are the only problem-specific (i.e., specific to Sokoban)
  parts of the code; everything else is generic and can be used without
  change for other search problems.
*/

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Scanner;

public class SokobanStateSpace implements StateSpace {
    /*
      We make Sokoban states and actions private since the search code
      cannot and should not look into the state.
    */

    // Represents a grid cell that is part of the grid.
    private static class GridCell {
        private int x;
        private int y;

        public GridCell(int x, int y) {
            this.x = x;
            this.y = y;
        }

        public GridCell(GridCell other) {
            this.x = other.x;
            this.y = other.y;
        }

        @Override
        public boolean equals(Object other_) {
            if (!(other_ instanceof GridCell)) {
                return false;
            }
            GridCell other = (GridCell)other_;
            return (this.x == other.x) && (this.y == other.y);
        }

        @Override
        public int hashCode() {
            return this.x * 31 + this.y;
        }

        public String toString() {
            return "cell " + x + " / " + y;
        }
    }

    private static class SokobanState implements State {
        private GridCell agentPos;
        private ArrayList<GridCell> boxPos;

        public SokobanState(GridCell agentPos, ArrayList<GridCell> boxPos) {
            this.agentPos = agentPos;
            this.boxPos = boxPos;
        }

        @Override
        public boolean equals(Object other_) {
            if (!(other_ instanceof SokobanState)) {
                return false;
            }
            SokobanState other = (SokobanState)other_;
            return this.boxPos.equals(other.boxPos) && this.agentPos.equals(other.agentPos);
        }

        @Override
        public int hashCode() {
            return agentPos.hashCode() * 17 + boxPos.hashCode();
        }

        // Returns the index of the box at GridCell pos, and -1 if there
        // is no box.
        public int getBoxIndex(GridCell pos) {
            for (int index = 0; index < boxPos.size(); ++index) {
                if (boxPos.get(index).equals(pos)) {
                    return index;
                }
            }
            return -1;
        }

        // Returns true if there is a box at GridCell pos.
        public boolean hasBoxAt(GridCell pos) {
            return getBoxIndex(pos) != -1;
        }

        public String toString() {
            String result = "agent: " + agentPos.toString() + "\n";
            for (int index = 0; index < boxPos.size(); ++index) {
                result += "box " + index + ": " + boxPos.get(index).toString();
                if (index != boxPos.size() -1) {
                    result += "\n";
                }
            }
            return result;
        }
    }


    // Represents a movement of the agent.
    private static class MoveAction implements Action {
        private GridCell from;
        private GridCell to;
        public int cost;

        public MoveAction(GridCell from, GridCell to, int cost) {
            this.from = from;
            this.to = to;
            this.cost = cost;
        }

        public int cost() {
            return cost;
        }

        public String toString() {
            return "move(" + from.toString() + ", " + to.toString() + ")[cost=" + cost + "]";
        }
    }

    // Represents the action where a box is pushed by one grid cell and
    // the agent moves along with it.
    private static class PushAction implements Action {
        public GridCell agentPos;
        public GridCell boxPos;
        public GridCell boxTargetPos;
        public int cost;

        public PushAction(
            GridCell agentPos, GridCell boxPos, GridCell boxTargetPos, int cost) {
            this.agentPos = agentPos;
            this.boxPos = boxPos;
            this.boxTargetPos = boxTargetPos;
            this.cost = cost;
        }

        public int cost() {
            return cost;
        }

        public String toString() {
            return "push(" + agentPos.toString() + ", " + boxPos.toString() + ", " + boxTargetPos.toString() + ")[cost=" + cost + "]";
        }
    }

    /*
      A state space instance must store the grid, the initial state and
      the goal condition.
    */

    // grid cell[i][j] represents the cost of moving to that cell
    private int[][] grid;
    private int numberOfBoxes;
    private int gridWidth;
    private int gridHeight;

    private SokobanState initState;
    private ArrayList<GridCell> goalBoxPos;


    private SokobanStateSpace(int[][] grid, int numberOfBoxes,
                              SokobanState initState,
                              ArrayList<GridCell> goalBoxPos) {
        this.grid = grid;
        this.gridWidth = this.grid[0].length;
        this.gridHeight = this.grid.length;
        this.numberOfBoxes = numberOfBoxes;
        this.initState = initState;
        this.goalBoxPos = goalBoxPos;
        System.out.println("Instantiating problem instance with "
                           + numberOfBoxes + " boxes in a " + gridWidth
                           + "x" + gridHeight + " grid.");
    }

    /*
      The following four methods define the interface of state spaces
      used by the search code.

      We use the method names "init", "isGoal", "succ" and "cost" to
      stay as close as possible to the names in the lecture slides.
      Without this constraint, it would be better to use more
      self-explanatory names like "getSuccessorStates" instead of
      "succ".

      All methods are const because the state space itself never
      changes.
    */

    public State init() {
        // Just return the initial state that we stored.
        return initState;
    }

    public boolean isGoal(State s_) {
        SokobanState s = (SokobanState) s_;

        /*
          A state is a goal state if all boxes of the state are
          at their goal positions.
        */
        return s.boxPos.equals(goalBoxPos);
    }

    public ArrayList<ActionStatePair> succ(State s_) {
        SokobanState s = (SokobanState) s_;
        GridCell pos = s.agentPos;
        // Create relevant neighbor cells
        GridCell north = new GridCell(pos.x, pos.y+1);
        GridCell south = new GridCell(pos.x, pos.y-1);
        GridCell west = new GridCell(pos.x-1, pos.y);
        GridCell east = new GridCell(pos.x+1, pos.y);
        GridCell north2 = new GridCell(pos.x, pos.y+2);
        GridCell south2 = new GridCell(pos.x, pos.y-2);
        GridCell west2 = new GridCell(pos.x-2, pos.y);
        GridCell east2 = new GridCell(pos.x+2, pos.y);

        ArrayList<ActionStatePair> result = new ArrayList<ActionStatePair>();

        /* We can only move to neighboring grid cells that aren't walls
         * and if there isn't a box
         */
        // move north
        if (gridCellIsAccessible(north) && !s.hasBoxAt(north)) {
            result.add(createMoveSuccessor(s, north));
        }

        // move south
        if (gridCellIsAccessible(south) && !s.hasBoxAt(south)) {
            result.add(createMoveSuccessor(s, south));
        }

        // move west
        if (gridCellIsAccessible(west) && !s.hasBoxAt(west)) {
            result.add(createMoveSuccessor(s, west));
        }

        // move east
        if (gridCellIsAccessible(east) && !s.hasBoxAt(east)) {
            result.add(createMoveSuccessor(s, east));
        }

         /* We can push only boxes that are on a neighboring cell and if
         the cell in the direction of the push is vacant
        */
        // push north
        if (s.hasBoxAt(north) && gridCellIsAccessible(north2) && !s.hasBoxAt(north2)) {
            result.add(createPushSuccessor(s, north, north2));
        }

        // push south
        if (s.hasBoxAt(south) && gridCellIsAccessible(south2) && !s.hasBoxAt(south2)) {
            result.add(createPushSuccessor(s, south, south2));
        }

        // push west
        if (s.hasBoxAt(west) && gridCellIsAccessible(west2) && !s.hasBoxAt(west2)) {
            result.add(createPushSuccessor(s, west, west2));
        }

        // push east
        if (s.hasBoxAt(east) && gridCellIsAccessible(east2) && !s.hasBoxAt(east2)) {
            result.add(createPushSuccessor(s, east, east2));
        }

        return result;
    }

    public int cost(Action a) {
        return a.cost();
    }

    // The following are helper functions to test applicability of an
    // action and to create successor states.
    private boolean gridCellIsAccessible(GridCell cell) {
        return (cell.x < gridWidth) && (cell.y < gridHeight) &&
            (cell.x >= 0) && (cell.y >= 0) && (this.grid[cell.y][cell.x] > 0);
    }

    private ActionStatePair createMoveSuccessor(SokobanState s, GridCell to) {
        ArrayList<GridCell> boxPos = new ArrayList<GridCell>(s.boxPos);
        int cost = this.grid[to.y][to.x];
        return new ActionStatePair(new MoveAction(s.agentPos, to, cost), new SokobanState(to, boxPos));
    }

    private ActionStatePair createPushSuccessor(SokobanState s, GridCell agentTo, GridCell boxTo) {
        assert(s.hasBoxAt(agentTo));
        int boxIndex = s.getBoxIndex(agentTo);
        ArrayList<GridCell> boxPos = new ArrayList<GridCell>(s.boxPos);
        boxPos.set(boxIndex, boxTo);
        int cost = this.grid[agentTo.y][agentTo.x];
        return new ActionStatePair(new PushAction(s.agentPos, agentTo, boxTo, cost), new SokobanState(agentTo, boxPos));
    }

    /*
      The following method instantiates the state space by reading the
      problem description from a file specified on the command line.
      The Sokoban state space is a *parameterized* one (i.e., the
      initial state depends on arguments specified by the user of the
      code).

      Syntax of input:
      - first line: grid width - grid height - number of boxes
      - second to (grid height +1)st line: 0/1 specifying if the grid cell can be entered
      - next line: initial agent position
      - final (number of boxes many) lines: initial and goal position of boxes
    */
    public static StateSpace buildFromCmdline(ArrayList<String> args) {
        if (args.size() != 1) {
            Errors.usageError("Need one input file argument.");
        }

        String filename = args.get(0);
        System.out.println("Reading input from file " + filename + "...");
        Scanner scanner;
        try {
            scanner = new Scanner(new File(filename));
        } catch (FileNotFoundException e) {
            Errors.fileError("Input file not found: " + filename);
            scanner = new Scanner(""); // unreachable; silences compiler
        }

        if (!scanner.hasNextLine()) {
            Errors.fileError("Input file is empty.");
        }

        String line = scanner.nextLine();
        Scanner lineScanner = new Scanner(line);
        int gridWidth = lineScanner.nextInt();
        int gridHeight = lineScanner.nextInt();
        int numBoxes = lineScanner.nextInt();
        if (lineScanner.hasNext())
            Errors.fileError("expected end of line");
            lineScanner.close();
        if (numBoxes < 0 || gridWidth < 0 || gridHeight < 0) {
            Errors.fileError("invalid size of grid or number of boxes");
        }

        int[][] grid = scanGrid(gridWidth, gridHeight, scanner);

        line = scanner.nextLine();
        lineScanner = new Scanner(line);
        GridCell initialAgentPos = new GridCell(lineScanner.nextInt(), lineScanner.nextInt());
        ArrayList<GridCell> initialBoxPos = new ArrayList<GridCell>();
        ArrayList<GridCell> goalBoxPos = new ArrayList<GridCell>();
        if (lineScanner.hasNext())
            Errors.fileError("expected end of line");
        lineScanner.close();

        for (int index = 0; index < numBoxes; ++index) {
            line = scanner.nextLine();
            lineScanner = new Scanner(line);
            initialBoxPos.add(new GridCell(lineScanner.nextInt(), lineScanner.nextInt()));
            goalBoxPos.add(new GridCell(lineScanner.nextInt(), lineScanner.nextInt()));
            if (lineScanner.hasNext())
                Errors.fileError("expected end of line");
            lineScanner.close();
        }

        if (scanner.hasNext())
            Errors.fileError("expected end of file");

        scanner.close();

        GridCell[] boxes = {new GridCell(1,1)};
        SokobanState init = new SokobanState(initialAgentPos, initialBoxPos);

        return new SokobanStateSpace(grid, numBoxes, init, goalBoxPos);
    }

    private static int[][] scanGrid(int gridWidth, int gridHeight, Scanner scanner) {
        int[][] result = new int[gridHeight][gridWidth];
        for (int i = 0; i < gridHeight; ++i) {
            String line = scanner.nextLine();
            Scanner lineScanner = new Scanner(line);
            for (int j = 0; j < gridWidth; ++j) {
                if (!lineScanner.hasNextInt())
                    Errors.fileError("line of grid specification ended too early");
                int value = lineScanner.nextInt();
                result[i][j] = value;
            }
            if (lineScanner.hasNext())
                Errors.fileError("expected end of line");
            lineScanner.close();
        }
        return result;
    }
}
