# -*- coding: utf-8 -*-

"""Demo module to show black-box interface to state spaces.
Deliberately obscures the internal contents of states to emphasize
that a general search algorithm should not care about it.

Uses a fixed initial state."""

from __future__ import print_function

from puzzle_state_space import PuzzleStateSpace

INITIAL_STATE = (2, None, 7, 4, 5, 8, 1, 6, 3)


class EightPuzzleWithFixedInitialState(object):
    def __init__(self):
        self._state_space = PuzzleStateSpace(INITIAL_STATE)

    def init(self):
        return EightPuzzleState(self._state_space.get_initial_state())

    def is_goal(self, state):
        s = state._internal_state
        return self._state_space.is_goal(s)

    def succ(self, state):
        s = state._internal_state
        return [(action, EightPuzzleState(succ))
                for action, succ in self._state_space.get_successors(s)]

    def cost(self, action):
        return self._state_space.get_action_cost(action)


class EightPuzzleState(object):
    def __init__(self, internal_state):
        self._internal_state = internal_state

    def dump(self):
        tiles = iter(self._internal_state)
        dashes = "----" * 3 + "-"
        for row in range(3):
            print(dashes)
            for col in range(3):
                print("| ", end="")
                tile = next(tiles)
                if tile is None:
                    print("  ", end="")
                else:
                    print(tile, " ", sep="", end="")
            print("|")
        print(dashes)


def create_state_space():
    return EightPuzzleWithFixedInitialState()
