package turingmachine;

import java.util.HashMap;

/**
 * A transition function maps tuples (q, a) to tuples (q', b, d), where
 *     q is a state from Q \ E
 *     q' is a state from Q
 *     a and b are symbols from Gamma
 *     d is a direction from {L, R, N}
 * The tuple (q, a) is called the TransitionCondition and the tuple
 * (q', b, d) is called the TransitionEffect.
 * We inherit from HashMap, so we also get all functions that are part of
 * HashMap by inheritance.
 *
 * Example (assume q1, q2 are States, and a, b, blank are Symbols):
 *      // Create a transition function
 *      TransitionFunction delta = new TransitionFunction();
 *      // Define its values
 *      delta.put(q1, a, q1, a, Direction.RIGHT); // read an 'a' in q1, stay in q1, leave the 'a' on the tape and move right
 *      delta.put(q1, b, q1, a, Direction.RIGHT); // read a 'b' in q1, stay in q1, replace the 'b' with an 'a' on the tape and move right
 *      delta.put(q1, blank, q2, blank, Direction.LEFT); // when you read a blank in q1, go to q2, leave the blank and move left
 *
 *      // Check if an entry exists
 *      TransitionCondition condition = new TransitionCondition(q1, a);
 *      if (!delta.containsKey(condition))
 *          throw new Exception();
 *
 *      // Query a value from delta
 *      TransitionCondition condition = new TransitionCondition(q1, a);
 *      TransitionEffect effect = delta.get(condition);
 *      State nextState = effect.getState();
 *      Symbol nextSymbol = effect.getSymbol();
 *      Direction direction = effect.getDirection();
 */
class TransitionFunction extends HashMap<TransitionCondition, TransitionEffect> {
    private static final long serialVersionUID = -8714703614651384280L;

    /**
     * Default constructor
     */
    public TransitionFunction() {
        super();
    }

    /**
     * Copy constructor
     */
    public TransitionFunction(TransitionFunction d) {
        super(d);
    }

    /**
     * Specialized put function. See example above.
     */
    public void put(State qFrom, Symbol sRead, State qTo, Symbol sWrite, Direction dir) {
        super.put(new TransitionCondition(qFrom, sRead),
                  new TransitionEffect(qTo, sWrite, dir));
    }
}
