package turingmachine;

import java.util.Deque;
import java.util.LinkedList;
import java.util.List;

/**
 * Tape models the infinite tape of a Turing machine. Obviously, we have limited
 * memory and cannot implement an infinite tape. This implementation increases the
 * size of tape dynamically if the read/write head of the Turing machine reaches
 * the current end of the tape, so the class actually represents the used part of
 * the tape. This will stay finite since we can only simulate a finite number of
 * steps.
 */
public class Tape {
    // We have to remember the blank symbol to increase the size of the tape dynamically.
    private final Symbol blank;
    // Visited part of the tape before the head
    private Deque<Symbol> alpha;
    // Visited part of the tape from current position to the right-most position
    // that was visited so far. Beta is always non-empty, because it contains at least
    // the symbol under the current position of the head.
    private Deque<Symbol> beta;

    /**
     * Initialize a new tape that contains just the given word with the head
     * positioned over the first symbol of the word. 
     * @param word content of the tape.
     * @param blank symbol that is used for all unvisited positions.
     */
    public Tape(List<Symbol> word, Symbol blank) {
        this.blank = blank;
        this.alpha = new LinkedList<Symbol>();
        this.beta = new LinkedList<Symbol>(word);
        // Beta should always contain at least the symbol under the current
        // position of the head. If the given word is empty, the head points
        // to a blank symbol that is on the tape by default.
        if (beta.isEmpty()) {
            beta.add(blank);
        }
    }

    /**
     * @return the symbol currently under the head.
     */
    public Symbol read() {
        // The first symbol of beta is the symbol under the head.
        return beta.peekFirst();
    }

    /**
     * Write the given symbol to the current position of the head.
     * @param symbol symbol to write
     */
    public void write(Symbol symbol) {
        // The first symbol of beta is the symbol under the head.
        // Replace it with the given symbol.
        beta.removeFirst();
        beta.addFirst(symbol);
    }

    /**
     * Move the head one position to the left.
     */
    public void moveLeft() {
        // To move left, we move one symbol from the end of alpha to the start
        // of beta. If alpha is empty, the symbol we move is a blank.
        // (All symbols before alpha are implicitly blank)
        if (alpha.isEmpty()) {
            beta.addFirst(blank);
        } else {
            beta.addFirst(alpha.removeLast());
        }
    }

    /**
     * Move the head one position to the right.
     */
    public void moveRight() {
        // To move right, we move one symbol from the start of beta to the end
        // of alpha. If beta becomes empty because of this move, we add a blank.
        // (All symbols after beta are implicitly blank)
        alpha.addLast(beta.removeFirst());
        if (beta.isEmpty()) {
            beta.addFirst(blank);
        }
    }

    /**
     * Print the current content of the used part of the tape
     * up to (but not including) the current position of the head.
     * Does not print a newline at the end.
     */
    public void dumpAlpha() {
        for (Symbol s : alpha) {
            System.out.print(s);
        }
    }

    /**
     * Print the current content of the used part of the tape
     * starting from (and including) the current position of the head.
     * Does not print a newline at the end.
     */
    public void dumpBeta() {
        for (Symbol s : beta) {
            System.out.print(s);
        }
    }

    /**
     * @return the number of tape positions that where used so far.
     * A position is used iff it is part of the original input or the head moved
     * there at some point.
     */
    public int usedSpace() {
        // Alpha and beta together make up the used part of the tape.
        return alpha.size() + beta.size();
    }
}
