/*
  This file contains the state space definition for the 4-pegs Tower of
  Hanoi where there is a use-specified number of disks.

  This file contains the only problem-specific (i.e., specific to the 4-pegs
  Tower of Hanoi) parts of the code; everything else is generic and can be used
  without change for other search problems.

  Representation:
   - Disks are represented as plain ints, and the value of the int also
     corresponds to the size of the disk and to the cost of moving the disk.
   - As disks can only be added and removed to/from top of pegs, pegs are
     represented as stacks.
   - The invariant that no disk can be above any smaller disk can easily be
     maintained by ensuring that no disk is added to a peg whose top disk
     is smaller.
*/

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.ArrayDeque;
import java.util.Iterator;
import java.util.Scanner;

public class FourPegsTowerOfHanoiStateSpace implements StateSpace {
    /*
      A peg is represented as an ArrayDeque, used as a stack (i.e. only using
      methods to access and modify the first element of the deque).
    */
    private static class Peg extends ArrayDeque<Integer> {
        public void addFirst(Integer disk) {
            /*
              Overwrite for asserting that disks added to the peg are
              always smaller than the previously added disks.
            */
            if (!this.isEmpty()) {
                assert this.peekFirst() > disk: "attempt to put a disk on a smaller disk!";
            }
            super.addFirst(disk);
        }
    }

    /*
      We make FourPegsTowerOfHanoi states and actions private since the search
      code cannot and should not look into the state. Since they are only
      accessed from this class, we can make them plain old datatypes without
      violating encapsulation.
    */
    private static class FourPegsTowerOfHanoiState implements State {
        /*
          State of a RestrictedBlocks world instance. The set of pegs is an
          represented as an array.
        */

        public Peg[] pegs;

        public FourPegsTowerOfHanoiState(Peg[] pegs) {
            this.pegs = pegs;
        }

        public String toString() {
            String result = "[";
            for (int index = 0; index < this.pegs.length; ++index) {
                if (index != 0)
                    result += ", ";
                result += this.pegs[index].toString();
            }
            result += "]";
            return result;
        }
    }

    private static class FourPegsTowerOfHanoiAction implements Action {
        /*
          Action that represents moving disk "diskID" from peg "srcPeg"
          to peg "destPeg".
        */

        public int srcPeg;
        public int destPeg;
        public int diskID;

        public FourPegsTowerOfHanoiAction(int diskID, int srcPeg, int destPeg) {
            this.diskID = diskID;
            this.srcPeg = srcPeg;
            this.destPeg = destPeg;
        }

        public int cost() {
            return 1;
        }

        public String toString() {
            return "move(" + diskID + ", " + srcPeg + ", " + destPeg + ")";
        }
    }

    private int numberOfDisks;
    private static int numberOfPegs;
    private static FourPegsTowerOfHanoiState initState;

    private FourPegsTowerOfHanoiStateSpace(int numberOfDisks) {
        this.numberOfDisks = numberOfDisks;
        System.out.println("Instantiating problem instance with "
                           + numberOfDisks + " disks...");
    }

    /*
      The following four methods define the interface of state spaces used by
      the search code.

      We use the method names "init", "isGoal", "succ" and "cost" to stay as
      close as possible to the names in the lecture slides. Without this
      constraint, it would be better to use more self-explanatory names like
      "getSuccessorStates" instead of "succ".

      All methods are const because the state space itself never changes.

    */

    public State init() {
        // Just return the initial state that we stored.
        return initState;
    }

    public boolean isGoal(State s_) {
        FourPegsTowerOfHanoiState s = (FourPegsTowerOfHanoiState) s_;

        Peg[] pegs = s.pegs;
        if (!pegs[0].isEmpty() || !pegs[1].isEmpty() || !pegs[2].isEmpty()) {
            // If any of the first three pegs are occupied, the goal has
            // not been reached.
            return false;
        }

        // Assert that all disks are on the last peg in the correct order.
        Iterator it = pegs[3].iterator();
        assert(it.hasNext());
        Integer previous_element = (Integer) it.next();
        assert previous_element == this.numberOfDisks - 1;
        while (it.hasNext())  {
            Integer element = (Integer) it.next();
            assert element < previous_element;
            previous_element = element;
        }
        assert previous_element == 0;
        return true;
    }

    public ArrayList<ActionStatePair> succ(State s_) {
        FourPegsTowerOfHanoiState s = (FourPegsTowerOfHanoiState) s_;
        ArrayList<ActionStatePair> result = new ArrayList<ActionStatePair>();
        for (int srcPegID = 0; srcPegID < this.numberOfPegs; ++srcPegID) {
            if (!s.pegs[srcPegID].isEmpty()) {
                for (int destPegID = 0; destPegID < this.numberOfPegs; ++destPegID) {
                    /*
                      We can move the top disk of srcPegID to destPegID if
                      the other peg is empty or its top disk is larger.
                    */
                    if (srcPegID != destPegID &&
                        (s.pegs[destPegID].isEmpty() ||
                         s.pegs[destPegID].peekFirst() > s.pegs[srcPegID].peekFirst())) {
                        result.add(createSuccessor(s, srcPegID, destPegID));
                    }
                }
            }
        }
        return result;
    }

    private ActionStatePair createSuccessor(FourPegsTowerOfHanoiState s,
                                            int srcPegID, int destPegID) {
        /*
          Helper function that generates a single (action, successorState) pair
          for moving a disk from one peg to another.
        */

        Peg[] newPegs = new Peg[s.pegs.length];
        for (int index = 0; index < newPegs.length; ++index) {
            newPegs[index] = (Peg) s.pegs[index].clone();
        }
        int diskID = newPegs[srcPegID].removeFirst();
        newPegs[destPegID].addFirst(diskID);

        FourPegsTowerOfHanoiAction action = new FourPegsTowerOfHanoiAction(diskID, srcPegID, destPegID);
        FourPegsTowerOfHanoiState succState = new FourPegsTowerOfHanoiState(newPegs);

        return new ActionStatePair(action, succState);
    }

    public int cost(Action a_) {
        FourPegsTowerOfHanoiAction a = (FourPegsTowerOfHanoiAction) a_;
        return a.cost();
    }

    public int h(State s_) {
        // This is an exemplary implementation of the "blind" heuristic, which
        // returns 0 only for goal states. Implement the heuristic described
        // in Exercise 5.2 a) to replace the blind heuristic.
        if (isGoal(s_)) {
            return 0;
        } else {
            return 1;
        }
    }

    /*
      The following method instantiates the state space by reading the
      number of disks from a file specified on the command line. The file
      must contain exactly on positive integer.
    */
    public static StateSpace buildFromCmdline(ArrayList<String> args) {
        if (args.size() != 1) {
            Errors.usageError("need one input file argument");
        }

        String filename = args.get(0);
        System.out.println("Reading input from file " + filename + "...");
        Scanner scanner;
        try {
            scanner = new Scanner(new File(filename));
        } catch (FileNotFoundException e) {
            Errors.fileError("input file not found: " + filename);
            scanner = new Scanner(""); // unreachable; silences compiler
        }

        int numDisks = scanner.nextInt();
        if (numDisks < 1)
            Errors.fileError("invalid number of disks");

        if (scanner.hasNext())
            Errors.fileError("expected end of file");
        scanner.close();

        FourPegsTowerOfHanoiStateSpace.numberOfPegs = 4;
        Peg[] init_pegs = new Peg[FourPegsTowerOfHanoiStateSpace.numberOfPegs];
        for (int peg_no = 0; peg_no < FourPegsTowerOfHanoiStateSpace.numberOfPegs; ++peg_no) {
            init_pegs[peg_no] = new Peg();
        }
        for (int diskID = numDisks - 1; diskID >= 0; --diskID) {
            init_pegs[0].addFirst(diskID);
        }
        FourPegsTowerOfHanoiStateSpace.initState = new FourPegsTowerOfHanoiState(init_pegs);

        return new FourPegsTowerOfHanoiStateSpace(numDisks);
    }
}
