package turingmachine;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * A deterministic Turing machine that can be simulated for a finite number of
 * steps or until resources run out.
 */
public class TuringMachine {
    // Input alphabet
    private final Set<Symbol> Sigma;
    private final TransitionFunction delta;
    // Initial state
    private final State q0;
    private final Symbol blank;
    // Set of final states
    private final Set<State> E;

    // Current state of the machine during a simulation.
    private State currentState;
    // Tape which is modified during a simulation.
    private Tape tape;
    // Number of steps simulated so far in the current simulation.
    private int steps;

    /**
     * Construct a Turing machine and check input for validity.
     * @param Q a set of states
     * @param Sigma input alphabet
     * @param Gamma tape alphabet
     * @param delta transition function
     * @param q0 initial state
     * @param blank blank symbol
     * @param E set of final states
     * @throws InvalidSpecificationException if the input does not describe a
     *         Turing machine (see exercise sheet).
     */
    public TuringMachine(Set<State> Q,
                         Set<Symbol> Sigma,
                         Set<Symbol> Gamma,
                         TransitionFunction delta,
                         State q0,
                         Symbol blank,
                         Set<State> E) throws InvalidSpecificationException {
        // Store the definitions for later.
        // We do not need to store Gamma or Q (they are not needed later on)
        this.Sigma = new HashSet<Symbol>(Sigma);
        this.delta = new TransitionFunction(delta);
        this.q0 = q0;
        this.blank = blank;
        this.E = new HashSet<State>(E);
        // The tape is not yet initialized so we are not at the start of a simulation yet.
        this.currentState = null;
        this.tape = null;

        // Check that delta is a total function from (Q \ E) x Gamma to Q x Gamma x {L, R, N}.
        // The mapping delta has to have one entry for each tuple of state q from (Q \E) and
        // symbol s from Gamma. The right-hand side must be a tuple (q', s', d) where
        // q' is from Q, s' is from Gamma, and d is a direction.
        Set<State> nonFinalStates = new HashSet<State>(Q);
        nonFinalStates.removeAll(E);
        for (State q : nonFinalStates) {
            for (Symbol s : Gamma) {
                TransitionCondition condition = new TransitionCondition(q, s);
                if (!delta.containsKey(condition)) {
                    System.out.println(q);
                    System.out.println(s);
                    throw new InvalidSpecificationException("Delta is not a total function.");
                }
                TransitionEffect effect = delta.get(condition);
                if (!Q.contains(effect.getState())) {
                    throw new InvalidSpecificationException("Delta maps to a state which is not in Q.");
                }
                if (!Gamma.contains(effect.getSymbol())) {
                    throw new InvalidSpecificationException("Delta maps to a symbol which is not in Gamma.");
                }
             }
        }
        // We also check that there are no additional definitions for other state symbol pairs. While this is technically
        // a requirement for a total function, an error here would not be critical because we never ask for such a value.
        // This check was not required for the exercise.
        for (TransitionCondition condition : delta.keySet()) {
            if (!nonFinalStates.contains(condition.getState())) {
                throw new InvalidSpecificationException("Delta contains a mapping for a state which is not in (Q \\ E).");
            }
            if (!Gamma.contains(condition.getSymbol())) {
                throw new InvalidSpecificationException("Delta contains a mapping for a symbol which is not in Gamma.");
            }
        }

        // Check that Sigma is a subset of Gamma.
        if (!Gamma.containsAll(Sigma)) {
            throw new InvalidSpecificationException("Sigma is no subset of Gamma.");
        }

        // Check that blank is in Gamma but not in Sigma.
        if (!Gamma.contains(blank)) {
            throw new InvalidSpecificationException("Gamma does not contain the blank symbol.");
        }
        if (Sigma.contains(blank)) {
            throw new InvalidSpecificationException("Sigma contains the blank symbol.");
        }

        // Check that E is a subset of Q.
        if (!Q.containsAll(E)) {
            throw new InvalidSpecificationException("E is no subset of Q.");
        }
    }

    /**
     * Reset the Turing machine to the initial configuration for the given word.
     * @param word the input of the Turing machine simulation.
     * @throws InvalidSpecificationException if the given word contains symbols not in Sigma.
     */
    public void initialize(List<Symbol> word) throws InvalidSpecificationException {
        // Check that all symbols of word are from the input alphabet.
        if (!Sigma.containsAll(word)) {
            throw new InvalidSpecificationException("Word contains symbols that are not in Sigma.");
        }
        // Reset to initial configuration for the input word.
        currentState = q0;
        tape = new Tape(word, blank);
        steps = 0;
    }

    /**
     * Simulate one step of the Turing machine. The machine must have been initialized and before
     * calling this function. Once the simulation is over, i.e., the Turing machine stops, the
     * machine must be re-initialized before calling step() again.
     * @throws RuntimeException if the Turing machine is not initialized or is in a final state.
     */
    private void step() {
        // Check if the machine is not initialized.
        if (tape == null) {
            throw new RuntimeException("step() was called before TM was initialized.");
        }
        // Check if the machine is in a final state.
        if (E.contains(currentState)) {
            throw new RuntimeException("step() was called after TM reached a final state.");
        }

        // Check which transition applies to the current situation and get its effect.
        Symbol s = tape.read();
        TransitionCondition condition = new TransitionCondition(currentState, s);
        TransitionEffect effect = delta.get(condition);

        // Update the symbol under the read/write head on the tape.
        tape.write(effect.getSymbol());
        // Update the position of the read/write head on the tape.
        if (effect.getDirection() == Direction.LEFT) {
            tape.moveLeft();
        } else if (effect.getDirection() == Direction.RIGHT) {
            tape.moveRight();
        } else {
            assert effect.getDirection() == Direction.NEUTRAL;
        }
        // Update the current state.
        currentState = effect.getState();
        steps++;
    }

    /**
     * Simulate the Turing machine until it stops, or resources run out.
     * The Turing machine must be initialized before calling this function.
     */
    public void run() {
        // We simulate up to Long.MAX_VALUE steps. Resources will run out
        // before we will reach this number of simulated steps.
        run(Integer.MAX_VALUE);
    }

    /**
     * Simulate the Turing machine for up to maxSteps steps, or until it stops,
     * or resources run out.
     * The Turing machine must be initialized before calling this function.
     * @param maxSteps maximal number of steps to simulate.
     */
    public void run(int maxSteps) {
        // Simulate steps until we are in a final state or simulated maxSteps steps.
        dumpConfiguration();
        while (!E.contains(currentState) && steps < maxSteps) {
            step();
            dumpConfiguration();
        }
        dumpStatistics();
        System.out.println();
        System.out.println();
    }

    /**
     * Print the current configuration of the Turing machine.
     */
    public void dumpConfiguration() {
        if (tape == null) {
            System.out.println("Not initialized.");
        } else {
            tape.dumpAlpha();
            System.out.print(currentState);
            tape.dumpBeta();
            System.out.println();
        }
    }

    /**
     * Print the number of steps simulated so far and the current size of the used tape.
     */
    public void dumpStatistics() {
        if (tape == null) {
            System.out.println("Not initialized.");
        } else {
            System.out.println("Steps so far: " + steps);
            System.out.println("Tape used so far: " + tape.usedSpace());
        }
    }
}
