package turingmachine;

import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;

public class SimulateTM {

    public static void main(String[] args) throws InvalidSpecificationException {
        // The given TM has 7 states. We do not put them in a set yet, so we
        // don't lose their order.
        State[] states = new State[7];
        for (int i = 0; i < states.length; ++i) {
            states[i] = new State(i);
        }

        // We need four symbols (a,b,c,x) in Sigma and an additional symbol for the blank.
        Symbol a = new Symbol("a");
        Symbol b = new Symbol("b");
        Symbol c = new Symbol("c");
        Symbol x = new Symbol("x");
        Symbol blank = new Symbol(".");

        // Sigma must not contain the blank.
        Set<Symbol> Sigma = new HashSet<Symbol>(Arrays.asList(a, b, c));
        // Gamma has to contain the blank and all symbols from Sigma.
        Set<Symbol> Gamma = new HashSet<Symbol>(Arrays.asList(a, b, c, x, blank));

        // Put all states in the set of states Q.
        Set<State> Q = new HashSet<State>(Arrays.asList(states));
        // Only the state q6 is a final state.
        Set<State> E = new HashSet<State>(Arrays.asList(states[6]));

        // Create the transition function
        TransitionFunction delta = new TransitionFunction();
        // Define all outgoing edges for q0
        delta.put(states[0], a,     states[1], x,     Direction.RIGHT);
        delta.put(states[0], b,     states[0], b,     Direction.NEUTRAL);
        delta.put(states[0], c,     states[0], c,     Direction.NEUTRAL);
        delta.put(states[0], x,     states[0], x,     Direction.NEUTRAL);
        delta.put(states[0], blank, states[6], blank, Direction.NEUTRAL);
        // Define all outgoing edges for q1
        delta.put(states[1], a,     states[1], a,     Direction.RIGHT);
        delta.put(states[1], b,     states[2], x,     Direction.RIGHT);
        delta.put(states[1], c,     states[1], c,     Direction.NEUTRAL);
        delta.put(states[1], x,     states[1], x,     Direction.RIGHT);
        delta.put(states[1], blank, states[1], blank, Direction.NEUTRAL);
        // Define all outgoing edges for q2
        delta.put(states[2], a,     states[2], a,     Direction.NEUTRAL);
        delta.put(states[2], b,     states[2], b,     Direction.RIGHT);
        delta.put(states[2], c,     states[3], x,     Direction.NEUTRAL);
        delta.put(states[2], x,     states[2], x,     Direction.RIGHT);
        delta.put(states[2], blank, states[2], blank, Direction.NEUTRAL);
        // Define all outgoing edges for q3
        delta.put(states[3], a,     states[4], a,     Direction.LEFT);
        delta.put(states[3], b,     states[3], b,     Direction.LEFT);
        delta.put(states[3], c,     states[3], c,     Direction.NEUTRAL);
        delta.put(states[3], x,     states[3], x,     Direction.LEFT);
        delta.put(states[3], blank, states[5], blank, Direction.RIGHT);
        // Define all outgoing edges for q4
        delta.put(states[4], a,     states[4], a,     Direction.LEFT);
        delta.put(states[4], b,     states[4], b,     Direction.LEFT);
        delta.put(states[4], c,     states[4], c,     Direction.LEFT);
        delta.put(states[4], x,     states[0], x,     Direction.RIGHT);
        delta.put(states[4], blank, states[4], blank, Direction.LEFT);
        // Define all outgoing edges for q5
        delta.put(states[5], a,     states[5], a,     Direction.NEUTRAL);
        delta.put(states[5], b,     states[5], b,     Direction.NEUTRAL);
        delta.put(states[5], c,     states[5], c,     Direction.NEUTRAL);
        delta.put(states[5], x,     states[5], x,     Direction.RIGHT);
        delta.put(states[5], blank, states[6], blank, Direction.NEUTRAL);

        // Create the Turing machine.
        TuringMachine tm = new TuringMachine(Q, Sigma, Gamma, delta, states[0], blank, E);

        // ======= Simulate the Turing machine on words where it terminates =====
        // Note: The machine accepts the language L = {a^n b^n c^n | n >= 0}

        System.out.println("Word: Empty word");
        // Simulate the Turing machine on the empty word
        tm.initialize(new LinkedList<Symbol>());
        tm.run();
        System.out.println("-------------------------------");

        System.out.println("Word: abc");
        // Simulate the Turing machine on the word abc
        tm.initialize(Arrays.asList(a, b, c));
        tm.run();
        System.out.println("-------------------------------");

        System.out.println("Word: aabbcc");
        // Simulate the Turing machine on the word aabbcc
        tm.initialize(Arrays.asList(a, a, b, b, c, c));
        tm.run();
        System.out.println("-------------------------------");

        System.out.println("Word: aaabbbccc");
        // Simulate the Turing machine on the word aaabbbccc
        tm.initialize(Arrays.asList(a, a, a, b, b, b, c, c, c));
        tm.run();
        System.out.println("-------------------------------");

        // ======= Simulate the Turing machine on words where it does not terminate =====

        System.out.println("Word: x");
        // Simulate the Turing machine on the word x
        tm.initialize(Arrays.asList(c));
        tm.run(2);
        System.out.println("-------------------------------");

        System.out.println("Word: aabbccc");
        // Simulate the Turing machine on the word aabbccc
        tm.initialize(Arrays.asList(a, a, b, b, c, c, c));
        tm.run(30);
        System.out.println("-------------------------------");

        System.out.println("Word: aaccbb");
        // Simulate the Turing machine on the word aaccbb
        tm.initialize(Arrays.asList(a, a, c, c, b, b));
        tm.run(4);
        System.out.println("-------------------------------");
    }

}
