/*
  This file contains the state space definition for blocks world.

  These are the only problem-specific (i.e., specific to the blocks world) parts
  of the code; everything else is generic and can be used without change for
  other search problems.
*/

import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Scanner;

public class FreecellStateSpace implements StateSpace {
    public enum Suit {
        CLUBS, SPADES, HEARTS, DIAMONDS
    }

    public static class Card {
        public Suit suit;
        public int rank;
        
        public Card(Suit suit, int rank) {
            this.suit = suit;
            this.rank = rank;
        }

        public Card(Card other) {
            this.suit = other.suit;
            this.rank = other.rank;
        }

        @Override
        public boolean equals(Object other_) {
            if (other_ instanceof Card) {
                Card other = (Card) other_;
                return (this.suit == other.suit) && (this.rank == other.rank);
            }
            return false;
        }

        public boolean isBlack() {
            return (this.suit == Suit.CLUBS) || (this.suit == Suit.SPADES);
        }

        public String toString() {
            String result = "";
            switch (this.suit) {
            case CLUBS:
                result += "C-";
                break;
            case SPADES:
                result += "S-";
                break;
            case HEARTS:
                result += "H-";
                break;
            case DIAMONDS:
                result += "D-";
                break;
            }
            result += this.rank;
            return result;
        }
    }

    /*
      The Pile class represents both table positions and card piles at the
      specific table position.
     */
    private static class Pile {
        public Pile(int maxSize) {
            this.ID = ++nextID;
            assert (maxSize > 0);
            this.maxSize = maxSize;
            this.cards = new Card[this.maxSize];
            for(int index = 0; index < this.maxSize; ++index) {
                this.cards[index] = null;
            }
        }

        public Pile(Pile other) {
            this.ID = other.ID;
            this.maxSize = other.maxSize;
            this.cards = new Card[this.maxSize];
            for(int index = 0; index < this.maxSize; ++index) {
                if (other.cards[index] != null) {
                    this.cards[index] = new Card(other.cards[index]);
                } else {
                    this.cards[index] = null;
                }
            }
        }

        public boolean isEmpty() {
            return cards[0] == null;
        }

        public boolean isFull() {
            return cards[this.maxSize-1] != null;
        }

        // removes the top card and returns it
        public Card removeTopmost() {
            assert (!isEmpty());
            for (int index = this.maxSize-1; index >= 0; --index) {
                if (this.cards[index] != null) {
                    Card result = this.cards[index];
                    this.cards[index] = null;
                    return result;
                }
            }
            assert(false);
            return null;
        }

        public Card getTopmost() {
            assert (!isEmpty());
            for (int index = this.maxSize-1; index >= 0; --index) {
                if (this.cards[index] != null) {
                    return this.cards[index];
                }
            }
            assert(false);
            return null;
        }

        public void add(Card card) {
            assert(!isFull());
            for (int index = 0; index < this.maxSize; ++index) {
                if (this.cards[index] == null) {
                    this.cards[index] = card;
                    return;
                }
            }
        }

        public String toString() {
            String result = "(";
            for (int index = 0; index < this.maxSize; ++index) {
                if (this.cards[index] == null) {
                    break;
                }
                if (index != 0) {
                    result += ", ";
                }
                result += this.cards[index].toString();
            }
            result += ")";
            return result;
        }
        
        // The identifier of this pile
        public int ID;
        // The maximal size of this pile
	public int maxSize;
        // The IDs of the cards that build this pile, (-1 for all "empty"
        // positions in the pile)
        public Card[] cards;

        private static int nextID = 0;
    }

    /*
      We make Freecell states and actions private since the search code
      cannot and should not look into the state. Since they are only accessed
      from this class, we can make them plain old datatypes without violating
      encapsulation.
    */
    private static class FreecellState implements State {
        /*
          State of a Freecell world instance. The set of piles is represented as
          an array.
        */

        public Pile[] piles;
        public int minClubsRank;
        public int minSpadesRank;
        public int minHeartsRank;
        public int minDiamondsRank;

        public FreecellState(Pile[] piles, int minClubsRank, int minSpadesRank,
                             int minHeartsRank, int minDiamondsRank) {
            this.piles = piles;

            for (int index = 0; index < this.piles.length; ++index) {
                assert (this.piles[index].ID == index);
            }

            this.minClubsRank = minClubsRank;
            this.minSpadesRank = minSpadesRank;
            this.minHeartsRank = minHeartsRank;
            this.minDiamondsRank = minDiamondsRank;            
        }

        public String toString() {
            String result = "[";
            for (int index = 0; index < this.piles.length; ++index) {
                if (index != 0) {
                    result += ", ";
                }
                result += this.piles[index].toString();
            }
            result += " | ";
            result += this.minClubsRank + ", " + this.minSpadesRank + ", "
                + this.minHeartsRank + ", " + this.minDiamondsRank + "]";
            return result;
        }
    }

    private static class MoveCardAction implements Action {
        public Card card;
        public int srcPile;
        public int destPile;

        public MoveCardAction(Card card, int srcPile, int destPile) {
            this.card = card;
            this.srcPile = srcPile;
            this.destPile = destPile;
        }

        public String toString() {
            return "move(" + card.toString() + ", " + srcPile + ", " + destPile + ")";
        }

        public int cost() {
            return this.card.rank + 1;
        }
    }

    private static class DepositCardAction implements Action {
        public Card card;
        public int srcPile;

        public DepositCardAction(Card card, int srcPile) {
            this.card = card;
            this.srcPile = srcPile;
        }

        public String toString() {
            return "deposit(" + card.toString() + ", " + srcPile + ")";
        }

        public int cost() {
            return 1;
        }
    }

    private int numberOfPiles;
    private int maxRank;

    private FreecellState initState;

    /*
      We use the fact that the set of states is actually a single state in
      Freecell problems.
     */
    private FreecellStateSpace(int numberOfPiles, int maxRank,
                               FreecellState initState) {
	this.numberOfPiles = numberOfPiles;
        this.maxRank = maxRank;
        this.initState = initState;
    }

    /*
      The following four methods define the interface of state spaces used by
      the search code.

      We use the method names "init", "isGoal", "succ" and "cost" to stay as
      close as possible to the names in the lecture slides. Without this
      constraint, it would be better to use more self-explanatory names like
      "getSuccessorStates" instead of "succ".

      All methods are const because the state space itself never changes.

    */

    public State init() {
        // Just return the initial state that we stored.
        return initState;
    }

    // The goal of each Freecell instance is that all cards are on the
    // foundations (which is the case if all piles are empty)
    public boolean isGoal(State s_) {
        FreecellState s = (FreecellState) s_;

        for (Pile p : s.piles) {
            if (!p.isEmpty()) {
                return false;
            }
        }
        return true;
    }

    public ArrayList<ActionStatePair> succ(State s_) {
        FreecellState s = (FreecellState) s_;

        ArrayList<ActionStatePair> result = new ArrayList<ActionStatePair>();

        /*
          We can only move cards that are at the top (in our representation:
          back) of a pile, and we can only move them to the top of another
          pile and only if that pile still has capacity.
        */

        for (int srcPileID = 0; srcPileID < this.numberOfPiles; ++srcPileID) {
            if (!s.piles[srcPileID].isEmpty()) {
                Card topCard = s.piles[srcPileID].getTopmost();

                // We can move the top tile of this pile to another pile
                for (int destPileID = 0; destPileID < this.numberOfPiles; ++destPileID) {
                    if (srcPileID != destPileID && !s.piles[destPileID].isFull() &&
                        (s.piles[destPileID].isEmpty() || (s.piles[destPileID].getTopmost().isBlack() != topCard.isBlack()))) {
                        result.add(createMoveSuccessor(s, srcPileID, destPileID));
                    }
                }

                switch (topCard.suit) {
                case CLUBS:
                    if (topCard.rank == s.minClubsRank) {
                        result.add(createDepositSuccessor(s, srcPileID));
                    }
                    break;
                case SPADES:
                    if (topCard.rank == s.minSpadesRank) {
                        result.add(createDepositSuccessor(s, srcPileID));
                    }
                    break;
                case HEARTS:
                    if (topCard.rank == s.minHeartsRank) {
                        result.add(createDepositSuccessor(s, srcPileID));
                    }
                    break;
                case DIAMONDS:
                    if (topCard.rank == s.minDiamondsRank) {
                        result.add(createDepositSuccessor(s, srcPileID));
                    }
                    break;
                }
            }
        }
        return result;
    }

    private ActionStatePair createDepositSuccessor(FreecellState s, int srcPileID) {
        Pile[] newPiles = new Pile[numberOfPiles];
        for (int index = 0; index < numberOfPiles; ++index) {
            newPiles[index] = new Pile(s.piles[index]);
        }
        Card card = newPiles[srcPileID].removeTopmost();

        FreecellState succState = null;
        switch (card.suit) {
        case CLUBS:
            succState = new FreecellState(newPiles, s.minClubsRank+1, s.minSpadesRank, s.minHeartsRank, s.minDiamondsRank);
            break;
        case SPADES:
            succState = new FreecellState(newPiles, s.minClubsRank, s.minSpadesRank+1, s.minHeartsRank, s.minDiamondsRank);
            break;
        case HEARTS:
            succState = new FreecellState(newPiles, s.minClubsRank, s.minSpadesRank, s.minHeartsRank+1, s.minDiamondsRank);
            break;
        case DIAMONDS:
            succState = new FreecellState(newPiles, s.minClubsRank, s.minSpadesRank, s.minHeartsRank, s.minDiamondsRank+1);
            break;
        }
        DepositCardAction action = new DepositCardAction(card, srcPileID);
        return new ActionStatePair(action, succState);
    }

    private ActionStatePair createMoveSuccessor(FreecellState s,
                                                int srcPileID, int destPileID) {
        /*
          Helper function that generates a single (action, successorState) pair
          for moving from one pile to another.
        */

        Pile[] newPiles = new Pile[numberOfPiles];
        for (int index = 0; index < numberOfPiles; ++index) {
            newPiles[index] = new Pile(s.piles[index]);
        }
        Card card = newPiles[srcPileID].removeTopmost();
        newPiles[destPileID].add(card);
        
        MoveCardAction action = new MoveCardAction(card, srcPileID, destPileID);
        FreecellState succState = new FreecellState(newPiles, s.minClubsRank, s.minSpadesRank, s.minHeartsRank, s.minDiamondsRank);

        return new ActionStatePair(action, succState);
    }

    public int cost(Action a) {
        return a.cost();
    }

    public static StateSpace buildFromCmdline(ArrayList<String> args) {
        if (args.size() != 1) {
            Errors.usageError("need one input file argument");
        }

        String filename = args.get(0);
        System.out.println("Reading input from file " + filename + "...");
        Scanner scanner;
        try {
            scanner = new Scanner(new File(filename));
        } catch (FileNotFoundException e) {
            Errors.fileError("input file not found: " + filename);
            scanner = new Scanner(""); // unreachable; silences compiler
        }

        int maxRank = scanner.nextInt();
        int numPiles = scanner.nextInt();

        if (maxRank < 0 || numPiles < 1)
            Errors.fileError("invalid number of cards per suit or piles");

        FreecellState init = scanInitState(numPiles, maxRank, scanner);

        if (scanner.hasNext())
            Errors.fileError("expected end of file");
        scanner.close();

        return new FreecellStateSpace(numPiles, maxRank, init);
    }

    private static FreecellState scanInitState(int numPiles, int maxRank, Scanner scanner) {
        boolean[] usedClubs = new boolean[maxRank];
        boolean[] usedSpades = new boolean[maxRank];
        boolean[] usedHearts = new boolean[maxRank];
        boolean[] usedDiamonds = new boolean[maxRank];

        Pile[] piles = new Pile[numPiles];
        for (int index = 0; index < numPiles; ++index) {
            int maxSize = scanner.nextInt();
            piles[index] = new Pile(maxSize);
            assert(piles[index].ID == index);

            while (true) {
                String cardAsString = scanner.next();
                if (cardAsString.equals("-1"))
                    break;
                String[] parts = cardAsString.split("-");
                assert (parts.length == 2);
                String suitAsString = parts[0];
                int cardID = Integer.parseInt(parts[1]);
                Suit suit = Suit.CLUBS;
                if (suitAsString.equals("C")) {
                    suit = Suit.CLUBS;
                    if (usedClubs[cardID]) {
                        Errors.fileError("duplicate card: " + cardAsString);
                    }
                    usedClubs[cardID] = true;
                } else if (suitAsString.equals("S")) {
                    suit = Suit.SPADES;
                    if (usedSpades[cardID]) {
                        Errors.fileError("duplicate card: " + cardAsString);
                    }
                    usedSpades[cardID] = true;
                } else if (suitAsString.equals("H")) {
                    suit = Suit.HEARTS;
                    if (usedHearts[cardID]) {
                        Errors.fileError("duplicate card: " + cardAsString);
                    }
                    usedHearts[cardID] = true;
                } else if (suitAsString.equals("D")) {
                    suit = Suit.DIAMONDS;
                    if (usedDiamonds[cardID]) {
                        Errors.fileError("duplicate card: " + cardAsString);
                    }
                    usedDiamonds[cardID] = true;
                } else {
                    Errors.fileError("invalid suit identifier: " + suitAsString);
                }

                if (cardID < 0 || cardID >= maxRank)
                    Errors.fileError("invalid card");
                piles[index].add(new Card(suit, cardID));
            }
        }
        return new FreecellState(piles, 0, 0, 0, 0);
    }
}
