package turingmachine;

import java.util.HashMap;

/**
 * A transition function maps tuples (z, a) to tuples (z', b, d), where
 *     z is a state from Z \ E
 *     z' is a state from Z 
 *     a and b are symbols from Gamma
 *     d is a direction from {L, R, N}
 * The tuple (z, a) is called the TransitionCondition and the tuple
 * (z', b, d) is called the TransitionEffect.
 * We inherit from HashMap, so we also get all functions that are part of
 * HashMap by inheritance.
 * 
 * Example (assume z1, z2 are States, and a, b, blank are Symbols):
 *      // Create a transition function
 *      TransitionFunction delta = new TransitionFunction();
 *      // Define its values
 *      delta.put(z1, a, z1, a, Direction.RIGHT); // read an 'a' in z1, stay in z1, leave the 'a' on the tape and move right
 *      delta.put(z1, b, z1, a, Direction.RIGHT); // read a 'b' in z1, stay in z1, replace the 'b' with an 'a' on the tape and move right
 *      delta.put(z1, blank, z2, blank, Direction.LEFT); // when you read a blank in z1, go to z2, leave the blank and move left
 *
 *      // Check if an entry exists
 *      TransitionCondition condition = new TransitionCondition(z1, a);
 *      if (!delta.containsKey(condition))
 *          throw new Exception();
 *
 *      // Query a value from delta
 *      TransitionCondition condition = new TransitionCondition(z1, a);
 *      TransitionEffect effect = delta.get(condition);
 *      State nextState = effect.getState();
 *      Symbol nextSymbol = effect.getSymbol();
 *      Direction direction = effect.getDirection();
 */
class TransitionFunction extends HashMap<TransitionCondition, TransitionEffect> {

    /**
     * Specialized put function. See example above.
     */
    public void put(State zFrom, Symbol sRead, State zTo, Symbol sWrite, Direction dir) {
        super.put(new TransitionCondition(zFrom, sRead),
                  new TransitionEffect(zTo, sWrite, dir));
    }
}