#include "GnomineSolver_Backbone.h"
#include "minefield.h"
#include <math.h>
#include <iostream>
#include <unordered_map>
#include <vector>
#include <algorithm>

enum backgroundOptions {
  NOMINE = 0,
  MINE,
  UNKNOWN,
  NOBACKBONE,
  CURRENT_FIELD,
  MODEL_NOMINE,
  MODEL_MINE,
  MODEL_NOTPOSSIBLE
};

static int colors[8][3] = {
  {0xcc, 0xcc, 0xcc},	/* light Grey  */
  {0xff, 0x00, 0x00},   /* Red */
  {0xff, 0xaa, 0x66},   /* light Orange */
  {0x99, 0x99, 0x99},   /* Grey */
  {0xff, 0xff, 0x66},   /* light Yellow */
  {0x00, 0xcc, 0x00},	/* dark Green */
  {0xcc, 0x00, 0x00},   /* dark Red */
  {0x00, 0x00, 0x00},	/* Black */
};

using namespace Minisat;

/*
 * Representation: Each field represents a Variable (Field 0 is Variable 0 etc).
 * If a variable is true, it means that there is a mine on the corresponding field
 */

GnomineSolver_Backbone::GnomineSolver_Backbone(GtkMineField *mfield) {
  this->mfield = mfield;
  state.satSolver = NULL;
  withGlobalConstraint = false;
  resetState();
  mfieldValues = new int[mfield->xsize*mfield->ysize];
  for(int i = 0; i < mfield->xsize*mfield->ysize; i++) {
    mfieldValues[i] = -1;
  }
  gtk_minefield_setupSolution(mfield, true);
  buildExtraGUI();
  updateMfieldValues();
};

GnomineSolver_Backbone::~GnomineSolver_Backbone() {
  delete mfieldValues;
  gtk_widget_destroy(gui.window);
};

void GnomineSolver_Backbone::resetState() {
  state.lambda.clear();
  state.backbone.clear();
  if(state.satSolver != NULL) {
    delete state.satSolver;
    state.satSolver = NULL;
  }
  state.currentField = -1;
  state.oldModel.clear();
  state.newModel.clear();
  state.finished = false;
};



void GnomineSolver_Backbone::checkMField() {
  setupSolver();

  std::unordered_map<int,bool> backbone = calculateBackbone();

};

void GnomineSolver_Backbone::setupSolver() {
  resetState();
  state.satSolver = new Solver();
  int size = mfield->xsize * mfield->ysize;

  //add the fields as variables. If the field is shown or is set mined add this as unit clause
  for(int i = 0; i < size; i++) {
    state.satSolver->newVar();
    if(mfieldValues[i] == 0) {
      state.satSolver->addClause(~mkLit(i));
    } else if(mfieldValues[i] == 1) {
      state.satSolver->addClause(mkLit(i));
    }
  }

  //go through all fields that have uncertain Neighbours and add the corresponding clauses to the Solver
  bool neighbours[8];
  for(int i = 0; i < size; i++) {
    if(!mfield->mines[i].shown) {
      continue;
    }
    int count = getUncertainNeighbours(i, neighbours);
    if(count > 0) {
      addClausesFromConstraint(i, state.satSolver, neighbours, count, mfield->mines[i].neighbours);
    }
  }

  //global flag constraint, if wanted
  if(withGlobalConstraint) {
    addGlobalFlagConstraint(state.satSolver);
  }
}

void GnomineSolver_Backbone::addClausesFromConstraint(int c, Solver * S, bool neighbours[8], int neighbourCount, int sum) {

  std::cout << "adding clauses from Constraint " << c << std::endl;

  //substract certain Neighbours that have a mine from sum
  for(int i = 0; i < 8; i++) {
    int nb = getNeighbour(c,i);
    if(nb != -1 && mfield->mines[nb].marked) {
      sum--;
    }
  }

  int possibleCombinations = pow(2, neighbourCount);
  int assignment[neighbourCount]; //if an entry in assignment is 1 we assign a mine to that position
  int positionMapping[neighbourCount];

  //this array maps, which variables the corresponding entries in the assignment are
  int cnt = 0;
  for(int i = 0; i < 8; i++) {
    if(neighbours[i]) {
      positionMapping[cnt] = getNeighbour(c, i);
      cnt++;
    }
  }

  //loop over all possible assignments
  for(int n = 0; n < possibleCombinations; n++) {
    
    //set up the assignment
    int computedSum = 0;
    for(int i = 0; i < neighbourCount; i++) {
      assignment[i] = ( (n << i) % possibleCombinations ) >> (neighbourCount-1);
      if(assignment[i] == 1) {
	computedSum++;
      }
    }
    if(computedSum != sum) {
      vec<Lit> lits;
      for(int i = 0; i < neighbourCount; i++) {
	if(assignment[i] == 0) {
	  lits.push( mkLit(positionMapping[i]) );
	} else {
	  lits.push( ~mkLit(positionMapping[i]) );
	}
      }

      S->addClause(lits);
    }//endif computedSum != sum
 
  }//endfor loop over all possible assignments

};

/*
 *Formula used: Binary encoding of at-most-k as presented in Frisch, Giannaros: "SAT Encodings of the At-Most-k Constraint, page 4
 */
void GnomineSolver_Backbone::addGlobalFlagConstraint(Minisat::Solver *s) {
  int totalFlags = mfield->mcount;
  int totalFields = mfield->xsize*mfield->ysize;
  int flaggedFields = 0;
  std::vector<int> *fieldmap = new std::vector<int>();
  for(int i = 0 ; i < totalFields; i++) {
    if(mfield->mines[i].shown == false) {
      if(mfield->mines[i].marked == MINE_MARKED) {
	flaggedFields++;
      } else {
	fieldmap->push_back(i);
      }
    }
  }
  int unknownFields = fieldmap->size();
  //finding out what the closes integer (rounded up) to log_2(totalFields) is
  int log2_n = 0;
  int uF_copy = unknownFields;
  while(uF_copy != 0) {
    uF_copy = uF_copy >> 1;
    log2_n++;
  }

  //<= totalFlags
  int newVars = atMostKConstraint(s, unknownFields, totalFlags-flaggedFields, log2_n, false, totalFields, *fieldmap);
  //>= totalFlags
  atMostKConstraint(s, unknownFields, unknownFields-totalFlags+flaggedFields, log2_n, true, totalFields+newVars, *fieldmap);

  delete fieldmap;


  


};

int GnomineSolver_Backbone::atMostKConstraint(Minisat::Solver *s, int n, int k, int log2_n, bool inverted, int startingPoint, std::vector<int>& fieldmap) {

  int B_start = startingPoint;
  int T_start = B_start + log2_n*k;

  //pushing new Variables
  for(int i = 0; i < k*n+k*log2_n; i++){
    state.satSolver->newVar();
  }
  
  for(int i = 0; i < n; i++) {

    vec<Lit> lits;
    if(!inverted) {
      lits.push(~mkLit(fieldmap.at(i)));
    } else {
      lits.push(mkLit(fieldmap.at(i)));
    }
    /*with symmetry breaking*/
    int start = std::max(1,k-n+(i+1))-1;
    int end = std::min(i+1,k);
    /*without symmetry breaking*/
    /*int start = 0;
      int end = k;*/
    for(int g = start; g < end; g++) {
      lits.push(mkLit(T_start+g*n+i));
    }
    s->addClause(lits);
   
    for(int g = start; g < end; g++) {
      for(int j = 0; j < log2_n; j++) {
	vec<Lit>lits;
	int b = B_start+g*log2_n+j;
	lits.push(~mkLit(T_start+g*n+i));
	if((i>>j)%2 == 1) {
	  lits.push(mkLit(b));
	} else {
	  lits.push(~mkLit(b));
	}
	s->addClause(lits);
      }//j
    }//g
    }//i
  return k*n+k*log2_n;
}


std::unordered_map<int,bool> GnomineSolver_Backbone::calculateBackbone() {
  Solver * s = state.satSolver;
  state.lambda.clear();
  state.backbone.clear();
  bool satisfiable = s->solve();
  if(!satisfiable) {
    std::cout << "your gnomine Problem isnt possible!" << std::endl;
    
    std::unordered_map<int,bool> dummy;
    return dummy;
    //TODO: how can i do this nicer??
  }

  insertModelIntoLambda(s, &state.lambda);

  std::cout << "lambda filled" << std::endl;

  //main loop (basically a foreach, guaranteed here since we always take away (at least) one element and never add elements to lambda)
  while(state.lambda.size() > 0) {
    checkFieldForBackboneWithMemory(-1);
  }//endwhile --> lambda is empty
  return state.backbone;

};

lbool GnomineSolver_Backbone::checkFieldForBackbonePrimitive(int var) {
  Solver* s = state.satSolver;
  bool sat_true = s->solve(mkLit(var));
  bool sat_false = s->solve(~mkLit(var));
  if(sat_true && !sat_false) {
    return l_True;
  } else if(!sat_true && sat_false) {
    return l_False;
  } else {
    return l_Undef;
  }
};


lbool GnomineSolver_Backbone::checkFieldForBackboneWithMemory(int var) {

    Solver* s = state.satSolver;
    std::unordered_map<int,bool>* lambda = &(state.lambda);
    std::unordered_map<int,bool>* backbone = &(state.backbone);

    /*if(mfield->changed) {
      updateMfieldValues();
      mfield->changed = false;
    }*/

    if(lambda->empty())  {
      bool sat = s->solve();
      if(!sat) {
        std::cout << "your gnomine Problem isnt possible!" << std::endl;
        return l_Undef;
      }
      insertModelIntoLambda(s, lambda);
      s->model.copyTo(state.oldModel);
    } else if(lambda->empty()) {
      state.finished = true;
      if(var == -1) { return l_Undef; }
    }



 
  std::pair<int,bool> elem;
  std::unordered_map<int,bool>::const_iterator it = lambda->find(var);

  //if we specify a variable but it is already clear whether it is a backbone or not we just return what it is
  if(it == lambda->end() && var == -1) {
    elem = *lambda->begin();
    lambda->erase(elem.first);
  } else if(it == lambda->end()) {
    if(backbone->find(var)!=backbone->end()) {
      return lbool(backbone->at(var));
    } else {
      return l_Undef;
    }

  } else {
    elem = *it;
    lambda->erase(var);
  }

  state.currentField = elem.first;

  if(state.newModel.size() >0) {
    state.newModel.copyTo(state.oldModel);
    state.newModel.clear();
  }

 
    Lit l;
    if(elem.second) {
      l = mkLit(elem.first);
    } else {
      l = ~mkLit(elem.first);
    }
    bool sat = s->solve(l);
    
    if(!sat) {
      backbone->insert( std::pair<int,bool>(elem.first, !elem.second) );
      s->addClause(~l);
      mfieldValues[elem.first] = !elem.second;
      gtk_minefield_setSolution(mfield, elem.first, !elem.second);
      gtk_extra_gui_update();
      return lbool(!elem.second);
    } else {
      s->model.copyTo(state.newModel);

      for(int i = 0; i < state.newModel.size(); i++) {
	if(state.newModel[i] == l_Undef) {
	  lambda->erase(i);
	} else {
	  std::unordered_map<int,bool>::iterator it = lambda->find(i);
	  //if the defined variable is found in lambda and has the same assignment, delete it
	  if(it != lambda->end() && ( ((*it).second && state.newModel[i] == l_True) || (!(*it).second && state.newModel[i] == l_False) ) ){
	    lambda->erase(i);
	  }
	}
      } //endfor loop over all variables

      gtk_extra_gui_update();
      return l_Undef;
    } //endif sat
};

void GnomineSolver_Backbone::insertModelIntoLambda(Solver * s, std::unordered_map<int,bool> * lambda) {

  vec<lbool> model;
  s->model.copyTo(model);
  for(int i = 0; i < mfield->xsize*mfield->ysize; i++) {
    if(mfield->mines[i].shown || mfield->mines[i].marked) { continue; }
    if(model[i] == l_True) {
      lambda->insert( std::pair<int,bool>(i,false) );
    } else if(model[i] == l_False) {
      lambda->insert( std::pair<int,bool>(i,true) );
    }
  }
}


void GnomineSolver_Backbone::updateMfieldValues() {
  for(int i = 0; i < mfield->xsize*mfield->ysize; i++) {
    int old = mfieldValues[i];
    if(mfieldValues[i] != 0) {
      if(mfield->mines[i].shown) {
	mfieldValues[i] = 0;
      } else if(mfield->mines[i].marked) {
	mfieldValues[i] = 1;
      } else {
	mfieldValues[i] = -1;
      }
    }//endif mfieldValues[i] != 0
  }
  setupSolver();
  gtk_extra_gui_update();
  for(int i = 0; i < mfield->ysize; i++) {
    for(int j = 0; j < mfield->xsize; j++) {
      if(mfieldValues[i*mfield->xsize+j] == 0){
       std::cout << '0';
      } else if(mfieldValues[i*mfield->xsize+j] == 1){
       std::cout << '1';
      } else {
       std::cout << '-';
      }
    }
    std::cout << std::endl;
  }
  std::cout << std::endl;
}

/*
 * This method sets those entries in the given bool-array on true which represent a neighbour which is still uncertain. Additionaly, it returns the count of uncertain Neighbours
 */
int GnomineSolver_Backbone::getUncertainNeighbours(int c, bool neighbours[8]) {
  int count = 0;
  for(int i = 0; i < 8; i++) {
    neighbours[i] = false;
    int nb = getNeighbour(c,i);
    if(nb != -1) {
      if(!mfield->mines[nb].shown && !mfield->mines[nb].marked) {
        count++;
	neighbours[i] = true;
      } else {
	neighbours[i] = false;
      }
    }
  }
  return count;
};

void GnomineSolver_Backbone::applyChanges() {
  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.analyzeRB)) == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.playRB), TRUE);
    std::cout << "Play Button active: " << gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.playRB)) << ", Analyze Button Active: " << gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.analyzeRB)) << std::endl;
    gtk_minefield_applyChanges(mfield);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.analyzeRB), TRUE);
  } else {
    gtk_minefield_applyChanges(mfield);
  }
};

void GnomineSolver_Backbone::analyzeField(int field) {
  checkFieldForBackboneWithMemory(field);
};

void GnomineSolver_Backbone::buildExtraGUI() {

  gui.window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW (gui.window), "Backbone Solver");
  gtk_window_set_default_size(GTK_WINDOW(gui.window), 510,600);
  gtk_widget_show(gui.window);

  GtkWidget * all_boxes = gtk_vbox_new(FALSE, 0);;
  gtk_container_add(GTK_CONTAINER(gui.window), all_boxes);
  gtk_widget_show(all_boxes);

  gui.fieldarea = gtk_drawing_area_new();
  gtk_widget_set_size_request(gui.fieldarea, 500, 500);
  gtk_box_pack_start(GTK_BOX(all_boxes), gui.fieldarea, TRUE, TRUE, 10);
  gtk_widget_show(gui.fieldarea);
  g_signal_connect (GTK_OBJECT (gui.fieldarea), "expose_event", 
    G_CALLBACK (GnomineSolver_Backbone::gtk_extra_gui_expose_proxy), this);

  cairo_t * cr = gdk_cairo_create(gui.fieldarea->window);
  cairo_rectangle(cr, 20, 20, 20, 20);
  cairo_set_source_rgb(cr,255,0,0);
  cairo_fill(cr);

  gui.buttonarea = gtk_vbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(all_boxes), gui.buttonarea, FALSE, FALSE, 10);
  gtk_widget_show(gui.buttonarea);

  GtkWidget * ba1 = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(gui.buttonarea), ba1, FALSE, FALSE, 10);
  gtk_widget_show(ba1);

  GtkWidget * ba2 = gtk_hbox_new(TRUE, 0);
  gtk_box_pack_start(GTK_BOX(gui.buttonarea), ba2, FALSE, FALSE, 10);
  gtk_widget_show(ba2);

  GtkWidget * nextFieldButton = gtk_button_new_with_label("next Field");
  gtk_box_pack_start(GTK_BOX(ba1), nextFieldButton, TRUE, TRUE, 10);
  gtk_widget_show(nextFieldButton);
  g_signal_connect(G_OBJECT(nextFieldButton), "clicked", G_CALLBACK(GnomineSolver_Backbone::checkNextField), this);

  GtkWidget * checkFieldButton = gtk_button_new_with_label("check entire Field");
  gtk_box_pack_start(GTK_BOX(ba1), checkFieldButton, TRUE, TRUE, 10);
  gtk_widget_show(checkFieldButton);
  g_signal_connect(G_OBJECT(checkFieldButton), "clicked", G_CALLBACK(GnomineSolver_Backbone::checkFieldCallback), this);

  GtkWidget * applyChangesButton = gtk_button_new_with_label("apply Changes");
  gtk_box_pack_start(GTK_BOX(ba1), applyChangesButton, TRUE, TRUE, 10);
  gtk_widget_show(applyChangesButton);
  g_signal_connect(G_OBJECT(applyChangesButton), "clicked", G_CALLBACK(GnomineSolver_Backbone::applyChangesCallback), this);

  gui.analyzeRB = gtk_radio_button_new_with_label(NULL, "analyze");
  gtk_widget_set_name(gui.analyzeRB, "analyzeRB");
  gtk_box_pack_start(GTK_BOX(ba2), gui.analyzeRB, TRUE, TRUE, 10);
  gui.playRB = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(gui.analyzeRB),"play");
  gtk_widget_set_name(gui.playRB, "playRB");
  gtk_box_pack_start(GTK_BOX(ba2), gui.playRB, TRUE, TRUE, 10);
  gtk_widget_show(gui.analyzeRB);
  gtk_widget_show(gui.playRB);
  g_signal_connect(G_OBJECT(gui.analyzeRB), "clicked", G_CALLBACK(GnomineSolver_Backbone::toggleAnalyzeModeCallback), this);
  g_signal_connect(G_OBJECT(gui.playRB), "clicked", G_CALLBACK(GnomineSolver_Backbone::toggleAnalyzeModeCallback), this);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.playRB), TRUE);

  GtkWidget * globalConstraintCheckbox = gtk_check_button_new_with_label("Global Flag Constraint");
  gtk_box_pack_start(GTK_BOX(ba2), globalConstraintCheckbox, TRUE, TRUE, 10);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(globalConstraintCheckbox), FALSE);
  gtk_widget_show(globalConstraintCheckbox);
  g_signal_connect(G_OBJECT(globalConstraintCheckbox), "clicked", G_CALLBACK(GnomineSolver_Backbone::toggleGlobalConstraintCallback), this);

  g_signal_emit(gui.fieldarea, "exposed", NULL);

};

gboolean GnomineSolver_Backbone::gtk_extra_gui_update() {
  gint width, height;
  GdkWindow * window = gtk_widget_get_window(gui.fieldarea);
  gdk_drawable_get_size( window, &width, &height);
  int xfields = mfield->xsize;
  int yfields = mfield->ysize;
  int fields = xfields*yfields;
  int fsize = 0;
  cairo_t * cr = gdk_cairo_create(window);

  //making it square
  if(width > height) { width = height; }
  else if(height > width) { height = width; }
  
  if(xfields > yfields) {
    fsize = width/xfields;
  } else {
    fsize = width/yfields;
  }

  for(int i = 0; i < fields; i++) {

    bool shown = mfield->mines[i].shown;
    bool marked = mfield->mines[i].marked;
    bool undetermined = !shown && !marked;
    bool inBackbone = state.backbone.find(i) != state.backbone.end();
    bool inLambda = state.lambda.find(i) != state.lambda.end();

    int x = (i%xfields)*fsize;
    int y = (i/xfields)*fsize;

    //set background color for the field depending if current field, unknown value, mine or no mine
    cairo_rectangle(cr, x + 2, y + 2, fsize-4, fsize-4);
    if(i == state.currentField) {
      cairo_set_source_rgb(cr, colors[CURRENT_FIELD][0]/255.0, colors[CURRENT_FIELD][1]/255.0, colors[CURRENT_FIELD][2]/255.0);
    } else if(shown){
      cairo_set_source_rgb(cr, colors[NOMINE][0]/255.0, colors[NOMINE][1]/255.0, colors[NOMINE][2]/255.0);
    } else if(marked) {
      cairo_set_source_rgb(cr, colors[MINE][0]/255.0, colors[MINE][1]/255.0, colors[MINE][2]/255.0);
    } else {
      cairo_set_source_rgb(cr, colors[UNKNOWN][0]/255.0, colors[UNKNOWN][1]/255.0, colors[UNKNOWN][2]/255.0);
    }
    cairo_fill(cr);


    //if mfield value is undetermined
    if(undetermined) {

      //if its not in lambda and not in backbone, it must have been tested and proved to be no backbone
      if(!inBackbone && !inLambda) {
	cairo_set_source_rgb(cr, colors[NOBACKBONE][0]/255.0, colors[NOBACKBONE][1]/255.0, colors[NOBACKBONE][2]/255.0);
	cairo_rectangle(cr, x+fsize/8, y+fsize/8, 6*fsize/8, 6*fsize/8);
	cairo_fill(cr);
      }

      if(inBackbone) {
	if(state.backbone.at(i) == TRUE) {
	  cairo_set_source_rgb(cr, colors[MODEL_MINE][0]/255.0, colors[MODEL_MINE][1]/255.0, colors[MODEL_MINE][2]/255.0);
	} else {
	  cairo_set_source_rgb(cr, colors[MODEL_NOMINE][0]/255.0, colors[MODEL_NOMINE][1]/255.0, colors[MODEL_NOMINE][2]/255.0);
	}
	cairo_rectangle(cr, x+fsize/8, y+fsize/8, 6*fsize/8, 6*fsize/8);
	cairo_fill(cr);

      } else { //undetermined and not in backbone
	cairo_move_to(cr,x+3, y+fsize-3);
	cairo_rel_line_to(cr, fsize-6, -fsize+6);
	cairo_set_source_rgb(cr, 0,0,0);
	cairo_stroke(cr);

	//paint old model
	if(state.oldModel.size() > 0)  {
	  if(state.oldModel[i] == l_True) {
	    cairo_set_source_rgb(cr, colors[MODEL_MINE][0]/255.0, colors[MODEL_MINE][1]/2550, colors[MODEL_MINE][2]/255.0);
	  } else if(state.oldModel[i] == l_False) {
	    cairo_set_source_rgb(cr, colors[MODEL_NOMINE][0]/255.0, colors[MODEL_NOMINE][1]/255.0, colors[MODEL_NOMINE][2]/255.0);
	  } else{
	    cairo_set_source_rgb(cr, colors[NOBACKBONE][0]/255.0, colors[NOBACKBONE][1]/255.0, colors[NOBACKBONE][2]/255.0);
}
	  cairo_rectangle(cr, x+fsize/8, y+fsize/8, fsize/4, fsize/4  );
	  cairo_fill(cr);
	}

	//paint new model
	if(state.newModel.size()) {
	  if(state.newModel[i] == l_True) {
	    cairo_set_source_rgb(cr, colors[MODEL_MINE][0]/255.0, colors[MODEL_MINE][1]/255.0, colors[MODEL_MINE][2]/255.0);
	  } else if(state.newModel[i] == l_False) {
	    cairo_set_source_rgb(cr, colors[MODEL_NOMINE][0]/255.0, colors[MODEL_NOMINE][1]/255.0, colors[MODEL_NOMINE][2]/255.0);
	  } else {
	    cairo_set_source_rgb(cr, colors[NOBACKBONE][0]/255.0, colors[NOBACKBONE][1]/255.0, colors[NOBACKBONE][2]/255.0);
}
	  cairo_rectangle(cr, x+5*fsize/8, y+5*fsize/8, fsize/4, fsize/4  );
	  cairo_fill(cr);
	} else if(state.oldModel.size() > 0 && state.newModel.size() == 0 && mfieldValues[i] == -1) { //no model for contrary of current field
	  cairo_set_source_rgb(cr,0,0,0);
	  cairo_rectangle(cr, x+5*fsize/8, y+5*fsize/8, fsize/4, fsize/4  );
	  cairo_fill(cr);
	}

      }       

    } //endif mfield value undetermined    


  }//endfor
 
  cairo_destroy(cr);
  return true;
};

static gboolean GnomineSolver_Backbone::gtk_extra_gui_expose_proxy(GtkWidget * widget, GdkEventExpose *event, gpointer data) {
  GnomineSolver_Backbone *_this = static_cast<GnomineSolver_Backbone*>(data);
  return _this->gtk_extra_gui_update();
};

static void GnomineSolver_Backbone::checkNextField(GtkWidget * widget, gpointer data) {
  GnomineSolver_Backbone *_this = static_cast<GnomineSolver_Backbone*>(data);
  _this->checkFieldForBackboneWithMemory(-1);
};

static void GnomineSolver_Backbone::checkFieldCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_Backbone *_this = static_cast<GnomineSolver_Backbone*>(data);
  _this->checkMField();
};

static void GnomineSolver_Backbone::applyChangesCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_Backbone *_this = static_cast<GnomineSolver_Backbone*>(data);
  _this->applyChanges();
};

static void GnomineSolver_Backbone::toggleAnalyzeModeCallback(GtkWidget *widget, gpointer data) {
  if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) == TRUE) {
    return;
  }
  GnomineSolver_Backbone *_this = static_cast<GnomineSolver_Backbone*>(data);
if(g_strcmp0(gtk_widget_get_name(widget), "analyzeRB") == 0) {
    gtk_minefield_setAnalyzeMode(_this->mfield, true);
 } else if (g_strcmp0(gtk_widget_get_name(widget), "playRB") == 0) {
    gtk_minefield_setAnalyzeMode(_this->mfield, false);
  }
};

static void GnomineSolver_Backbone::toggleGlobalConstraintCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_Backbone *_this = static_cast<GnomineSolver_Backbone*>(data);
  _this->withGlobalConstraint = !_this->withGlobalConstraint;
  _this->setupSolver();
}
