/* -*- mode:C; tab-width:8; c-basic-offset:8; indent-tabs-mode:true -*- */

/*
 * Mines for GNOME
 * Author:        Pista <szekeres@cyberspace.mht.bme.hu>
 *
 * Score support: horape@compendium.com.ar
 * Mine Resizing: djb@redhat.com
 *
 * This game is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 */

#include <config.h>

#include <iostream>
#include <time.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include <libgames-support/games-gtk-compat.h>
#include <libgames-support/games-runtime.h>
#include <libgames-support/games-preimage.h>

#include "minefield.h"
#include "gnomineSolver_CSP.h"

/* Auxillary data so we can use a single index to reference
   surrounding cells. */
static const struct {
  gint x;
  gint y;
} neighbour_map[8] = {
  {
  -1, 1}, {
  0, 1}, {
  1, 1}, {
  1, 0}, {
  1, -1}, {
  0, -1}, {
  -1, -1}, {
  -1, 0}
};

/* The colours for the numbers. The empty first entry allows us
 * to use the number as a direct index. */
static guint16 num_colors[9][3] = {
  {0x0000, 0x0000, 0x0000},	/* Black, not used */
  {0x0000, 0x0000, 0xffff},	/* Blue  */
  {0x0000, 0xa0a0, 0x0000},	/* Green */
  {0xffff, 0x0000, 0x0000},	/* Red   */
  {0x0000, 0x0000, 0x7fff},	/* Dark Blue */
  {0xa0a0, 0x0000, 0x0000},	/* Dark Red   */
  {0x0000, 0xffff, 0xffff},	/* Cyan */
  {0xa0a0, 0x0000, 0xa0a0},	/* Dark Violet */
  {0x0000, 0x0000, 0x0000}	/* Black */
};

/* The signal list for the widget. */
enum {
  MARKS_CHANGED_SIGNAL = 0,
  EXPLODE_SIGNAL,
  LOOK_SIGNAL,
  UNLOOK_SIGNAL,
  WIN_SIGNAL,
  HINT_SIGNAL,
  ANALYZE_SIGNAL,
  UNCOVER_SIGNAL,
  LAST_SIGNAL
};

/* The list of actions that can be performed when a mose button is
   pressed. */
enum {
  NO_ACTION,
  SHOW_ACTION,
  CLEAR_ACTION,
  FLAG_ACTION
};

/* Static data for the minefield class. */
static gint minefield_signals[LAST_SIGNAL];
static GtkWidgetClass *parent_class;
static int * bgcolors;

/*  Prototypes */
static inline gint get_cell_index_no_checks (GtkMineField * mfield, guint x,
				       guint y);
static gint get_cell_index (GtkMineField * mfield, guint x, guint y);
static void setup_sign (sign * signp, const char *file, guint minesizepixels);
static void gtk_mine_draw (GtkMineField * mfield, guint x, guint y);
static gint gtk_minefield_button_press (GtkWidget * widget,
					GdkEventButton * event);
static gint gtk_minefield_button_release (GtkWidget * widget,
					  GdkEventButton * event);
static void gtk_minefield_check_field (GtkMineField * mfield, gint x, gint y);
static void gtk_minefield_class_init (GtkMineFieldClass * class_);
static gboolean gtk_minefield_expose (GtkWidget * widget, GdkEventExpose * event);
static void gtk_minefield_init (GtkMineField * mfield);
static void gtk_minefield_lose (GtkMineField * mfield);
static gint gtk_minefield_motion_notify (GtkWidget * widget,
					 GdkEventMotion * event);
static void gtk_minefield_multi_release (GtkMineField * mfield, guint x,
					 guint y, guint c, guint really);
static void gtk_minefield_randomize (GtkMineField * mfield, int curloc);
static void gtk_minefield_realize (GtkWidget * widget);
static void gtk_minefield_setup_signs (GtkMineField * mfield);
static void gtk_minefield_show (GtkMineField * mfield, guint x, guint y);
static void gtk_minefield_size_allocate (GtkWidget * widget,
					 GtkAllocation * allocation);
static void gtk_minefield_size_request (GtkWidget * widget,
					GtkRequisition * requisition);
static void gtk_minefield_set_mark (GtkMineField * mfield, guint x,
				    guint y, int mark);
static void gtk_minefield_toggle_mark (GtkMineField * mfield, guint x,
				       guint y);
static void gtk_minefield_unrealize (GtkWidget * widget);
static void gtk_minefield_win (GtkMineField * mfield);
static int gtk_minefield_check_cell (GtkMineField * mfield, guint x, guint y);
static void gtk_minefield_multi_press (GtkMineField * mfield, guint x,
				       guint y, gint c);
static gboolean gtk_minefield_solve_square (GtkMineField * mfield, guint x,
					    guint y, guint c);
/* end prototypes */

void gtk_minefield_setupBgColors(GtkMineField * mfield) {
  int size = mfield->xsize*mfield->ysize;
  if(bgcolors != NULL) {
    delete bgcolors;
  }
  bgcolors = new int[3*size];
  int nullcolor[3] = {-1,-1,-1};
  for(int i = 0; i < size; i++) {
    gtk_minefield_set_field_bg(mfield,nullcolor, i);
  }
}

void gtk_minefield_set_field_bg(GtkMineField * mfield, int clr[3], int c) {
  bgcolors[3*c+0] = clr[0];
  bgcolors[3*c+1] = clr[1];
  bgcolors[3*c+2] = clr[2];
  gtk_mine_draw(mfield, c%mfield->xsize, c/mfield->xsize);
}

void gtk_minefield_setupSolution(GtkMineField * mfield, bool repaint) {
  int size = mfield->xsize*mfield->ysize;
  if(mfield->mfieldSolution != NULL) {
    delete mfield->mfieldSolution;
  }
  mfield->mfieldSolution = new int[size];
  for(int i = 0; i < mfield->xsize*mfield->ysize; i++) {
    if(repaint) {
      gtk_minefield_setSolution(mfield,i,-1);
    } else {
      mfield->mfieldSolution[i] = -1;
    }
  }
}

void gtk_minefield_setSolution(GtkMineField * mfield, int c, int solution) {
  mfield->mfieldSolution[c] = solution;
  gtk_mine_draw(mfield, c%mfield->xsize, c/mfield->xsize);
}

void gtk_minefield_clearAdditionalInfo(GtkMineField *mfield) {
  gtk_minefield_setupBgColors(mfield);
  gtk_minefield_setupSolution(mfield, true);
  for(int i = 0; i < mfield->xsize*mfield->ysize; i++) {
    gtk_mine_draw(mfield, i%mfield->xsize, i/mfield->xsize);
  }
}

/* The abstraction of the coordinate system. Note that this is inline
   code that does no checking, use it sparsely. If in doubt, use
   get_cell_index instead. */
static inline gint
get_cell_index_no_checks (GtkMineField * mfield, guint x, guint y)
{
  return x + y * mfield->xsize;
}

void gtk_minefield_applyChanges(GtkMineField * mfield) {
  mfield->applyingChanges = true;
  int minesizepixels = mfield->minesizepixels;

  for(int c = 0;  c < mfield->xsize*mfield->ysize; c++) {
    if(mfield->mfieldSolution[c] == 0 && mfield->mines[c].shown != 1) {
      GdkEventButton event;
      event.button = 1;
      event.type = GDK_BUTTON_PRESS;
      event.x = (c%mfield->xsize) * minesizepixels + minesizepixels/2;
      event.y = (c/mfield->xsize) * minesizepixels + minesizepixels/2;
      event.state = 0;
      gtk_minefield_button_press(GTK_WIDGET(mfield), &event);
      GdkEventButton event2;
      event2.button = 1;
      event2.type = GDK_BUTTON_RELEASE;
      event2.x = event.x;
      event2.y = event.y; 
      event.state = 0;
      gtk_minefield_button_release(GTK_WIDGET(mfield), &event2);
    } else if(mfield->mfieldSolution[c] == 1 && mfield->mines[c].marked != MINE_MARKED) {
      GdkEventButton event;
      event.button = 3;
      event.type = GDK_BUTTON_PRESS;
      event.x = (c%mfield->xsize) * minesizepixels + minesizepixels/2;
      event.y = (c/mfield->xsize) * minesizepixels + minesizepixels/2;
      gtk_minefield_button_press(GTK_WIDGET(mfield), &event);
      GdkEventButton event2;
      event2.button = 3;
      event2.type = GDK_BUTTON_RELEASE;
      event2.x = event.x;
      event2.y = event.y;
      gtk_minefield_button_release(GTK_WIDGET(mfield), &event2);
    }
  }
  mfield->applyingChanges = false;
  gtk_minefield_clearAdditionalInfo(mfield);
  g_signal_emit (G_OBJECT (mfield),
         minefield_signals[UNCOVER_SIGNAL], 0, NULL);
}

void gtk_minefield_setAnalyzeMode(GtkMineField * mfield, bool mode) {
  mfield->analyzeMode = mode;
}


/* Converts 2D minefield coordinates into a 1D array index. Note that
   this is used extensively for checking the validity of
   coordinates. If the coordinates are not valid then it returns
   -1. */
static gint
get_cell_index (GtkMineField * mfield, guint x, guint y)
{
  if (x < mfield->xsize && y < mfield->ysize)
    return get_cell_index_no_checks (mfield, x, y);

  return -1;
}

/* Set up a pixbuf containing an object we overlay a cell with: flags,
  mines and explosions, but not numbers. This also takes care of
  any dmemory previously allocated to the sign. This function 
  should be treated as local to gtk_minefield_setup_signs. */
static void
setup_sign (sign * signp, const char *file, guint minesizepixels)
{
  if (!signp->preimage && file != NULL)
    signp->preimage = games_preimage_new_from_file (file, NULL);

  if (signp->scaledpixbuf)
    g_object_unref (signp->scaledpixbuf);

  signp->scaledpixbuf = NULL;
  signp->width = signp->height = minesizepixels - 2;

  if (signp->preimage) {
    signp->scaledpixbuf = games_preimage_render (signp->preimage,
						 signp->width,
						 signp->height);
  }

  if (!signp->scaledpixbuf) {
    signp->scaledpixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB,
					  TRUE, 8,
					  signp->width, signp->height);
    gdk_pixbuf_fill (signp->scaledpixbuf, 0x00000000);
    if (signp->preimage)
      g_object_unref (signp->preimage);
    signp->preimage = NULL;
  }
}

static void
gtk_minefield_setup_signs (GtkMineField * mfield)
{
  static GtkWidget *warning_dialog = NULL;
  static gchar *warning_message = NULL;
  gchar *flagfile, *minefile, *questionfile, *bangfile, *warningfile;
  const char *dname = games_runtime_get_directory (GAMES_RUNTIME_GAME_PIXMAP_DIRECTORY);

  flagfile = g_build_filename (dname, "flag.svg", NULL);
  minefile = g_build_filename (dname, "mine.svg", NULL);
  questionfile = g_build_filename (dname, "flag-question.svg", NULL);
  bangfile = g_build_filename (dname, "bang.svg", NULL);
  warningfile = g_build_filename (dname, "warning.svg", NULL);

  if ((!flagfile || !minefile || !questionfile || !bangfile || !warningfile)
      && (warning_message == NULL)) {
    warning_message =
      _
      ("Unable to find required images.\n\nPlease check your gnome-games installation.");
  }

  setup_sign (&mfield->flag, flagfile, mfield->minesizepixels);
  setup_sign (&mfield->mine, minefile, mfield->minesizepixels);
  setup_sign (&mfield->question, questionfile, mfield->minesizepixels);
  setup_sign (&mfield->bang, bangfile, mfield->minesizepixels);
  setup_sign (&mfield->warning, warningfile, mfield->minesizepixels);

  g_free(flagfile);
  g_free(minefile);
  g_free(questionfile);
  g_free(bangfile);
  g_free(warningfile);

  if ((!mfield->flag.preimage ||
       !mfield->mine.preimage ||
       !mfield->question.preimage ||
       !mfield->bang.preimage ||
       !mfield->warning.preimage) && (warning_message == NULL)) {
    warning_message =
      _
      ("Required images have been found, but refused to load.\n\nPlease check your installation of gnome-games and its dependencies.");
  }


  if (warning_message && !warning_dialog) {
    GtkWindow *parent =
      GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (mfield)));
    warning_dialog =
      gtk_message_dialog_new (parent, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
			      GTK_BUTTONS_NONE, "%s", _("Could not load images"));

    gtk_dialog_add_button (GTK_DIALOG (warning_dialog),
			   GTK_STOCK_QUIT, GTK_RESPONSE_CLOSE);

    gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG
					      (warning_dialog),
					      "%s", warning_message);
    g_signal_connect (warning_dialog, "response", G_CALLBACK (gtk_main_quit),
		      NULL);
    gtk_widget_show (warning_dialog);
  }

}

static void
gtk_minefield_setup_numbers (GtkMineField * mfield)
{
  int minesizepixels, pixel_sz, i;
  static guint last_size = 0;

  minesizepixels = mfield->minesizepixels;

  pixel_sz = minesizepixels - 2;
  if (pixel_sz > 999)
    pixel_sz = 999;

  if (last_size == pixel_sz)
    return;
  last_size = pixel_sz;

  for (i = 0; i < 9; i++) {
    gchar text[2];
    PangoLayout *layout;
    PangoAttrList *alist;
    PangoAttribute *attr;
    PangoFontDescription *font_desc;
    PangoRectangle extent;
    guint64 font_size;

    /* free an existing layout ... */
    if (mfield->numstr[i].layout)
      g_object_unref (mfield->numstr[i].layout);

    text[0] = '0' + i;
    text[1] = '\0';
    layout = gtk_widget_create_pango_layout (GTK_WIDGET (mfield), text);

    /* set attributes for the layout */
    alist = pango_attr_list_new ();

    /* colour */
    attr = pango_attr_foreground_new (num_colors[i][0],
				      num_colors[i][1], num_colors[i][2]);
    attr->start_index = 0;
    attr->end_index = G_MAXUINT;
    pango_attr_list_insert (alist, attr);

    /* do the font */
    font_desc = pango_font_description_new ();
    pango_font_description_set_family (font_desc, "Sans");
    font_size = pixel_sz * PANGO_SCALE * .85;
    pango_font_description_set_absolute_size (font_desc, font_size);
    pango_font_description_set_weight (font_desc, PANGO_WEIGHT_BOLD);
    attr = pango_attr_font_desc_new (font_desc);

    attr->start_index = 0;
    attr->end_index = G_MAXUINT;
    pango_attr_list_insert (alist, attr);

    pango_layout_set_attributes (layout, alist);
    pango_layout_set_alignment (layout, PANGO_ALIGN_CENTER);

    pango_font_description_free (font_desc);
    pango_attr_list_unref (alist);

    mfield->numstr[i].layout = layout;

    pango_layout_get_extents (layout, NULL, &extent);

    /* The +1 is necessary since these coordinates are
     * with respect to minesizepixels, not pixel_sz (the
     * difference is 2). */
    mfield->numstr[i].dx = (pixel_sz - extent.width / PANGO_SCALE) / 2 + 1;
    mfield->numstr[i].dy = (pixel_sz - extent.height / PANGO_SCALE) / 2 + 1;
  }
}

static void
gtk_minefield_realize (GtkWidget * widget)
{
  GtkMineField *mfield;
  GtkAllocation allocation;
  GtkStyle *style;
  GdkWindow *window;
  GdkWindowAttr attributes;
  gint attributes_mask;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MINEFIELD (widget));

  mfield = GTK_MINEFIELD (widget);
  gtk_widget_set_realized (widget, TRUE);

  gtk_widget_get_allocation (widget, &allocation);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = allocation.x;
  attributes.y = allocation.y;
  attributes.width = allocation.width;
  attributes.height = allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= GDK_EXPOSURE_MASK |
    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK;

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  window = gdk_window_new (gtk_widget_get_parent_window (widget),
                           &attributes, attributes_mask);
  gtk_widget_set_window (widget, window);
  gdk_window_set_user_data (window, mfield);

  style = gtk_style_attach (gtk_widget_get_style (widget), window);
  gtk_widget_set_style (widget, style);
  gtk_style_set_background (style, window, GTK_STATE_ACTIVE);
}

static void
gtk_minefield_unrealize (GtkWidget * widget)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_MINEFIELD (widget));

  if (GTK_WIDGET_CLASS (parent_class)->unrealize)
    (*GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}

/* The frame makes sure that the minefield is allocated the correct size */
/* This is the standard allocate routine - it could be removed and the parents routine inherited */
static void
gtk_minefield_size_allocate (GtkWidget * widget, GtkAllocation * allocation)
{
  guint minesizepixels, width, height;
  guint xofs, yofs;
  GtkMineField *mfield;
  GdkWindow *window;

  gtk_widget_set_allocation (widget, allocation);
  window = gtk_widget_get_window (widget);

  mfield = GTK_MINEFIELD (widget);

  if (gtk_widget_get_realized (widget)) {
    minesizepixels = MIN (allocation->width / mfield->xsize,
		    allocation->height / mfield->ysize);
    mfield->minesizepixels = minesizepixels;
    width = mfield->xsize * minesizepixels;
    height = mfield->ysize * minesizepixels;
    xofs = allocation->x + (allocation->width - width) / 2;
    yofs = allocation->y + (allocation->height - height) / 2;


    if (!mfield->thick_line)
      mfield->thick_line = gdk_gc_new (window);
    gdk_gc_copy (mfield->thick_line, gtk_widget_get_style (widget)->black_gc);
    gdk_gc_set_line_attributes (mfield->thick_line,
				MAX (1, 0.1 * minesizepixels),
				GDK_LINE_SOLID,
				GDK_CAP_ROUND, GDK_JOIN_ROUND);

    gdk_window_move_resize (window, xofs, yofs, width, height);
  }
}

static void
gtk_minefield_size_request (GtkWidget * widget, GtkRequisition * requisition)
{
  GtkMineField *mf = GTK_MINEFIELD (widget);
  /* request the minimum size - to allow the widget window to be resized */
  requisition->width = mf->xsize * MINESIZE_MIN;
  requisition->height = mf->ysize * MINESIZE_MIN;
}

static void
gtk_mine_draw (GtkMineField * mfield, guint x, guint y)
{
  int c = get_cell_index (mfield, x, y);

  int noshadow;
  gboolean clicked;
  int n, nm;
  guint minesizepixels;
  static GdkGC *dots;
  static const char stipple_data[] = { 0x03, 0x03, 0x0c, 0x0c };
  static GdkPixmap *stipple = NULL;
  GtkStyle *style;
  GtkWidget *widget = GTK_WIDGET (mfield);
  GdkRectangle rect;
  GdkWindow *window;
  g_return_if_fail (c != -1);

  window = gtk_widget_get_window (widget);
  style = gtk_widget_get_style (widget);

  /* This gives us a dotted line to increase the contrast between
   * buttons and the "sea". */
  if (stipple == NULL) {
    stipple = gdk_bitmap_create_from_data (NULL, stipple_data, 4, 4);
    dots = gdk_gc_new (window);
    gdk_gc_copy (dots, style->dark_gc[2]);
    gdk_gc_set_stipple (dots, stipple);
    g_object_unref (stipple);
    gdk_gc_set_fill (dots, GDK_STIPPLED);
  }

  minesizepixels = mfield->minesizepixels;

  noshadow = mfield->mines[c].shown;
  clicked = mfield->mines[c].down;

  /* gtk_paint_box needs a clipping rectangle. */
  rect.x = x * minesizepixels;
  rect.y = y * minesizepixels;
  rect.width = minesizepixels;
  rect.height = minesizepixels;

  bool specialbg = bgcolors[3*c+0] > -1 && bgcolors[3*c+1] > -1 && bgcolors[3*c+2] > -1;


  if (!specialbg && noshadow) {		/* draw grid on ocean floor */
	  gtk_paint_box (style,
		   window,
		   clicked ? GTK_STATE_ACTIVE : GTK_STATE_NORMAL,
		   GTK_SHADOW_IN,
		   &rect,
		   widget,NULL, x * minesizepixels, y * minesizepixels, minesizepixels, minesizepixels);
    if (y == 0) {		/* top row only */
      gdk_draw_line (window,	/* top */
		     dots, x * minesizepixels, 0, x * minesizepixels + minesizepixels - 1, 0);
    }
    if (x == 0) {		/* left column only */
      gdk_draw_line (window,	/* left */
		     dots, 0, y * minesizepixels, 0, y * minesizepixels + minesizepixels - 1);
    }
    gdk_draw_line (window,	/* right */
		   dots,
		   x * minesizepixels + minesizepixels - 1,
		   y * minesizepixels,
		   x * minesizepixels + minesizepixels - 1, y * minesizepixels + minesizepixels - 1);
    gdk_draw_line (window,	/* bottom */
		   dots,
		   x * minesizepixels,
		   y * minesizepixels + minesizepixels - 1,
		   x * minesizepixels + minesizepixels - 1, y * minesizepixels + minesizepixels - 1);

  } else if (!specialbg) {			/* draw shadow around possible mine location */
    gtk_paint_box (style,
		   window,
		   clicked ? GTK_STATE_ACTIVE : GTK_STATE_SELECTED,
		   clicked ? GTK_SHADOW_IN : GTK_SHADOW_OUT,
		   &rect,
		   widget,
		   "button", x * minesizepixels, y * minesizepixels, minesizepixels, minesizepixels);
  } else {
    cairo_t * cr = gdk_cairo_create(window);
    cairo_rectangle(cr,rect.x+1, rect.y+1, rect.width-2, rect.height-2);
    cairo_set_source_rgb(cr, bgcolors[3*c+0]/255.0, bgcolors[3*c+1]/255.0, bgcolors[3*c+2]/255.0);
    cairo_fill(cr);
    cairo_destroy(cr);
  }

 
  if(!mfield->mines[c].shown && mfield->mfieldSolution[c] != -1) {
    if(mfield->mfieldSolution[c] == 0) {
      cairo_t * cr = gdk_cairo_create(window);
      cairo_rectangle(cr,(int)(rect.x+rect.width/4), (int)(rect.y+rect.height/4), (int)(rect.width/2), (int)(rect.height/2));
      cairo_set_source_rgb(cr, 0, 1, 0);
      cairo_fill(cr);
      cairo_destroy(cr);
    } else {
      cairo_t * cr = gdk_cairo_create(window);
      cairo_rectangle(cr,(int)(rect.x+rect.width/4), (int)(rect.y+rect.height/4), (int)(rect.width/2), (int)(rect.height/2));
      cairo_set_source_rgb(cr,1, 0, 0);
      cairo_fill(cr);
      cairo_destroy(cr);
    }
  }

  if (mfield->mines[c].shown && !mfield->mines[c].mined) {
    n = mfield->mines[c].neighbours;
    g_assert (n >= 0 && n <= 9);

    nm = mfield->mines[c].neighbourmarks;
    g_assert (nm >= 0 && nm <= 9);

    if (mfield->use_overmine_warning && n < nm) {
      gdk_draw_pixbuf (window, NULL,
		       mfield->warning.scaledpixbuf, 0, 0,
		       x * minesizepixels + (minesizepixels - mfield->warning.width) / 2,
		       y * minesizepixels + (minesizepixels - mfield->warning.height) / 2,
		       mfield->warning.width, mfield->warning.height,
		       GDK_RGB_DITHER_NORMAL, 0, 0);
    }

    if (n != 0) {
      gdk_draw_layout (window,
		       style->black_gc,
		       x * minesizepixels + mfield->numstr[n].dx,
		       y * minesizepixels + mfield->numstr[n].dy,
		       PANGO_LAYOUT (mfield->numstr[n].layout));
    }

  } else if (mfield->mines[c].marked == MINE_QUESTION) {
    gdk_draw_pixbuf (window, NULL,
		     mfield->question.scaledpixbuf, 0, 0,
		     x * minesizepixels + (minesizepixels - mfield->flag.width) / 2,
		     y * minesizepixels + (minesizepixels - mfield->flag.height) / 2,
		     mfield->flag.width, mfield->flag.height,
		     GDK_RGB_DITHER_NORMAL, 0, 0);
  } else if (mfield->mines[c].marked == MINE_MARKED) {
    gdk_draw_pixbuf (window, NULL,
		     mfield->flag.scaledpixbuf, 0, 0,
		     x * minesizepixels + (minesizepixels - mfield->flag.width) / 2,
		     y * minesizepixels + (minesizepixels - mfield->flag.height) / 2,
		     mfield->flag.width, mfield->flag.height,
		     GDK_RGB_DITHER_NORMAL, 0, 0);

    if (mfield->lose && mfield->mines[c].mined != 1) {
      int x1 = x * minesizepixels + 0.1 * minesizepixels;
      int y1 = y * minesizepixels + 0.1 * minesizepixels;
      int x2 = x * minesizepixels + 0.9 * minesizepixels;
      int y2 = y * minesizepixels + 0.9 * minesizepixels;

      gdk_draw_line (window, mfield->thick_line, x1, y1, x2, y2);
      gdk_draw_line (window, mfield->thick_line, x1, y2, x2, y1);
    }
  } else if (mfield->lose && mfield->mines[c].mined) {
    gdk_draw_pixbuf (window, NULL,
		     mfield->mine.scaledpixbuf, 0, 0,
		     x * minesizepixels + (minesizepixels - mfield->flag.width) / 2,
		     y * minesizepixels + (minesizepixels - mfield->flag.height) / 2,
		     mfield->flag.width, mfield->flag.height,
		     GDK_RGB_DITHER_NORMAL, 0, 0);
  }
  if (mfield->lose && mfield->mines[c].mined && mfield->mines[c].shown) {
    gdk_draw_pixbuf (window, NULL,
		     mfield->bang.scaledpixbuf, 0, 0,
		     x * minesizepixels + (minesizepixels - mfield->bang.width) / 2,
		     y * minesizepixels + (minesizepixels - mfield->bang.height) / 2,
		     mfield->bang.width, mfield->bang.height,
		     GDK_RGB_DITHER_NORMAL, 0, 0);

  }
}

static gboolean
gtk_minefield_expose (GtkWidget * widget, GdkEventExpose * event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MINEFIELD (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (gtk_widget_is_drawable (widget)) {
    guint x1, y1, x2, y2, x, y;
    GtkMineField *mfield = GTK_MINEFIELD (widget);
    GdkRectangle *area = &event->area;

    /* mine square numbers must be resized to fit the mine size */
    gtk_minefield_setup_signs (mfield);
    gtk_minefield_setup_numbers (mfield);

    if (area) {
      x1 = area->x / mfield->minesizepixels;
      y1 = area->y / mfield->minesizepixels;
      x2 = (area->x + area->width - 1) / mfield->minesizepixels;
      y2 = (area->y + area->height - 1) / mfield->minesizepixels;
    } else {
      x1 = 0;
      y1 = 0;
      x2 = mfield->xsize - 1;
      y2 = mfield->ysize - 1;
    }

    /* These are necessary because we get an expose call before a
     * resize at the old size, but after we have changed our data. */
    if (x2 >= mfield->xsize)
      x2 = mfield->xsize - 1;
    if (y2 >= mfield->ysize)
      y2 = mfield->ysize - 1;

    for (x = x1; x <= x2; x++)
      for (y = y1; y <= y2; y++)
	gtk_mine_draw (mfield, x, y);
  }
  return FALSE;
}

static int
gtk_minefield_check_cell (GtkMineField * mfield, guint x, guint y)
{
  guint changed;
  gint c;
  guint i;
  gint nx, ny;

  changed = 0;

  for (i = 0; i < 8; i++) {
    nx = x + neighbour_map[i].x;
    ny = y + neighbour_map[i].y;
    if ((c = get_cell_index (mfield, nx, ny)) != -1) {
      if (mfield->mines[c].shown == 0 &&
	  mfield->mines[c].marked == MINE_NOMARK) {
	mfield->mines[c].shown = 1;
	mfield->shown++;
	gtk_mine_draw (mfield, nx, ny);
	changed = 1;
      }
    }
  }
  return changed;
}


static void
gtk_minefield_check_field (GtkMineField * mfield, gint x, gint y)
{
  gint c;
  guint changed;

  gint x1, y1, x2, y2;
  gint cx1, cx2, cy1, cy2;

  cx1 = cx2 = x;
  cy1 = cy2 = y;

  do {
    x1 = cx1 - 1;
    y1 = cy1 - 1;
    x2 = cx2 + 1;
    y2 = cy2 + 1;

    if (x1 < 0)
      x1 = 0;
    if (y1 < 0)
      y1 = 0;
    if (x2 >= mfield->xsize)
      x2 = mfield->xsize - 1;
    if (y2 >= mfield->ysize)
      y2 = mfield->ysize - 1;

    changed = 0;
    for (x = x1; x <= x2; x++) {
      for (y = y1; y <= y2; y++) {
	c = get_cell_index_no_checks (mfield, x, y);
	if (mfield->mines[c].neighbours == 0 && mfield->mines[c].shown == 1) {
	  changed |= gtk_minefield_check_cell (mfield, x, y);
	  if (changed) {
	    if (x < cx1)
	      cx1 = x;
	    if (x > cx2)
	      cx2 = x;
	    if (y < cy1)
	      cy1 = y;
	    if (y > cy2)
	      cy2 = y;
	  }
	}
      }
    }
  } while (changed);

  g_signal_emit (G_OBJECT (mfield),
		 minefield_signals[UNCOVER_SIGNAL], 0, NULL);

  if (mfield->shown == mfield->xsize * mfield->ysize - mfield->mcount) {
    gtk_minefield_win (mfield);
  }
}

static void
gtk_minefield_lose (GtkMineField * mfield)
{
  guint i, x, y;

  g_signal_emit (G_OBJECT (mfield),
		 minefield_signals[EXPLODE_SIGNAL], 0, NULL);

  mfield->lose = 1;

  /* draw mines and wrong markings */
  for (i = 0; i <mfield->xsize * mfield->ysize; i++) {
    if (mfield->mines[i].mined || mfield->mines[i].marked) {
      y = i / mfield->xsize;
      x = i % mfield->xsize;
      gtk_mine_draw (mfield, x, y);
    }
  }
}

static void
gtk_minefield_win (GtkMineField * mfield)
{
  guint x, y, c;

  /* mark any unmarked mines and update displayed total */
  for (x = 0; x < mfield->xsize; x++) {
    for (y = 0; y < mfield->ysize; y++) {
      c = x + y * mfield->xsize;
      if (mfield->mines[c].shown == 0 &&	/* not shown & not marked */
	  mfield->mines[c].marked != MINE_MARKED) {

	mfield->mines[c].marked = MINE_MARKED;	/* mark it */
	gtk_mine_draw (mfield, x, y);	/* draw it */
	mfield->flag_count++;	/* up the count */
	g_signal_emit (GTK_OBJECT (mfield),	/* display the count */
		       minefield_signals[MARKS_CHANGED_SIGNAL], 0, NULL);
      }
    }
  }


  mfield->win = 1;

  /* now stop the clock.  (MARKS_CHANGED_SIGNAL starts it) */
  /* Make sure this is the last thing called so it is safe to
   * start a new game in the win_signal handler. */
  g_signal_emit (GTK_OBJECT (mfield), minefield_signals[WIN_SIGNAL], 0, NULL);
}

static void
gtk_minefield_randomize (GtkMineField * mfield, int curloc)
{
  guint i, j;
  guint x, y;
  guint n;
  gint cidx;
  gboolean adj_found;

  /*if(specialMap) {
    mfield->mcount = 0;
    for(i = 0; i< mfield->xsize*mfield->ysize; i++) {
        mfield->mines[i].mined = false;
    }
    mfield->mines[3].mined = true;
    mfield->mines[9].mined = true;
    mfield->mines[15].mined = true;
    mfield->mines[21].mined = true;
    mfield->mines[22].mined = true;
    mfield->mines[23].mined = true;
    mfield->mines[10].mined = true;

    mfield->mcount = 7;
    
  } else {*/
    /* randomly set the mines, but avoid the current and adjacent locations */

    x = curloc % mfield->xsize;
    y = curloc / mfield->xsize;

    for (n = 0; n < mfield->mcount;) {
      i = g_rand_int_range (mfield->grand, 0, mfield->xsize * mfield->ysize);

      if (!mfield->mines[i].mined && i != curloc) {
        adj_found = FALSE;

        for (j = 0; j < 8; j++)
	  adj_found |=
	    (i ==
	     get_cell_index (mfield, x + neighbour_map[j].x, y + neighbour_map[j].y));

        if (!adj_found) {
  	  mfield->mines[i].mined = 1;
	  n++;
        }
      }
    }

  /* load neighborhood numbers */
  for (x = 0; x < mfield->xsize; x++) {
    for (y = 0; y < mfield->ysize; y++) {
      n = 0;
      for (i = 0; i < 8; i++) {
	if (((cidx = get_cell_index (mfield, x + neighbour_map[i].x,
			       y + neighbour_map[i].y)) != -1) &&
	    mfield->mines[cidx].mined) {
	  n++;
	}
      }
      mfield->mines[x + mfield->xsize * y].neighbours = n;

      n = 0;
      for (i = 0; i < 8; i++) {
	if (((cidx = get_cell_index (mfield, x + neighbour_map[i].x,
			       y + neighbour_map[i].y)) != -1) &&
	    mfield->mines[cidx].marked == MINE_MARKED) {
	  n++;
	}
      }
      mfield->mines[x + mfield->xsize * y].neighbourmarks = n;
    }
  }
}

static void
gtk_minefield_show (GtkMineField * mfield, guint x, guint y)
{
  int c = get_cell_index (mfield, x, y);

  g_return_if_fail (c != -1);

  /* make sure first click isn't on a mine */
  if (!mfield->in_play) {
    mfield->in_play = 1;
    gtk_minefield_randomize (mfield, c);
  }

  if (mfield->mines[c].marked != MINE_MARKED && mfield->mines[c].shown != 1) {
    mfield->mines[c].shown = 1;
    mfield->shown++;
    gtk_mine_draw (mfield, x, y);

    if (mfield->mines[c].mined == 1) {
      gtk_minefield_lose (mfield);
    } else {
      gtk_minefield_check_field (mfield, x, y);
    }

    /*std::cout << "The minefield is currently:" << std::endl;
    for(int y = 0; y < mfield->ysize; y++) {
      for(int x = 0; x < mfield->xsize; x++) {
        mine m = mfield->mines[y*mfield->xsize + x];
        if(m.shown) {
	  std::cout << m.neighbours;
	} else {
	  std::cout << "-";
        }
      }
      std::cout << std::endl;
    }*/
  }
}

static void
gtk_minefield_set_mark (GtkMineField * mfield, guint x, guint y, int mark)
{
  int c = get_cell_index (mfield, x, y);
  int change_count, i, nx, ny, c2;
  gboolean was_valid, is_valid;
   
  g_return_if_fail (c != -1);

  /* Cannot mark if square already revealed */
  if (mfield->mines[c].shown != 0)
    return;

  /* Don't change if already has this mark */
  if (mfield->mines[c].marked == mark)
    return;
  /* Decide if we are adding or removing a mark */
  if (mark == MINE_MARKED) {
    change_count = 1;
  } else {
    if (mfield->mines[c].marked == MINE_MARKED)
      change_count = -1;
    else
      change_count = 0;
  }
   
  /* If mark count changed update counters in adjacent squares */
  if (change_count != 0)
  {
    mfield->flag_count += change_count;
    for (i = 0; i < 8; i++) {
      nx = x + neighbour_map[i].x;
      ny = y + neighbour_map[i].y;
      if ((c2 = get_cell_index (mfield, nx, ny)) == -1)
	continue;
       
      was_valid = mfield->mines[c2].neighbourmarks <= mfield->mines[c2].neighbours;
      mfield->mines[c2].neighbourmarks += change_count;
      is_valid = mfield->mines[c2].neighbourmarks <= mfield->mines[c2].neighbours;

      /* Redraw if too many marks placed */
      if (is_valid != was_valid)
	gtk_mine_draw (mfield, nx, ny);
    }
  }

  /* Update marking */
  mfield->mines[c].marked = mark;
  gtk_mine_draw (mfield, x, y);
  g_signal_emit (GTK_OBJECT (mfield),
		 minefield_signals[MARKS_CHANGED_SIGNAL], 0, NULL);
}
    
static void
gtk_minefield_toggle_mark (GtkMineField * mfield, guint x, guint y)
{
  int mark = MINE_NOMARK, c = get_cell_index (mfield, x, y);

  switch (mfield->mines[c].marked) {
  case MINE_NOMARK:
    /* If we've used all the flags don't plant any more,
     * this should be an indication to the player that they
     * have made a mistake. */
    if (mfield->flag_count == mfield->mcount && mfield->use_question_marks)
      mark = MINE_QUESTION;
    else
      mark = MINE_MARKED;
    break;

  case MINE_MARKED:
    if (mfield->use_question_marks)
      mark = MINE_QUESTION;
    break;
  }

  gtk_minefield_set_mark (mfield, x, y, mark);
}

static void
gtk_minefield_multi_press (GtkMineField * mfield, guint x, guint y, gint c)
{
  guint i;
  gint nx, ny, c2;

  for (i = 0; i < 8; i++) {
    nx = x + neighbour_map[i].x;
    ny = y + neighbour_map[i].y;
    if ((c2 = get_cell_index (mfield, nx, ny)) == -1)
      continue;
    if (mfield->mines[c2].marked != MINE_MARKED && !mfield->mines[c2].shown) {
      mfield->mines[c2].down = 1;
      gtk_mine_draw (mfield, nx, ny);
    }
  }
  mfield->multi_mode = 1;
}

static gboolean
gtk_minefield_solve_square (GtkMineField * mfield, guint x, guint y, guint c)
{
   gint nc, i, nx, ny, empty_count = 0, unknown[8][2], set_count = 0;

   /* Look for unmarked neighbour squares */
   for(i = 0; i < 8; i++) {
      nx = x + neighbour_map[i].x;
      ny = y + neighbour_map[i].y;
      nc = get_cell_index (mfield, nx, ny);
      if(nc < 0)
	continue;
      if(!mfield->mines[nc].shown) {
	 if(mfield->mines[nc].marked != MINE_MARKED) {
	    unknown[set_count][0] = nx;
	    unknown[set_count][1] = ny;
	    set_count++;
	 }
	 empty_count++;
      }
   }

   if(mfield->mines[c].neighbours != empty_count || set_count == 0)
     return FALSE;

   for(i = 0; i < set_count; i++)
     gtk_minefield_set_mark (mfield, unknown[i][0], unknown[i][1], MINE_MARKED);
   
   return TRUE;
}

static void
gtk_minefield_multi_release (GtkMineField * mfield, guint x, guint y, guint c,
			     guint really)
{
  gint nx, ny, i, c2;
  guint lose = 0;

  if (c >= mfield->xsize * mfield->ysize || mfield->analyzeMode) /* The release was outside the main area. */
    return;

  mfield->multi_mode = 0;

  std::cout << really << std::endl;
  if(mfield->mines[c].neighbours != mfield->mines[c].neighbourmarks) {std::cout << "smt with neighbors" << std::endl;}
  if(mfield->mines[c].marked == MINE_MARKED) {std::cout << "mine is marked" << std::endl;}
  if(!mfield->mines[c].shown) {std::cout <<"mine not shown" << std::endl;}

  if (mfield->mines[c].neighbours != mfield->mines[c].neighbourmarks ||
      mfield->mines[c].marked == MINE_MARKED || !mfield->mines[c].shown)
    really = 0;

  for (i = 0; i < 8; i++) {
    nx = x + neighbour_map[i].x;
    ny = y + neighbour_map[i].y;
    if ((c2 = get_cell_index (mfield, nx, ny)) == -1)
      continue;
    if (mfield->mines[c2].down) {
      mfield->mines[c2].down = 0;
      if (really && (mfield->mines[c2].shown == 0)) {
        std::cout << "here i should be " << std::endl;
	mfield->mines[c2].shown = 1;
	mfield->shown++;
	if (mfield->mines[c2].mined == 1) {
	  lose = 1;
	}
      }
      gtk_mine_draw (mfield, nx, ny);
    }
  }
  if (lose) {
    gtk_minefield_lose (mfield);
  } else if (really) {
    gtk_minefield_check_field (mfield, x, y);
  }
}

static gint
gtk_minefield_motion_notify (GtkWidget * widget, GdkEventMotion * event)
{
  GtkMineField *mfield;
  guint x, y;
  gint c;
  guint minesizepixels;

  g_return_val_if_fail (widget != NULL, 0);
  g_return_val_if_fail (GTK_IS_MINEFIELD (widget), 0);
  g_return_val_if_fail (event != NULL, 0);

  mfield = GTK_MINEFIELD (widget);

  minesizepixels = mfield->minesizepixels;

  if (mfield->lose || mfield->win)
    return FALSE;

  x = event->x / minesizepixels;
  y = event->y / minesizepixels;

  c = get_cell_index (mfield, x, y);
  if (c == -1)
    return 0;

  /* If mouse pointer over a cell that wasn't first pressed => dragging.*/
  if (c != mfield->celldown) {
    /* If left or middle mouse button down. */
    if (mfield->buttondown[0] || mfield->buttondown[1]) {
      mfield->mines[mfield->celldown].down = 0;
      gtk_mine_draw (mfield, mfield->celldownx, mfield->celldowny);

      if (mfield->multi_mode)
	gtk_minefield_multi_release (mfield, mfield->celldownx,
				     mfield->celldowny, mfield->celldown, 0);
      mfield->celldownx = x;
      mfield->celldowny = y;
      mfield->celldown = c;
      mfield->mines[c].down = 1;
      gtk_mine_draw (mfield, x, y);

      /* Clear action is active and the current cell is shown. */
      if (mfield->action == CLEAR_ACTION && mfield->mines[c].shown)
        gtk_minefield_multi_press (mfield, x, y, c);
    } else if (mfield->buttondown[2]) {
      /*  Update clicked field on right click drag.*/
      mfield->mines[mfield->celldown].down = 0;
      mfield->action = NO_ACTION;
      gtk_mine_draw (mfield, mfield->celldownx, mfield->celldowny);

      mfield->celldownx = x;
      mfield->celldowny = y;
      mfield->celldown = c;
      mfield->mines[c].down = 1;
    }
  }
  return FALSE;
}

static gint
gtk_minefield_button_press (GtkWidget * widget, GdkEventButton * event)
{
  GtkMineField *mfield;
  guint x, y;
  gint c;
  guint minesizepixels;

  g_return_val_if_fail (widget != NULL, 0);
  g_return_val_if_fail (GTK_IS_MINEFIELD (widget), 0);
  g_return_val_if_fail (event != NULL, 0);

  mfield = GTK_MINEFIELD (widget);

  minesizepixels = mfield->minesizepixels;

  if (mfield->lose || mfield->win)
    return FALSE;

  /* Left or right mouse button has been clicked. */
  if (event->button <= 3 && !mfield->buttondown[1]) {
    /* Translate mouse coordinates to minefield coordinates
     * and do some sanity checking. */
    x = event->x / minesizepixels;
    y = event->y / minesizepixels;
    c = get_cell_index (mfield, x, y);
    if (c == -1)
      return FALSE;
      
    /* If this is the first button pressed (on a cell),
     * record where it was pressed. */
    if (!mfield->buttondown[0] && !mfield->buttondown[1]) {
      mfield->celldownx = x;
      mfield->celldowny = y;
      mfield->celldown = c;
      mfield->mines[c].down = 1;
    }
    mfield->buttondown[event->button - 1]++;

    /* Redraw the cell, which is now being pressed. */
    gtk_mine_draw (mfield, x, y);

    /* Determine what action to do. Normally this is
     * left button = show, middle = clear and right = flag.
     * Unfortunately we have to detect left+right because
     * MS Minesweeper did this and some people will be used to
     * it. As well as that left + shift is also clear for people
     * with two button mice and less dexterity. In addition
     * we also want left = clear when that is the only 
     * reasonable action (i.e. we click on a cleared square)
     * since this makes it even easier for two-button mice,
     * but we didn't think of it soon enough not to have to worry 
     * about all the extra legacy crap. */
    switch (event->button) {
    case 1: /* Left click. */
      mfield->action = SHOW_ACTION;
      if ((event->state & GDK_SHIFT_MASK) || (mfield->buttondown[2]) || (mfield->mines[c].shown))
        mfield->action = CLEAR_ACTION;
      /* Ctrl + left = right to make game playable on touchpad */
      if (event->state & GDK_CONTROL_MASK)
        mfield->action = FLAG_ACTION;
      break;
    case 2: /* Middle click. */
      mfield->action = CLEAR_ACTION;
      break;
    case 3: /* Right click. */
      mfield->action = FLAG_ACTION;
      if (mfield->buttondown[0])
        mfield->action = CLEAR_ACTION;
      break;
    }

    /* Now actually do the actions. Most of the real work
     * is done in the button_release handler. */
    if (mfield->action == CLEAR_ACTION) {
      gtk_minefield_multi_press (mfield, x, y, c);
    } else if (mfield->action == FLAG_ACTION) {
      if (mfield->buttondown[2] == 1 || (event->state & GDK_CONTROL_MASK && mfield->buttondown[0] == 1)) {
        gtk_minefield_toggle_mark (mfield, x, y);
      }
    }
    if (mfield->action != FLAG_ACTION) {
      g_signal_emit (GTK_OBJECT (mfield),
		     minefield_signals[LOOK_SIGNAL], 0, NULL);
    }
  }
  return FALSE;
}

static gint
gtk_minefield_button_release (GtkWidget * widget, GdkEventButton * event)
{
  GtkMineField *mfield;
  gboolean really;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_MINEFIELD (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  mfield = GTK_MINEFIELD (widget);

  if (mfield->lose || mfield->win)
    return FALSE;

  /* If mouse button released and gtk_minefield_button_press caught it too. */
  if (event->button <= 3 && mfield->buttondown[event->button - 1]) {

    if(mfield->analyzeMode == true && event->button == 1) {
      g_signal_emit (GTK_OBJECT (mfield),
		 minefield_signals[ANALYZE_SIGNAL], 0, NULL);
      mfield->mines[mfield->celldown].down = 0;
      mfield->action = NO_ACTION;
      mfield->buttondown[event->button - 1] = 0;
      gtk_mine_draw (mfield, mfield->celldownx, mfield->celldowny);
      for(int i = 0; i < 8; i++) {
        int c2;
        int nx = mfield->celldownx + neighbour_map[i].x;
        int ny = mfield->celldowny + neighbour_map[i].y;
        if ((c2 = get_cell_index (mfield, nx, ny)) == -1) { continue; }
        if (mfield->mines[c2].down) {
          mfield->mines[c2].down = 0;
	  gtk_mine_draw (mfield, nx, ny);
        }
      }
      g_signal_emit (GTK_OBJECT (mfield),
		     minefield_signals[UNLOOK_SIGNAL], 0, NULL);
      return FALSE;
    }

    switch (mfield->action) {
    case SHOW_ACTION:
      gtk_minefield_show (mfield, mfield->celldownx, mfield->celldowny);
      break;
    case CLEAR_ACTION:
       if (mfield->use_autoflag)
         really = ! gtk_minefield_solve_square (mfield, mfield->celldownx,
                        mfield->celldowny, mfield->celldown);
       else
	     really = TRUE;
       gtk_minefield_multi_release (mfield, mfield->celldownx, mfield->celldowny,
				      mfield->celldown, really);
       break;
    }
    if (!mfield->lose && !mfield->win) {
      g_signal_emit (GTK_OBJECT (mfield),
		     minefield_signals[UNLOOK_SIGNAL], 0, NULL);
    }
    mfield->mines[mfield->celldown].down = 0;
    mfield->action = NO_ACTION;
    mfield->buttondown[event->button - 1] = 0;
    gtk_mine_draw (mfield, mfield->celldownx, mfield->celldowny);
  }
  return FALSE;
}


static void
gtk_minefield_class_init (GtkMineFieldClass * class_)
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class_);
  GtkObjectClass *object_class = GTK_OBJECT_CLASS (class_);

  parent_class = g_type_class_peek_parent (class_);

  widget_class->realize = gtk_minefield_realize;
  widget_class->unrealize = gtk_minefield_unrealize;
  widget_class->size_allocate = gtk_minefield_size_allocate;
  widget_class->size_request = gtk_minefield_size_request;
  widget_class->expose_event = gtk_minefield_expose;
  widget_class->button_press_event = gtk_minefield_button_press;
  widget_class->button_release_event = gtk_minefield_button_release;
  widget_class->motion_notify_event = gtk_minefield_motion_notify;

  class_->marks_changed = NULL;
  class_->explode = NULL;
  class_->look = NULL;
  class_->unlook = NULL;
  class_->win = NULL;
  class_->analyze_toggle = NULL;
  class_->uncover_action = NULL;

  minefield_signals[MARKS_CHANGED_SIGNAL] =
    g_signal_new ("marks_changed",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkMineFieldClass, marks_changed),
		  NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

  minefield_signals[EXPLODE_SIGNAL] =
    g_signal_new ("explode",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkMineFieldClass, explode),
		  NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  minefield_signals[LOOK_SIGNAL] =
    g_signal_new ("look",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkMineFieldClass, look),
		  NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  minefield_signals[UNLOOK_SIGNAL] =
    g_signal_new ("unlook",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkMineFieldClass, unlook),
		  NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  minefield_signals[WIN_SIGNAL] =
    g_signal_new ("win",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkMineFieldClass, win),
		  NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
  minefield_signals[HINT_SIGNAL] =
    g_signal_new ("hint-used",
		  G_OBJECT_CLASS_TYPE (object_class),
		  G_SIGNAL_RUN_FIRST,
		  G_STRUCT_OFFSET (GtkMineFieldClass, hint_used),
		  NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

  minefield_signals[ANALYZE_SIGNAL]  =
    g_signal_new("analyze-toggle",
  	         G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMineFieldClass, analyze_toggle),
		 NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

  minefield_signals[UNCOVER_SIGNAL]  =
    g_signal_new("uncover-action",
  	         G_OBJECT_CLASS_TYPE(object_class),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(GtkMineFieldClass, uncover_action),
		 NULL, NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

static void
gtk_minefield_init (GtkMineField * mfield)
{
  mfield->xsize = 0;
  mfield->ysize = 0;
  mfield->mines = NULL;
  mfield->started = FALSE;
  mfield->celldown = -1;
  mfield->action = NO_ACTION;

  mfield->flag.preimage = NULL;
  mfield->mine.preimage = NULL;
  mfield->question.preimage = NULL;
  mfield->bang.preimage = NULL;
  mfield->warning.preimage = NULL;
  mfield->grand = g_rand_new ();
  mfield->thick_line = NULL;
  mfield->analyzeMode = false;
  mfield->applyingChanges = false;
}

void
gtk_minefield_set_size (GtkMineField * mfield, guint xsize, guint ysize)
{
  g_return_if_fail (mfield != NULL);
  g_return_if_fail (GTK_IS_MINEFIELD (mfield));

  if ((mfield->xsize == xsize) && (mfield->ysize == ysize))
    return;

  mfield->mines = g_realloc (mfield->mines, sizeof (mine) * xsize * ysize);

  mfield->xsize = xsize;
  mfield->ysize = ysize;

  if (gtk_widget_get_visible (GTK_WIDGET (mfield))) {
    gtk_widget_queue_resize (GTK_WIDGET (mfield));
  }
}

GtkWidget *
gtk_minefield_new (void)
{
  return GTK_WIDGET (g_object_new (GTK_TYPE_MINEFIELD, NULL));
}

GType
gtk_minefield_get_type (void)
{
  static GType minefield_type = 0;

  if (minefield_type == 0) {
    static const GTypeInfo minefield_info = {
      sizeof (GtkMineFieldClass),
      NULL,			/* base_init */
      NULL,			/* base_finalize */
      (GClassInitFunc) gtk_minefield_class_init,
      NULL,			/* class_finalize */
      NULL,			/* class_data */
      sizeof (GtkMineField),
      0,			/* n_preallocs */
      (GInstanceInitFunc) gtk_minefield_init,
      NULL,			/* value table */
    };

    minefield_type = g_type_register_static (GTK_TYPE_WIDGET,
					     "GtkMineField",
					     &minefield_info, 0);
  }

  return minefield_type;
}

void
gtk_minefield_restart (GtkMineField * mfield)
{
  guint i;

  g_return_if_fail (mfield != NULL);
  g_return_if_fail (GTK_IS_MINEFIELD (mfield));

  mfield->flag_count = 0;
  mfield->shown = 0;
  mfield->lose = 0;
  mfield->win = 0;
  mfield->buttondown[0] = 0;
  mfield->buttondown[1] = 0;
  mfield->buttondown[2] = 0;
  mfield->celldown = -1;
  mfield->multi_mode = 0;
  mfield->in_play = 0;

  for (i = 0; i < mfield->xsize * mfield->ysize; i++) {
    mfield->mines[i].marked = MINE_NOMARK;
    mfield->mines[i].mined = 0;
    mfield->mines[i].shown = 0;
    mfield->mines[i].down = 0;
  }

  if (mfield->started == FALSE)
    mfield->started = TRUE;
  else
    gtk_widget_queue_draw (GTK_WIDGET (mfield));
}

void
gtk_minefield_set_use_question_marks (GtkMineField * mfield,
				      gboolean use_question_marks)
{
  g_return_if_fail (mfield != NULL);
  g_return_if_fail (GTK_IS_MINEFIELD (mfield));

  mfield->use_question_marks = use_question_marks;
}

void
gtk_minefield_set_use_overmine_warning (GtkMineField * mfield,
					gboolean use_overmine_warning)
{
  g_return_if_fail (mfield != NULL);
  g_return_if_fail (GTK_IS_MINEFIELD (mfield));

  mfield->use_overmine_warning = use_overmine_warning;

  gtk_widget_queue_draw (GTK_WIDGET (mfield));
}

void
gtk_minefield_set_use_autoflag (GtkMineField * mfield,
				gboolean use_autoflag)
{
  g_return_if_fail (mfield != NULL);
  g_return_if_fail (GTK_IS_MINEFIELD (mfield));

  mfield->use_autoflag = use_autoflag;

  gtk_widget_queue_draw (GTK_WIDGET (mfield));
}

/* Hunt for a hint to give the player. Revealed squares are handled here,
 * everything else is passed back up. The comments below detail the
 * strategy for revealing squares. */
gint
gtk_minefield_hint (GtkMineField * mfield)
{
  gint i, x, y;
  gint a, c;
  mine *m;
  guint ncase1, ncase2, ncase3;
  guint *case1list, *case2list, *case3list;
  guint *case1ptr, *case2ptr, *case3ptr;
  gint size;
  gint retval;

  g_return_val_if_fail (mfield != NULL, MINEFIELD_HINT_NO_GAME);
  g_return_val_if_fail (GTK_IS_MINEFIELD (mfield), MINEFIELD_HINT_NO_GAME);
  if (!mfield->in_play)
    return MINEFIELD_HINT_NO_GAME;

  /* We search for three cases:
   *
   * Case 1: we look for squares adjacent to both a mine and a revealed
   * square since these are most likely to help the player and resolve
   * ambiguous situations.
   *
   * Case 2: we look for squares that are adjacent to a mine
   * (this will only occur in the rare case that a square is completely
   * encircled by mines, but at that point this case is probably
   * useful).
   *
   * Case 3: we look for any unrevealed square without a mine (as a
   * consequence of the previous cases this won't be adjacent to a
   * mine).
   */

  /* This code is pretty diabolical,
   * Yet it is perfectly logical,
   * Is it C ?
   * Is it me ?
   * Or is it just pathological ?
   */

  size = mfield->xsize * mfield->ysize;
  case1ptr = case1list = g_malloc (size * sizeof (guint));
  case2ptr = case2list = g_malloc (size * sizeof (guint));
  case3ptr = case3list = g_malloc (size * sizeof (guint));
  ncase1 = ncase2 = ncase3 = 0;

  for (i = 0; i < size; i++) {
    m = mfield->mines + i;
    if (!m->mined && !m->marked && !m->shown) {
      ncase3++;
      *case3ptr++ = i;
      if (m->neighbours > 0) {
	ncase2++;
	*case2ptr++ = i;
	x = i % mfield->xsize;
	y = i / mfield->ysize;
	for (a = 0; a < 8; a++) {
	  c = get_cell_index (mfield,
			x + neighbour_map[a].x, y + neighbour_map[a].y);
	  if ((c != -1) && mfield->mines[c].shown) {
	    ncase1++;
	    *case1ptr++ = i;
	    break;
	  }
	}
      }
    }
  }

  if (ncase1 > 0) {
    a = g_rand_int_range (mfield->grand, 0, ncase1);
    i = case1list[a];
  } else if (ncase2 > 0) {
    a = g_rand_int_range (mfield->grand, 0, ncase2);
    i = case2list[a];
  } else if (ncase3 > 0) {
    a = g_rand_int_range (mfield->grand, 0, ncase3);
    i = case3list[a];
  } else {
    retval = MINEFIELD_HINT_ALL_MINES;
    goto cleanup;
  }

  x = i % mfield->xsize;
  y = i / mfield->xsize;

  /* Makes sure that the program knows about the successful
   * hint before a possible win. */
  g_signal_emit (GTK_OBJECT (mfield),
		 minefield_signals[HINT_SIGNAL], 0, NULL);
  gtk_minefield_show (mfield, x, y);
  gtk_mine_draw (mfield, x, y);
  retval = MINEFIELD_HINT_ACCEPTED;

cleanup:
  g_free (case1list);
  g_free (case2list);
  g_free (case3list);

  return retval;
}
