#include <math.h>
#include <vector>
#include <iostream>
#include "gnomineSolver_CSP.h"
#include <string.h>
#include <sstream>

#include <stdlib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

enum backgroundOptions {
  VALUE_KNOWN = 0,
  VALUE_UNKNOWN,
  ACTIVE_VARIABLE,
  ACTIVE_CONSTRAINT,
  COMBINATION_POSSIBLE,
  COMBINATION_IMPOSSIBLE
};

static int background_colors[6][3] = {
  {0x66, 0xaa, 0xff},	/* light Blue  */
  {0xff, 0xaa, 0x66},   /* light Orange */
  {0xff, 0xff, 0x99},   /* lightish Yellow */
  {0x66, 0x66, 0x7f},	/* Darkish Blue */
  {0x00, 0xff, 0x00},	/* Green */
  {0xff, 0x00, 0x00},	/* Red   */
};


GnomineSolver_CSP::GnomineSolver_CSP(GtkMineField *mfield) {
  this->mfield=mfield;
  buildExtraGUI();
  int size = mfield->xsize * mfield->ysize;
  mfieldValues = new int[size];
  for(int i = 0; i < size; i++) {
      mfieldValues[i] = -1;
  }
  resetInfo();
  createCurrentState();
}

GnomineSolver_CSP::~GnomineSolver_CSP() {
  gtk_widget_destroy(gui.window);
  delete mfieldValues;
  mfield = NULL;
}




/***************
 *check Methods*
 ***************/

void GnomineSolver_CSP::analyzeField(int field) {
  nextConstraint(false, field);
};

/**
 * checks all Constraints on the current Minefield
 */
void GnomineSolver_CSP::checkMField() {
  resetInfo();
  createCurrentState();
  currentState.activeConstraints = getActiveConstraints();

  //loop over all Constraints
  for(std::vector<int>::iterator it= currentState.activeConstraints.begin(); it != currentState.activeConstraints.end(); ++it) {
    std::cout << "handling Constraint at position " << *it << std::endl;
    currentState.uncertainValues = prepareConstraint(*it);
    int size = currentState.uncertainValues.size();
    int erasedElements = 0;


    //copy uncertainValues in an array so we can loop over the array easy while changing the uncertainValues vector
    int uCArray [size];
    int i = 0;
    for(std::vector<int>::iterator it2 = currentState.uncertainValues.begin(); it2 != currentState.uncertainValues.end(); ++it2) {
      uCArray[i] = *it2;
      i++;
    }

    //loop over all uncertain Values
    for(i = 0; i < size; i++) {
      bool changed = checkVariable(uCArray[i], currentState.uncertainValues);
      if(changed) {
    currentState.uncertainValues.erase(currentState.uncertainValues.begin()+i-erasedElements);
    erasedElements++;
    int c = getNeighbour(*it, uCArray[i]);
    int val = info.numbers[uCArray[i]];
    std::cout << "val has value " << val << std::endl;
    mfieldValues[c] = val ;
    gtk_minefield_setSolution(GTK_MINEFIELD(mfield), c, val);


      }
    }
  }
  std::cout << "finished checking fields!" << std::endl;
};

/*
 * checks for the given Variable whether it can be assigned a definite Value with the constraint given
 */
bool GnomineSolver_CSP::checkVariable(int var, std::vector<int> uncertainValues) {
  prepareVariable(var);
  std::cout << "Testing variable " << var << std::endl;
  std::cout << "Testing value 0" << std::endl;
  fillNumberArray(info.numbers, -1, uncertainValues);
  info.numbers[var] = 0;
  if(!validCombinationExists(info.numbers, info.sum, uncertainValues)) { //if there is no valid combination x must be the other value!
    info.numbers[var] = 1;
    std::cout << "the field at position " << var << " must be 1 " << std::endl;
    return true;
  }
  std::cout << "testing value 1" << std::endl;
  fillNumberArray(info.numbers, -1, uncertainValues);
  info.numbers[var] = 1;
  if(!validCombinationExists(info.numbers, info.sum, uncertainValues)) {
    info.numbers[var] = 0;
    std::cout << "the field at position " << var << " must be 0" << std::endl;
    return true;
  }
  // both values have valid combinations --> x[var] stays unsure
  info.numbers[var] = -1;
  return false;
};

/*
 * checks for the given neighbour-array with the given uncertain Values if there exists an assignment that will satisfy
 * the constraint sum(arrayelements) = correct_sum
 */
bool GnomineSolver_CSP::validCombinationExists(int numbers[8], int correct_sum, std::vector<int>  uncertainValues) {
  int orig_numbers[8];
  memcpy(orig_numbers,numbers,sizeof(orig_numbers));
  info.tested_combinations = 0;

  for(uint i = 0; i < (uint) info.possible_combinations; i++) {
    memcpy(numbers, orig_numbers, sizeof(numbers));
    if(info.possible_combinations > 1) {
      fillNumberArray(numbers, i, uncertainValues);
    }
    bool valid = combinationIsValid(numbers, correct_sum);
    info.tested_combinations++;
    if(valid) {
      std::cout << "a valid combination exists" << std::endl;
      return true;
    }
  }
  std::cout << "no valid combination possible" << std::endl;
  return false;
};

/*
 * checks whether the given assignment in numbers[8] is valid, meaning that the sum of this array equals the parameter correct_sum
 */
bool GnomineSolver_CSP::combinationIsValid(int numbers[8], int correct_sum) {
  
  int sum = 0;  
  for(int i = 0; i < 8; i++) {
    sum += numbers[i];
  }
  if(sum == correct_sum) {
    info.active_constraint_state = COMBINATION_POSSIBLE;
    gtk_extra_gui_update();
    return true;
  } else {
    info.active_constraint_state = COMBINATION_IMPOSSIBLE;
    gtk_extra_gui_update();
    return false;
  }
};



/*****************
 *prepare Methods*
 *****************/

/*
 * prepares class-Variables (for current global state and gui) to fit to the given active Constraint
 * should always be called when the active Constraint changes
 */
std::vector<int> GnomineSolver_CSP::prepareConstraint(int c) {
  resetInfo();
  gtk_minefield_setupBgColors(mfield);
  std::vector<int> uncertainValues;
  info.sum = mfield->mines[c].neighbours;

  //setup the number and states array in info
  for(int i = 0; i < 8; i++) {
    int neighbour_c = getNeighbour(c,i);
    if (neighbour_c > -1) {
      int n = mfieldValues[neighbour_c];
      if(n == 0) {
	info.numbers[i] = 0;
	info.states[i] = VALUE_KNOWN;
      } else if(n == 1) {
	info.numbers[i] = 1;
	info.states[i] = VALUE_KNOWN;
      } else {
	uncertainValues.push_back(i);
	info.numbers[i] = -1;
	info.states[i] = VALUE_UNKNOWN;
      }
    } else {
      info.numbers[i] = 0;
      info.states[i] = VALUE_KNOWN;
    }
  }

  //set the background for the original minefield
  for(int i = 0; i < 8; i++) {
    int neighbour_c = getNeighbour(c,i);
    if(info.numbers[i] != -1) { //we know the value of xi
      setBackgroundMinefield(neighbour_c,VALUE_KNOWN);
    } else {
      setBackgroundMinefield(neighbour_c,VALUE_UNKNOWN);
    }
  }
  setBackgroundMinefield(c,ACTIVE_CONSTRAINT);
  info.active_constraint_state = ACTIVE_CONSTRAINT;
  info.field = -1;
  gtk_extra_gui_update();

  return uncertainValues;
};

/*
 * prepares class-Variables (for current global state and gui) to fit to the given active Variable
 * should always be called when the active Variable changes
 */
void GnomineSolver_CSP::prepareVariable(int var) {
  for(int i  = 0; i < 8; i++) {
    if(info.numbers[i] > -1) {
      info.states[i] = VALUE_KNOWN;
    } else {
      info.states[i] = VALUE_UNKNOWN;
    }
  }
  info.field = var;
  info.states[var] = ACTIVE_VARIABLE;
  info.active_constraint_state = ACTIVE_CONSTRAINT;
  info.possible_combinations = pow(2,currentState.uncertainValues.size()-1);
  info.tested_combinations = 0;
  info.numbers[var] = -1;
  gtk_extra_gui_update();
};




/****************
 *helper Methods*
 ****************/

/*
 * finds all active constraints (meaning the fields that have neighbours that are neither defined as
 * mine nor as safe spot
 */
std::vector<int> GnomineSolver_CSP::getActiveConstraints() {
  std::vector<int> activeConstraints;
  for(int c = 0 ; c < (int) (mfield->xsize*mfield->ysize); c++) {
    if(mfield->mines[c].shown && hasUncertainNeighbours(c)) {
      activeConstraints.push_back(c);
      std::cout << c << " ";
    }
  }
  std::cout << std::endl;
  return activeConstraints;
};

/*
 * checks whether a field has any neigbours that are neither defined as mine nor as safe spot
 */
bool GnomineSolver_CSP::hasUncertainNeighbours(int c) {
  for(int i = 0; i < 8; i++) {
    int nb = getNeighbour(c,i);
    if(nb != -1) {
      if(mfieldValues[nb] == -1) {
      //if(! mfield->mines[nb].shown) {
	return true;
      }
    }
  }
  return false;
};


/*
 * fill in a possible assignment to all uncertain Values of the neighbour fields of the current constraint (=numbers[8]).
 * n is an encoding for the assignment of the uncertain Values (binary, eg for 6 uncertain Values: 5 = 01001 would mean first uncertain value is 0, second 1, ...)
 * the active Variable is also in the uncertainValues array, but we dont change it
 */
void GnomineSolver_CSP::fillNumberArray(int numbers[8], int n, std::vector<int> uncertainValues) {
  //n == -1 means we want to reset the array to its original condition (uncertain Values are represented by -1 since they are neither 0 nor 1)
  if(n == -1) {
    for(int i = 0; i < uncertainValues.size(); i++) {
      numbers[uncertainValues.at(i)] = -1;
    }
  } else { // n > -1
    std::cout << n << std::endl;
    bool activeVariablePassed = false;
    for(int i = 0; i < uncertainValues.size(); i++) {
      //if we are at the current active constraint, dont change the value!
      if(uncertainValues.at(i) == info.field) { 
        activeVariablePassed = true;
        continue; 
      }
      int j = i;
      if(activeVariablePassed) {
        j--;
      }
      numbers[uncertainValues.at(i)] = ( (n << (7-j)) %256 ) >> 7;
    }
    for(int i = 0; i < 8; i++) {
      std::cout << numbers[i] << " ";
    }std::cout << std::endl;
  }
};

/*************************
 *variable updates/resets*
 *************************/

/*
 * sets all attributes of info back to -1
 */
void GnomineSolver_CSP::resetInfo() {
  for(int i = 0; i < 8; i++) {
    info.numbers[i] = -1;
    info.states[i] = -1;
  }
  info.sum = -1;
  info.field = -1;
  info.possible_combinations = -1;
  info.tested_combinations = -1;
  info.active_constraint_state = -1;
};

/*
 * initialize the Variables of currentState (to -1, null, empty, ...)
 */
void GnomineSolver_CSP::createCurrentState() {
  currentState.activeConstraints.clear();
  currentState.uncertainValues.clear();
  currentState.activeConstraintsPos = -1;
  int size = mfield->xsize * mfield->ysize;
  for(int i = 0; i < size; i++) {
    if(mfield->mines[i].shown) {
      mfieldValues[i] = 0;
    } else if (mfield->mines[i].marked) {
      mfieldValues[i] = 1;
    } /*else {
      mfieldValues[i] = -1;
    }*/
  }
  currentState.valuePossible[0] = -1;
  currentState.valuePossible[1] = -1;
  gtk_extra_gui_update();
  gtk_minefield_setupBgColors(mfield);
};

/*
 * updates all changes on the actual Minefield (as defined in minefield.cpp
 * to our local copy of the minefield-states
 */
void GnomineSolver_CSP::updateMfieldValues() {
  for(int i = 0; i < mfield->xsize*mfield->ysize; i++) {
    if(mfieldValues[i] == -1 && mfield->mines[i].shown) {
      mfieldValues[i] = 0;
    } else if (mfieldValues[i] == -1 && mfield->mines[i].marked) {
      mfieldValues[i] = 1;
    } else if (mfield->mfieldSolution[i] == -1) {
        mfieldValues[i] = -1;
    }
  }
  resetInfo();
  createCurrentState();
  std::cout << "update Mfield values" << std::endl;
}

/*
 * tells the (original) minefield gui to change the bgcolor of mine c
 */
void GnomineSolver_CSP::setBackgroundMinefield(int c, int state) {
  if(c == -1) return;
  gtk_minefield_set_field_bg(mfield, background_colors[state], c);
};


/***********************
 *create/manipulate gui*
 ***********************/

/**
 * create extra gui for showing how the CSP algorithm works
 */
void GnomineSolver_CSP::buildExtraGUI() {
  gui.window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW (gui.window), "Constraint Satisfaction Problem Solver");
  gtk_window_set_default_size(GTK_WINDOW(gui.window), 500,500);
  gtk_widget_show(gui.window);

  GtkWidget *all_boxes = gtk_vbox_new(TRUE, 0);
  gtk_widget_show(all_boxes);
  gtk_container_add (GTK_CONTAINER(gui.window), all_boxes);

  GtkWidget *top_box = gtk_hbox_new(TRUE,0);
  gtk_box_pack_start(GTK_BOX(all_boxes), top_box, TRUE, TRUE, 10);
  gtk_widget_show(top_box);

  gui.fieldarea = gtk_drawing_area_new();
  g_signal_connect (GTK_OBJECT (gui.fieldarea), "expose_event", 
    G_CALLBACK (GnomineSolver_CSP::gtk_extra_gui_expose_proxy), this);
  gtk_box_pack_start(GTK_BOX(top_box), gui.fieldarea, TRUE, TRUE, 10);
  gtk_widget_show(gui.fieldarea);

  gui.textarea = gtk_label_new("test");
  gtk_box_pack_start(GTK_BOX(top_box), gui.textarea, TRUE, TRUE, 10);
  gtk_widget_show(gui.textarea);

  gui.buttonarea = gtk_vbox_new(TRUE,0);
  gtk_box_pack_start(GTK_BOX(all_boxes), gui.buttonarea, TRUE, TRUE, 10);
  gtk_widget_show(gui.buttonarea);

  GtkWidget * nextButtonarea = gtk_hbox_new(TRUE,0);
  gtk_box_pack_start(GTK_BOX(gui.buttonarea), nextButtonarea, TRUE, TRUE, 10);
  gtk_widget_show(nextButtonarea);

  GtkWidget * nextConstraintButton = gtk_button_new_with_label("next Constraint");
  GtkWidget * nextVariableButton = gtk_button_new_with_label("next Variable");
  GtkWidget * nextStepButton = gtk_button_new_with_label("next Step");
  gtk_box_pack_start(GTK_BOX(nextButtonarea), nextConstraintButton, TRUE, TRUE, 10);
  gtk_box_pack_start(GTK_BOX(nextButtonarea), nextVariableButton, TRUE, TRUE, 10);
  gtk_box_pack_start(GTK_BOX(nextButtonarea), nextStepButton, TRUE, TRUE, 10);
  gtk_widget_show(nextConstraintButton);
  gtk_widget_show(nextVariableButton);
  gtk_widget_show(nextStepButton);
  g_signal_connect(G_OBJECT(nextConstraintButton), "clicked", G_CALLBACK(GnomineSolver_CSP::nextConstraintCallback), this);
  g_signal_connect(G_OBJECT(nextVariableButton), "clicked", G_CALLBACK(GnomineSolver_CSP::nextVariableCallback), this);
  g_signal_connect(G_OBJECT(nextStepButton), "clicked", G_CALLBACK(GnomineSolver_CSP::nextStepCallback), this);

  GtkWidget * modifyarea = gtk_hbox_new(TRUE,0);
  gtk_box_pack_start(GTK_BOX(gui.buttonarea), modifyarea, TRUE, TRUE, 10);
  gtk_widget_show(modifyarea);

  GtkWidget * checkButton = gtk_button_new_with_label("check Field");
  GtkWidget * applyButton = gtk_button_new_with_label("apply Changes");
  gtk_box_pack_start(GTK_BOX(modifyarea), checkButton, TRUE, TRUE, 10);
  gtk_box_pack_start(GTK_BOX(modifyarea), applyButton, TRUE, TRUE, 10);
  gtk_widget_show(checkButton);
  gtk_widget_show(applyButton);
  g_signal_connect(G_OBJECT(checkButton), "clicked", G_CALLBACK(GnomineSolver_CSP::checkFieldCallback), this);
  g_signal_connect(G_OBJECT(applyButton), "clicked", G_CALLBACK(GnomineSolver_CSP::applyChangesCallback), this);

  GtkWidget * analyzearea = gtk_vbox_new(TRUE,0);
  gtk_box_pack_start(GTK_BOX(modifyarea), analyzearea, TRUE, TRUE, 10);
  gui.analyzeRB = gtk_radio_button_new_with_label(NULL, "analyze");
  gtk_widget_set_name(gui.analyzeRB, "analyzeRB");
  gtk_box_pack_start(GTK_BOX(analyzearea), gui.analyzeRB, TRUE, TRUE, 10);
  gui.playRB = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(gui.analyzeRB),"play");
  gtk_widget_set_name(gui.playRB, "playRB");
  gtk_box_pack_start(GTK_BOX(analyzearea), gui.playRB, TRUE, TRUE, 10);
  gtk_widget_show(gui.analyzeRB);
  gtk_widget_show(gui.playRB);
  gtk_widget_show(analyzearea);
  g_signal_connect(G_OBJECT(gui.analyzeRB), "clicked", G_CALLBACK(GnomineSolver_CSP::toggleAnalyzeModeCallback), this);
  g_signal_connect(G_OBJECT(gui.playRB), "clicked", G_CALLBACK(GnomineSolver_CSP::toggleAnalyzeModeCallback), this);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.playRB), TRUE);
};

gboolean GnomineSolver_CSP::gtk_extra_gui_update() {
  GtkWidget *widget = gui.fieldarea;
  cairo_t * cr = gdk_cairo_create(widget->window);

  int box_width = 210/3;
  int box_height = 210/3;


  for(int i = 0; i < 8; i++) {
    int x, y;
    switch(i) {
    case 0:
      x = 0*box_width; y = 0*box_height; break;
    case 1:
      x = 1*box_width; y = 0*box_height; break;
    case 2:
      x = 2*box_width; y = 0*box_height; break;
    case 3:
      x = 0*box_width; y = 1*box_height; break;
    case 4:
      x = 2*box_width; y = 1*box_height; break;
    case 5:
      x = 0*box_width; y = 2*box_height; break;
    case 6:
      x = 1*box_width; y = 2*box_height; break;
    case 7:
      x = 2*box_width; y = 2*box_height; break;
    }

    cairo_rectangle(cr, x, y, box_width, box_height);
    if(info.states[i] > -1) {
      cairo_set_source_rgb(cr, background_colors[info.states[i]][0]/255.0,
			 background_colors[info.states[i]][1]/255.0,
			 background_colors[info.states[i]][2]/255.0);
    } else {
      cairo_set_source_rgb(cr,0.5,0.5,0.5);
    }
    cairo_fill(cr);
    if(info.states[i] == VALUE_KNOWN) {
      cairo_set_source_rgb(cr, 0,0,0);
      cairo_move_to(cr, x+box_width/4, y+box_height-box_height/4);
      cairo_set_font_size(cr,50);
      if(info.numbers[i] == 0) {
	cairo_show_text(cr, "0");
      } else { 
	cairo_show_text(cr, "1");
      }

    } else { // state != VALUE_KNOWN
      if(info.numbers[i] == 0) {
	cairo_set_source_rgb(cr,0,0,0);
      } else {
	cairo_set_source_rgb(cr,0.5,0.5,0.5);
      }
      cairo_move_to(cr, x+box_width/4, y+box_height-box_height/4);
      cairo_set_font_size(cr, 30);
      cairo_show_text(cr, "0");

      if(info.numbers[i] == 0) {
	cairo_set_source_rgb(cr,0.5,0.5,0.5);
      }
      cairo_move_to(cr, x+box_width/2, y+box_height-box_height/4);
      cairo_show_text(cr, "/");

      if(info.numbers[i] == 1) {
	cairo_set_source_rgb(cr,0,0,0);
      }
      cairo_move_to(cr, x+box_width*3/5, y+box_height-box_height/4);
      cairo_show_text(cr, "1");

    }

  }

  cairo_rectangle(cr, box_width, box_height, box_width, box_height);
  if(info.active_constraint_state >-1) {
  cairo_set_source_rgb(cr, background_colors[info.active_constraint_state][0]/255.0,
			 background_colors[info.active_constraint_state][1]/255.0,
			 background_colors[info.active_constraint_state][2]/255.0);
  } else {
    cairo_set_source_rgb(cr, 0.5,0.5,0.5);
  }
  cairo_fill(cr);
  cairo_set_source_rgb(cr,0,0,0);
  cairo_set_font_size(cr,50);
  cairo_move_to(cr, box_width+box_width/4, 2*box_height-box_height/4);
  if(info.sum == -1) {
    cairo_show_text(cr, "?");
  } else {
    gpointer num[8] = {"1", "2", "3", "4", "5", "6", "7", "8"};
    cairo_show_text(cr, num[info.sum-1]);
  }
  cairo_destroy(cr);

  std::stringstream line1, line2, line3, line4, line5, line6;

  line1 << "examining Field " << info.field;
  line2;
  if(info.field == -1) {
    line2 << "Value: -1";
  } else {
    line2 << "Value: " << info.numbers[info.field];
  }
  line3 << "Possible Combinations: " << info.possible_combinations;
  line4 << "Tested Combinations: " << info.tested_combinations;
  std::string s;
  if(currentState.valuePossible[0] == -1) {
    s = "We dont know yet";
  } else if(currentState.valuePossible[0] == 0) {
    s = "Not possible";
  }else {
    s = "Possible";
  }
  line5 << "Value 0 possible: " << s;
   if(currentState.valuePossible[1]== -1) {
    s = "We dont know yet";
  } else if(currentState.valuePossible[1] == 0) {
    s = "Not possible";
  }else {
    s = "Possible";
  }
  line6 << "Value 1 possible: " << s;


  std::stringstream allLines;
  allLines  << line1.str() << "\n" << line2.str() << "\n" << line3.str() << "\n" << line4.str() << "\n" << line5.str() << "\n" << line6.str();

  gtk_label_set_label(GTK_LABEL(gui.textarea), allLines.str().c_str());


  return true;
};


/********************
 *callback functions*
 ********************/

/**
 *a hack so i can call the function without static and have access to member variables
 */
static gboolean GnomineSolver_CSP::gtk_extra_gui_expose_proxy(GtkWidget * widget, GdkEventExpose *event, gpointer data) {
  GnomineSolver_CSP *_this = static_cast<GnomineSolver_CSP*>(data);
  return _this->gtk_extra_gui_update();
};

static void GnomineSolver_CSP::checkField_proxy(void *arg) {
  return reinterpret_cast<GnomineSolver_CSP*>(arg)->checkMField();
};

static void GnomineSolver_CSP::nextConstraintCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_CSP *_this = static_cast<GnomineSolver_CSP*>(data);
  //_this->nextConstraint(false, 10); //test for jumpToConstraint
  _this->nextConstraint();
};

static void GnomineSolver_CSP::nextVariableCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_CSP *_this = static_cast<GnomineSolver_CSP*>(data);
  _this->nextVariable();
};

static void GnomineSolver_CSP::nextStepCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_CSP *_this = static_cast<GnomineSolver_CSP*>(data);
  _this->nextStep();
};

static void GnomineSolver_CSP::checkFieldCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_CSP *_this = static_cast<GnomineSolver_CSP*>(data);
  _this->checkMField();
};

static void GnomineSolver_CSP::applyChangesCallback(GtkWidget *widget, gpointer data) {
  GnomineSolver_CSP *_this = static_cast<GnomineSolver_CSP*>(data);
  _this->applyChanges();
};

static void GnomineSolver_CSP::toggleAnalyzeModeCallback(GtkWidget *widget, gpointer data) {
  if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) == TRUE) {
    return;
  }
  GnomineSolver_CSP *_this = static_cast<GnomineSolver_CSP*>(data);
if(g_strcmp0(gtk_widget_get_name(widget), "analyzeRB") == 0) {
    gtk_minefield_setAnalyzeMode(_this->mfield, true);
 } else if (g_strcmp0(gtk_widget_get_name(widget), "playRB") == 0) {
    gtk_minefield_setAnalyzeMode(_this->mfield, false);
  }
};

/*****************
 * button Methods*
 *****************/


/*
 *applies all the domain reductions the solver found out so far
 */
void GnomineSolver_CSP::applyChanges() {
  if(gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gui.analyzeRB)) == TRUE) {
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.playRB), TRUE);
    gtk_minefield_applyChanges(mfield);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(gui.analyzeRB), TRUE);
  } else {
    gtk_minefield_applyChanges(mfield);
  }
};

/*
 *prepares the next Constraint. If we looped through all the constraints, start at the first again
 */
void GnomineSolver_CSP::nextConstraint(bool refreshActiveConstraints, int jumpToConstraint) {
  if(jumpToConstraint != -1) {
    currentState.activeConstraints = getActiveConstraints();
    bool found = false;
    for(int i = 0; i < currentState.activeConstraints.size(); i++) {
      if(currentState.activeConstraints.at(i) == jumpToConstraint) {
	currentState.activeConstraintsPos = i;
	found = true;
	break;
      }
    }
    if(!found) {
      return;
    }
  } else if(refreshActiveConstraints || currentState.activeConstraints.empty()) {
    currentState.activeConstraints = getActiveConstraints();
    currentState.activeConstraintsPos = 0;
  } else if(currentState.activeConstraintsPos == currentState.activeConstraints.size()-1) {
    currentState.activeConstraintsPos = 0;
  } else {
    currentState.activeConstraintsPos++;
  }
  info.field = -1;
  if(currentState.activeConstraints.empty()) {
    currentState.activeConstraintsPos = -1;
    return;
  }

  currentState.uncertainValues = prepareConstraint(currentState.activeConstraints.at(currentState.activeConstraintsPos));
  gtk_extra_gui_update();
};


/*
 * this function prepares the next variable. If the current state is on the last variable already, it jumps back to the first
 */
void GnomineSolver_CSP::nextVariable() {
  std::cout << "next Variable!" << std::endl;
  //updateMfieldValues();
  //if we dont have a constraint prepare it
  if(currentState.activeConstraints.empty()) {
    nextConstraint();
    //if the list is still empty, there are no active constraints --> nothing to do
    if(currentState.activeConstraints.empty()) {
      return;
    }
  }
  //if we have no uncertain Values anymore, get to next constraint (the true makes it so the activeConstraints are refreshed)
  if(currentState.uncertainValues.empty()) {
    nextConstraint(true);
    if(currentState.activeConstraints.empty()) {
      return;
    }
  }

  //if we checked through all variables, start anew
  if(info.field == currentState.uncertainValues.back()) {
    info.field = currentState.uncertainValues.front();
  } else if(info.field == -1) {
    info.field = currentState.uncertainValues.front();
  } else  {
    std::vector<int>::iterator it;
    for(it = currentState.uncertainValues.begin(); *it != info.field; it++);
    it++;
    info.field = *it;
  }
  currentState.valuePossible[0] = -1;
  currentState.valuePossible[1] = -1;
  fillNumberArray(info.numbers, -1, currentState.uncertainValues);
  prepareVariable(info.field);
  gtk_extra_gui_update();
};


void GnomineSolver_CSP::nextStep() {
  //updateMfieldValues();
  if(currentState.activeConstraints.empty()) {
    nextConstraint();
    //if the list is still empty, there are no active constraints --> nothing to do
    if(currentState.activeConstraints.empty()) {
      return;
    }
  }
  if(info.field == -1 || (currentState.valuePossible[0] + currentState.valuePossible[1]) > 0 ) {
    if(currentState.valuePossible[0] + currentState.valuePossible[1] == 2) {
      info.numbers[info.field] = -1;
    }
    nextVariable();
    if(currentState.activeConstraints.empty()) {
      return;
    }
  }

  if(info.numbers[info.field] == -1) { //this variable hasnt been tested yet
    fillNumberArray(info.numbers, -1, currentState.uncertainValues);
    info.numbers[info.field] = 0;
    info.tested_combinations = 0;
  } else if(info.numbers[info.field] == 0 && currentState.valuePossible[0] > -1) { //value 0 is calculated through, now for value 1
    fillNumberArray(info.numbers, -1, currentState.uncertainValues);
    info.numbers[info.field] = 1;
    info.tested_combinations = 0;
  } else if(info.numbers[info.field] == 1 && currentState.valuePossible[1] > -1) { //value 1 is calculated and also not possible --> impossible problem
      std::cout << "your gnomine Problem is impossible" << std::endl;
      nextConstraint(true);
      return;
  }

  //now we have a valid state we can fill and test
  if(info.possible_combinations > 1) {
    std::cout << "more than one possible combination!" << std::endl;
    fillNumberArray(info.numbers, info.tested_combinations, currentState.uncertainValues);
  }

  bool valid = combinationIsValid(info.numbers, info.sum);
  info.tested_combinations++;
  if(valid) {
    std::cout << "the value " << info.numbers[info.field] << " for variable " << info.field << " on constraint " << currentState.activeConstraints.at(currentState.activeConstraintsPos) << " is valid " << std::endl;
    currentState.valuePossible[info.numbers[info.field]] = 1;
  }

  if(info.tested_combinations == info.possible_combinations && !valid) {
    std::cout << "the value " << info.numbers[info.field] << " for variable " << info.field << " on constraint " << currentState.activeConstraints.at(currentState.activeConstraintsPos) << " isn't valid" << std::endl;
    currentState.valuePossible[info.numbers[info.field]] = 0;
  }
  gtk_extra_gui_update();

  if(currentState.valuePossible[0] + currentState.valuePossible[1] == 1) { //one is valid, the other one isnt
    int valid;
    if(currentState.valuePossible[0] == 1) {
      valid = 0;
    } else {
      valid = 1;
    }   
    //log which field we change to which value and change it in our mfieldValues Array
    int c = getNeighbour(currentState.activeConstraints.at(currentState.activeConstraintsPos), info.field);
    mfieldValues[c] = valid;
    info.numbers[info.field] = valid;
    gtk_minefield_setSolution(GTK_MINEFIELD(mfield), c, valid);
   
    //delete the current field from uncertain Values, since its now set to a value
    int pos = 0;
    for(std::vector<int>::iterator it = currentState.uncertainValues.begin(); *it != info.field; it++) { pos++; }
    currentState.uncertainValues.erase(currentState.uncertainValues.begin()+pos);

    info.field = -1;

    if(currentState.uncertainValues.empty()) {
      nextConstraint(true);
    }

  }

};
